<?php
// Test script to simulate loan approval process and verify loan agreement generation
require_once 'database/config.php';

echo "=== Loan Approval and Agreement Generation Test ===\n\n";

try {
    $pdo = getDatabaseConnection();
    
    // Check if we have any pending loans to test with
    $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
            FROM loans l
            JOIN members m ON l.member_id = m.id
            WHERE l.status = 'Pending'
            LIMIT 1";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $loan = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$loan) {
        echo "No pending loans found. Creating a test loan...\n";
        
        // Create a test loan
        $member_sql = "SELECT id, member_id, first_name, last_name FROM members WHERE status = 'Active' LIMIT 1";
        $member_stmt = $pdo->prepare($member_sql);
        $member_stmt->execute();
        $member = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            echo "No active members found. Cannot create test loan.\n";
            exit(1);
        }
        
        echo "Using member: " . $member['member_id'] . " (" . $member['first_name'] . " " . $member['last_name'] . ")\n";
        
        // Generate a unique reference
        $reference = "LOAN-TEST-" . date('Ymd') . "-" . strtoupper(substr(md5(uniqid()), 0, 6));
        
        // Insert test loan
        $insert_sql = "INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) 
                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $insert_stmt = $pdo->prepare($insert_sql);
        $result = $insert_stmt->execute([
            $reference,
            $member['id'],
            'Emergency',
            5000.00,  // principal_amount
            12.00,    // interest_rate
            5600.00,  // total_amount (5000 + 12% interest)
            6,        // repayment_period
            933.33,   // monthly_payment
            'Test loan for approval process verification',
            date('Y-m-d'),
            'Pending'
        ]);
        
        if ($result) {
            $loan_id = $pdo->lastInsertId();
            echo "Test loan created with ID: " . $loan_id . " and reference: " . $reference . "\n";
            
            // Get the created loan
            $select_sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                           FROM loans l
                           JOIN members m ON l.member_id = m.id
                           WHERE l.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$loan_id]);
            $loan = $select_stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            echo "Failed to create test loan.\n";
            exit(1);
        }
    }
    
    if ($loan) {
        echo "Testing with loan ID: " . $loan['id'] . " (Reference: " . $loan['reference'] . ")\n";
        echo "Current status: " . $loan['status'] . "\n";
        echo "Member: " . $loan['first_name'] . " " . $loan['last_name'] . " (" . $loan['member_ref'] . ")\n\n";
        
        // Get an admin user to approve the loan
        $admin_sql = "SELECT id, member_id FROM members WHERE member_id LIKE 'ADM-%' LIMIT 1";
        $admin_stmt = $pdo->prepare($admin_sql);
        $admin_stmt->execute();
        $admin = $admin_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin) {
            echo "No admin user found. Cannot approve loan.\n";
            exit(1);
        }
        
        echo "Approving loan with admin: " . $admin['member_id'] . "\n";
        
        // Now we need to properly call the update function that triggers agreement generation
        // Include the loans API to access the updateLoan function
        require_once 'api/loans.php';
        
        // Simulate the API call to update the loan status to 'Approved'
        // We need to call the updateLoan function directly
        $update_data = [
            'status' => 'Approved',
            'approval_date' => date('Y-m-d'),
            'approved_by' => $admin['member_id']  // Use member reference ID, not numeric ID
        ];
        
        echo "Calling updateLoan function...\n";
        
        // Capture output since the function echoes JSON
        ob_start();
        updateLoan($loan['id'], $update_data);
        $output = ob_get_clean();
        
        echo "API Response: " . $output . "\n\n";
        
        // Parse the JSON response
        $response = json_decode($output, true);
        
        if ($response && isset($response['message'])) {
            echo "Loan approval process completed: " . $response['message'] . "\n";
            
            // Now verify that the loan agreement was generated
            // The agreement should be saved in the documents/loan_agreements directory
            $expected_filename = 'loan_agreement_' . $loan['reference'] . '_' . date('Y-m-d') . '.pdf';
            $expected_path = 'documents/loan_agreements/' . $expected_filename;
            
            echo "Checking for generated agreement at: " . $expected_path . "\n";
            
            // Wait a moment for file system operations
            sleep(2);
            
            if (file_exists($expected_path)) {
                echo "SUCCESS: Loan agreement generated successfully!\n";
                echo "File location: " . $expected_path . "\n";
                
                // Also check if the agreement was saved in the database
                $doc_sql = "SELECT * FROM documents WHERE title LIKE ? AND file_path = ? ORDER BY created_at DESC LIMIT 1";
                $doc_stmt = $pdo->prepare($doc_sql);
                $doc_stmt->execute(['%'.$loan['reference'].'%', $expected_path]);
                $document = $doc_stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($document) {
                    echo "SUCCESS: Loan agreement also saved in database!\n";
                    echo "Document ID: " . $document['id'] . "\n";
                    echo "Title: " . $document['title'] . "\n";
                } else {
                    echo "WARNING: Agreement file exists but not found in database.\n";
                }
            } else {
                echo "FAILURE: Loan agreement was not generated.\n";
                echo "Expected file path: " . $expected_path . "\n";
                
                // Check if there are any files in the loan_agreements directory
                $dir = 'documents/loan_agreements/';
                if (is_dir($dir)) {
                    $files = scandir($dir);
                    echo "Files in loan_agreements directory:\n";
                    foreach ($files as $file) {
                        if ($file !== '.' && $file !== '..') {
                            echo "  - " . $file . "\n";
                        }
                    }
                } else {
                    echo "Directory does not exist: " . $dir . "\n";
                }
                
                // Let's also check the general documents directory
                $general_dir = 'documents/';
                if (is_dir($general_dir)) {
                    $files = scandir($general_dir);
                    echo "Files in general documents directory:\n";
                    foreach ($files as $file) {
                        if ($file !== '.' && $file !== '..' && strpos($file, 'loan_agreement') !== false) {
                            echo "  - " . $file . "\n";
                        }
                    }
                }
            }
        } else {
            echo "Failed to approve loan through API.\n";
            if ($output) {
                echo "API Output: " . $output . "\n";
            }
        }
    } else {
        echo "No loan found to test with.\n";
    }
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}

echo "\n=== Test Complete ===\n";
?>