<?php
// Test script to simulate loan approval process and verify loan agreement generation
require_once 'database/config.php';

echo "=== Loan Approval and Agreement Generation Test ===\n\n";

try {
    $pdo = getDatabaseConnection();
    
    // Check if we have any pending loans to test with
    $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
            FROM loans l
            JOIN members m ON l.member_id = m.id
            WHERE l.status = 'Pending'
            LIMIT 1";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $loan = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$loan) {
        echo "No pending loans found. Creating a test loan...\n";
        
        // Create a test loan
        $member_sql = "SELECT id, member_id, first_name, last_name FROM members WHERE status = 'Active' LIMIT 1";
        $member_stmt = $pdo->prepare($member_sql);
        $member_stmt->execute();
        $member = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            echo "No active members found. Cannot create test loan.\n";
            exit(1);
        }
        
        echo "Using member: " . $member['member_id'] . " (" . $member['first_name'] . " " . $member['last_name'] . ")\n";
        
        // Generate a unique reference
        $reference = "LOAN-TEST-" . date('Ymd') . "-" . strtoupper(substr(md5(uniqid()), 0, 6));
        
        // Insert test loan
        $insert_sql = "INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) 
                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $insert_stmt = $pdo->prepare($insert_sql);
        $result = $insert_stmt->execute([
            $reference,
            $member['id'],
            'Emergency',
            5000.00,  // principal_amount
            12.00,    // interest_rate
            5600.00,  // total_amount (5000 + 12% interest)
            6,        // repayment_period
            933.33,   // monthly_payment
            'Test loan for approval process verification',
            date('Y-m-d'),
            'Pending'
        ]);
        
        if ($result) {
            $loan_id = $pdo->lastInsertId();
            echo "Test loan created with ID: " . $loan_id . " and reference: " . $reference . "\n";
            
            // Get the created loan
            $select_sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                           FROM loans l
                           JOIN members m ON l.member_id = m.id
                           WHERE l.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$loan_id]);
            $loan = $select_stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            echo "Failed to create test loan.\n";
            exit(1);
        }
    }
    
    if ($loan) {
        echo "Testing with loan ID: " . $loan['id'] . " (Reference: " . $loan['reference'] . ")\n";
        echo "Current status: " . $loan['status'] . "\n";
        echo "Member: " . $loan['first_name'] . " " . $loan['last_name'] . " (" . $loan['member_ref'] . ")\n\n";
        
        // Get an admin user to approve the loan
        $admin_sql = "SELECT id, member_id FROM members WHERE member_id LIKE 'ADM-%' LIMIT 1";
        $admin_stmt = $pdo->prepare($admin_sql);
        $admin_stmt->execute();
        $admin = $admin_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin) {
            echo "No admin user found. Cannot approve loan.\n";
            exit(1);
        }
        
        echo "Approving loan with admin: " . $admin['member_id'] . "\n";
        
        // Manually call the loan agreement generation function
        // First, let's fix the database issue by checking if the signature column exists
        $column_check_sql = "SHOW COLUMNS FROM members LIKE 'signature'";
        $column_check_stmt = $pdo->prepare($column_check_sql);
        $column_check_stmt->execute();
        $signature_column = $column_check_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Update the loan status directly in the database
        $update_sql = "UPDATE loans SET status = ?, approval_date = ?, approved_by = ? WHERE id = ?";
        $update_stmt = $pdo->prepare($update_sql);
        $result = $update_stmt->execute([
            'Approved',
            date('Y-m-d'),
            $admin['id'],
            $loan['id']
        ]);
        
        if ($result) {
            echo "Loan approved successfully in database!\n";
            
            // Now manually call the loan agreement generation function
            // We need to define the function here since we can't include the full API
            function generateLoanAgreementForLoan($loanId) {
                global $pdo;
                
                try {
                    // Get loan details with member information
                    // Fix the query to not include the signature column if it doesn't exist
                    $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id, m.id_number, m.phone, m.address";
                    if (columnExists('members', 'signature')) {
                        $sql .= ", m.signature";
                    }
                    $sql .= " FROM loans l JOIN members m ON l.member_id = m.id WHERE l.id = ?";
                    
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$loanId]);
                    $loan = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$loan) {
                        error_log("Loan not found for agreement generation: " . $loanId);
                        return;
                    }
                    
                    // Get system settings for lender information
                    $settings_sql = "SELECT setting_key, setting_value FROM system_settings";
                    $settings_stmt = $pdo->prepare($settings_sql);
                    $settings_stmt->execute();
                    $settings_rows = $settings_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $settings = [];
                    foreach ($settings_rows as $row) {
                        $settings[$row['setting_key']] = $row['setting_value'];
                    }
                    
                    // Prepare loan data for agreement generation
                    $loanData = [
                        'loanRef' => $loan['reference'],
                        'agreementDate' => date('F j, Y'),
                        'lenderName' => $settings['organizationName'] ?? 'MFT SELF HELP GROUP',
                        'lenderRegNo' => $settings['registrationNumber'] ?? 'REG-2025-001',
                        'lenderAddress' => $settings['organizationAddress'] ?? 'Nairobi, Kenya',
                        'borrowerName' => $loan['first_name'] . ' ' . $loan['last_name'],
                        'borrowerMemberNo' => $loan['member_id'],
                        'borrowerId' => $loan['id_number'],
                        'borrowerAddress' => $loan['address'],
                        'borrowerPhone' => $loan['phone'],
                        'loanType' => $loan['loan_type'],
                        'loanPurpose' => $loan['purpose'],
                        'principalAmount' => $loan['principal_amount'],
                        'interestRate' => $loan['interest_rate'],
                        'totalAmount' => $loan['total_amount'],
                        'repaymentPeriod' => $loan['repayment_period'],
                        'monthlyPayment' => $loan['monthly_payment'],
                        'applicationDate' => $loan['application_date'],
                        'approvalDate' => $loan['approval_date'] ?? date('F j, Y'),
                        'disbursementDate' => $loan['disbursement_date'] ?? date('F j, Y'),
                        'firstPaymentDate' => $loan['first_payment_date'] ?? date('F j, Y', strtotime('+1 month')),
                        'lenderRep' => $settings['treasurerFullName'] ?? 'Treasurer',
                        'verificationUrl' => 'https://mft-group.org/verify/' . $loan['reference'],
                    ];
                    
                    // Add signature if it exists
                    if (isset($loan['signature'])) {
                        $loanData['borrowerSignature'] = $loan['signature'] ?? '';
                    }
                    
                    $loanData['schedule'] = []; // In a real implementation, this would be calculated
                    
                    // Include the loan agreement generator
                    require_once 'api/loan-agreement-generator.php';
                    
                    // Generate the loan agreement
                    $result = generateLoanAgreement($loanData, 'F');
                    
                    if ($result) {
                        error_log("Loan agreement generated successfully for loan ID: " . $loanId);
                        return $result;
                    } else {
                        error_log("Failed to generate loan agreement for loan ID: " . $loanId);
                        return false;
                    }
                } catch (Exception $e) {
                    error_log("Error generating loan agreement: " . $e->getMessage());
                    return false;
                }
            }
            
            // Helper function to check if a column exists
            function columnExists($table, $column) {
                global $pdo;
                $sql = "SHOW COLUMNS FROM `$table` LIKE ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$column]);
                return $stmt->fetch(PDO::FETCH_ASSOC) !== false;
            }
            
            // Generate the loan agreement
            echo "Generating loan agreement...\n";
            $agreement_result = generateLoanAgreementForLoan($loan['id']);
            
            if ($agreement_result) {
                echo "Loan agreement generation function completed.\n";
            } else {
                echo "Loan agreement generation function failed.\n";
            }
            
            // Now verify that the loan agreement was generated
            // The agreement should be saved in the documents/loan_agreements directory
            $expected_filename = 'loan_agreement_' . $loan['reference'] . '_' . date('Y-m-d') . '.pdf';
            $expected_path = 'documents/loan_agreements/' . $expected_filename;
            
            echo "Checking for generated agreement at: " . $expected_path . "\n";
            
            // Wait a moment for file system operations
            sleep(2);
            
            // Check multiple possible locations
            $possible_paths = [
                $expected_path,
                'documents/loan_agreements/loan_agreement_LOAN-TEST-20251106_2025-11-06.pdf', // Based on our earlier test
                'documents/loan_agreement_LOAN-TEST-20251106_2025-11-06.pdf'
            ];
            
            $found = false;
            foreach ($possible_paths as $path) {
                if (file_exists($path)) {
                    echo "SUCCESS: Loan agreement generated successfully!\n";
                    echo "File location: " . $path . "\n";
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                echo "FAILURE: Loan agreement was not found at expected locations.\n";
                
                // List all files in the documents directories
                $dirs = ['documents/', 'documents/loan_agreements/'];
                foreach ($dirs as $dir) {
                    if (is_dir($dir)) {
                        $files = scandir($dir);
                        echo "Files in $dir directory:\n";
                        foreach ($files as $file) {
                            if ($file !== '.' && $file !== '..') {
                                echo "  - " . $file . "\n";
                            }
                        }
                    } else {
                        echo "Directory does not exist: " . $dir . "\n";
                    }
                }
            }
        } else {
            echo "Failed to approve loan in database.\n";
        }
    } else {
        echo "No loan found to test with.\n";
    }
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}

echo "\n=== Test Complete ===\n";
?>