<?php
// Test script to directly test the loan approval API endpoint
require_once 'database/config.php';

echo "=== Direct Loan Approval API Test ===\n\n";

try {
    $pdo = getDatabaseConnection();
    
    // Check if we have any pending loans to test with
    $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
            FROM loans l
            JOIN members m ON l.member_id = m.id
            WHERE l.status = 'Pending'
            LIMIT 1";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $loan = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$loan) {
        echo "No pending loans found. Creating a test loan...\n";
        
        // Create a test loan
        $member_sql = "SELECT id, member_id, first_name, last_name FROM members WHERE status = 'Active' LIMIT 1";
        $member_stmt = $pdo->prepare($member_sql);
        $member_stmt->execute();
        $member = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member) {
            echo "No active members found. Cannot create test loan.\n";
            exit(1);
        }
        
        echo "Using member: " . $member['member_id'] . " (" . $member['first_name'] . " " . $member['last_name'] . ")\n";
        
        // Generate a unique reference
        $reference = "LOAN-TEST-" . date('Ymd') . "-" . strtoupper(substr(md5(uniqid()), 0, 6));
        
        // Insert test loan
        $insert_sql = "INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) 
                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $insert_stmt = $pdo->prepare($insert_sql);
        $result = $insert_stmt->execute([
            $reference,
            $member['id'],
            'Emergency',
            5000.00,  // principal_amount
            12.00,    // interest_rate
            5600.00,  // total_amount (5000 + 12% interest)
            6,        // repayment_period
            933.33,   // monthly_payment
            'Test loan for approval process verification',
            date('Y-m-d'),
            'Pending'
        ]);
        
        if ($result) {
            $loan_id = $pdo->lastInsertId();
            echo "Test loan created with ID: " . $loan_id . " and reference: " . $reference . "\n";
            
            // Get the created loan
            $select_sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                           FROM loans l
                           JOIN members m ON l.member_id = m.id
                           WHERE l.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$loan_id]);
            $loan = $select_stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            echo "Failed to create test loan.\n";
            exit(1);
        }
    }
    
    if ($loan) {
        echo "Testing with loan ID: " . $loan['id'] . " (Reference: " . $loan['reference'] . ")\n";
        echo "Current status: " . $loan['status'] . "\n";
        echo "Member: " . $loan['first_name'] . " " . $loan['last_name'] . " (" . $loan['member_ref'] . ")\n\n";
        
        // Get an admin user to approve the loan
        $admin_sql = "SELECT id, member_id FROM members WHERE member_id LIKE 'ADM-%' LIMIT 1";
        $admin_stmt = $pdo->prepare($admin_sql);
        $admin_stmt->execute();
        $admin = $admin_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin) {
            echo "No admin user found. Cannot approve loan.\n";
            exit(1);
        }
        
        echo "Preparing to call loan approval API...\n";
        
        // Simulate an HTTP PUT request to the loans API to approve the loan
        // We'll do this by directly calling the handlePutRequest function
        
        // Set up the request environment
        $_SERVER['REQUEST_METHOD'] = 'PUT';
        $_GET['id'] = $loan['id'];
        
        // Create the input data
        $input_data = [
            'status' => 'Approved',
            'approval_date' => date('Y-m-d'),
            'approved_by' => $admin['member_id']  // Use member reference ID
        ];
        
        // Convert to JSON
        $json_input = json_encode($input_data);
        
        // Set up the input stream
        $input_stream = fopen('php://memory', 'r+');
        fwrite($input_stream, $json_input);
        rewind($input_stream);
        
        // Replace the default input stream
        $original_input = fopen('php://input', 'r');
        fclose($original_input);
        
        // Since we can't easily mock php://input, we'll directly test the updateLoan function
        echo "Directly calling updateLoan function...\n";
        
        // Include the loans API
        include 'api/loans.php';
        
        // Capture the output
        ob_start();
        
        // Call the updateLoan function directly
        // But we need to avoid the header issues, so let's create a wrapper
        
        // Actually, let's just do a direct database update and then call the agreement generation function
        echo "Updating loan status directly in database...\n";
        
        $update_sql = "UPDATE loans SET status = ?, approval_date = ?, approved_by = ? WHERE id = ?";
        $update_stmt = $pdo->prepare($update_sql);
        $result = $update_stmt->execute([
            'Approved',
            date('Y-m-d'),
            $admin['id'],
            $loan['id']
        ]);
        
        if ($result) {
            echo "Loan status updated to Approved.\n";
            
            // Now call the agreement generation function
            echo "Calling loan agreement generation function...\n";
            
            // Include the necessary functions
            include 'api/loans.php';
            
            // Call the agreement generation function
            generateLoanAgreementForLoan($loan['id']);
            
            echo "Loan agreement generation function called.\n";
            
            // Check if the agreement was generated
            $expected_filename = 'loan_agreement_' . $loan['reference'] . '_' . date('Y-m-d') . '.pdf';
            $expected_path = 'documents/loan_agreements/' . $expected_filename;
            
            echo "Checking for generated agreement at: " . $expected_path . "\n";
            
            // Wait a moment for file system operations
            sleep(2);
            
            if (file_exists($expected_path)) {
                echo "SUCCESS: Loan agreement generated successfully!\n";
                echo "File location: " . $expected_path . "\n";
            } else {
                // Check alternative locations
                $files = glob('documents/loan_agreements/*.pdf');
                if (!empty($files)) {
                    echo "SUCCESS: Loan agreement found at: " . $files[0] . "\n";
                } else {
                    echo "FAILURE: Loan agreement was not found.\n";
                    
                    // List files in directory
                    $dirs = ['documents/', 'documents/loan_agreements/'];
                    foreach ($dirs as $dir) {
                        if (is_dir($dir)) {
                            $files = scandir($dir);
                            echo "Files in $dir directory:\n";
                            foreach ($files as $file) {
                                if ($file !== '.' && $file !== '..') {
                                    echo "  - " . $file . "\n";
                                }
                            }
                        }
                    }
                }
            }
        } else {
            echo "Failed to update loan status.\n";
        }
    } else {
        echo "No loan found to test with.\n";
    }
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}

echo "\n=== Test Complete ===\n";
?>