// MFT GROUP Real-Time Filtering Utility
// This script provides reusable real-time filtering functionality for various dashboards

/**
 * Real-time filtering utility class
 */
class RealTimeFilter {
    /**
     * Initialize real-time filtering for a table
     * @param {string} tableId - ID of the table to filter
     * @param {Array} filterFields - Array of filter field selectors
     * @param {Object} options - Configuration options
     */
    static initTableFilter(tableId, filterFields, options = {}) {
        const table = document.getElementById(tableId);
        if (!table) {
            console.warn(`Table with ID ${tableId} not found`);
            return;
        }
        
        const config = {
            debounceDelay: 300,
            caseSensitive: false,
            ...options
        };
        
        // Get all filter elements
        const filterElements = filterFields.map(field => document.querySelector(field));
        
        // Attach event listeners to filter elements
        filterElements.forEach(element => {
            if (!element) return;
            
            const eventType = element.tagName === 'SELECT' ? 'change' : 'input';
            element.addEventListener(eventType, this.debounce(() => {
                this.applyFilters(table, filterElements, config);
            }, config.debounceDelay));
        });
        
        // Also apply filters immediately if there are initial values
        this.applyFilters(table, filterElements, config);
    }
    
    /**
     * Apply filters to a table
     * @param {HTMLElement} table - Table element to filter
     * @param {Array} filterElements - Array of filter input elements
     * @param {Object} config - Configuration options
     */
    static applyFilters(table, filterElements, config) {
        const tbody = table.querySelector('tbody');
        if (!tbody) return;
        
        const rows = tbody.querySelectorAll('tr');
        const filters = filterElements.map(element => {
            if (!element) return null;
            return {
                element: element,
                value: element.value.toLowerCase(),
                column: parseInt(element.dataset.column) || -1,
                type: element.dataset.filterType || 'text'
            };
        }).filter(f => f !== null);
        
        rows.forEach(row => {
            let showRow = true;
            
            // Check each filter
            for (const filter of filters) {
                if (!filter.value) continue;
                
                let cellText = '';
                
                // If column is specified, filter by that column
                if (filter.column >= 0) {
                    const cell = row.cells[filter.column];
                    if (cell) {
                        cellText = cell.textContent.toLowerCase();
                    }
                } else {
                    // Otherwise, search all cells in the row
                    cellText = Array.from(row.cells)
                        .map(cell => cell.textContent.toLowerCase())
                        .join(' ');
                }
                
                // Apply filter based on type
                switch (filter.type) {
                    case 'select':
                        if (filter.value !== 'all' && !cellText.includes(filter.value)) {
                            showRow = false;
                        }
                        break;
                    case 'text':
                    default:
                        if (!cellText.includes(filter.value)) {
                            showRow = false;
                        }
                        break;
                }
                
                if (!showRow) break;
            }
            
            row.style.display = showRow ? '' : 'none';
        });
    }
    
    /**
     * Debounce function to limit rate of function calls
     * @param {Function} func - Function to debounce
     * @param {number} delay - Delay in milliseconds
     * @returns {Function} Debounced function
     */
    static debounce(func, delay) {
        let timeoutId;
        return function (...args) {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => func.apply(this, args), delay);
        };
    }
    
    /**
     * Clear all filters
     * @param {Array} filterFields - Array of filter field selectors
     * @param {Function} callback - Optional callback function to call after clearing
     */
    static clearFilters(filterFields, callback) {
        filterFields.forEach(field => {
            const element = document.querySelector(field);
            if (!element) return;
            
            if (element.tagName === 'SELECT') {
                element.value = element.querySelector('option:first-child')?.value || '';
            } else {
                element.value = '';
            }
        });
        
        if (callback && typeof callback === 'function') {
            callback();
        }
    }
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = RealTimeFilter;
}