// MFT GROUP Real-Time Client
// This script handles real-time updates using WebSocket or long-polling

class MFTRealTimeClient {
    constructor(baseURL = null) {
        console.log('=== REALTIME CLIENT INIT ===');
        console.log('Constructor received baseURL:', baseURL);
        
        // Use the getApiBaseUrl function if available, otherwise default to '/api'
        // Fix: Ensure we properly handle the baseURL parameter and fallback
        if (baseURL) {
            this.baseURL = baseURL;
        } else if (typeof getApiBaseUrl === 'function') {
            this.baseURL = getApiBaseUrl();
        } else {
            // Fallback to a more intelligent default based on the current URL
            const pathname = window.location.pathname;
            const origin = window.location.origin;
            console.log('Using fallback URL construction');
            console.log('Pathname:', pathname);
            console.log('Origin:', origin);
            
            // SPECIAL CASE FOR MFT DEPLOYMENT:
            // When the web server root is the MFT directory but accessed via /mft/ URL
            if (pathname.toLowerCase().startsWith('/mft/')) {
                this.baseURL = origin + '/MFT/api';
            } else if (pathname.toLowerCase().includes('/mft/')) {
                // Determine the actual case of the MFT directory from the URL
                const pathParts = pathname.split('/');
                for (let i = 0; i < pathParts.length; i++) {
                    if (pathParts[i].toLowerCase() === 'mft') {
                        this.baseURL = origin + '/MFT/api';
                        break;
                    }
                }
                // Fallback if we couldn't determine the case
                if (!this.baseURL) {
                    this.baseURL = origin + '/MFT/api';
                }
            } else {
                this.baseURL = origin + '/MFT/api';
            }
        }
        
        // Fix for file:// protocol issues - ensure we have a valid HTTP/HTTPS URL
        if (this.baseURL && this.baseURL.startsWith('file://')) {
            // If we're running from file://, we need to use localhost
            this.baseURL = 'http://localhost/MFT/api';
        }
        
        this.websocket = null;
        this.pollingInterval = null;
        this.isConnected = false;
        this.dashboardType = null;
        this.memberId = null;
        this.callbacks = [];
        this.useWebSocket = false; // Set to true if you implement WebSocket
        this.lastUpdate = 0; // Track the last update timestamp
        this.retryCount = 0; // Track retry attempts
        this.maxRetries = 5; // Maximum retry attempts
        
        console.log('MFTRealTimeClient initialized with baseURL:', this.baseURL);
    }
    
    // Connect to real-time updates
    connect(dashboardType, memberId = null) {
        this.dashboardType = dashboardType;
        this.memberId = memberId;
        
        if (this.useWebSocket) {
            this.connectWebSocket();
        } else {
            this.startLongPolling();
        }
    }
    
    // Disconnect from real-time updates
    disconnect() {
        if (this.websocket) {
            this.websocket.close();
            this.websocket = null;
        }
        
        if (this.pollingInterval) {
            clearInterval(this.pollingInterval);
            this.pollingInterval = null;
        }
        
        this.isConnected = false;
    }
    
    // Connect using WebSocket (if implemented)
    connectWebSocket() {
        try {
            // This is a placeholder - in a real implementation, you would connect to a WebSocket server
            console.warn('WebSocket connection not implemented. Using long-polling instead.');
            this.startLongPolling();
        } catch (error) {
            console.error('WebSocket connection failed:', error);
            // Fall back to long-polling
            this.startLongPolling();
        }
    }
    
    // Start long-polling for updates
    startLongPolling() {
        console.log('Starting long-polling for real-time updates');
        this.isConnected = true;
        this.retryCount = 0; // Reset retry count on successful start
        
        // Start polling immediately
        this.pollForUpdates();
        
        // Set up interval for regular polling (every 30 seconds for more frequent updates)
        this.pollingInterval = setInterval(() => {
            this.pollForUpdates();
        }, 30000);
    }
    
    // Poll for updates
    async pollForUpdates() {
        try {
            const params = new URLSearchParams({
                dashboard: this.dashboardType,
                last_update: this.lastUpdate // Use the tracked last update timestamp
            });
            
            if (this.memberId) {
                params.append('member_id', this.memberId);
            }
            
            // Fix: Ensure the URL is constructed correctly using the baseURL
            const url = `${this.baseURL}/websocket-server.php?${params}`;
            console.log('Polling URL:', url);
            
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            console.log('Polling response status:', response.status);
            
            // Reset retry count on successful response
            this.retryCount = 0;
            
            // Check if response is OK before parsing JSON
            if (!response.ok) {
                const text = await response.text();
                console.error('Polling error response:', text.substring(0, 200));
                // Add a delay before next poll to prevent rapid error loops
                await new Promise(resolve => setTimeout(resolve, 5000));
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const contentType = response.headers.get('content-type');
            console.log('Polling content type:', contentType);
            
            if (!contentType || !contentType.includes('application/json')) {
                const text = await response.text();
                console.error('Polling response is not JSON:', text.substring(0, 200));
                // Add a delay before next poll to prevent rapid error loops
                await new Promise(resolve => setTimeout(resolve, 3000));
                throw new Error('Response is not JSON');
            }
            
            const data = await response.json();
            
            // Update the last update timestamp
            if (data.timestamp) {
                this.lastUpdate = data.timestamp;
            }
            
            if (data.status === 'updated') {
                console.log('Received real-time update:', data);
                this.handleUpdate(data.data);
            } else if (data.status === 'no_updates') {
                console.log('No updates available');
            } else {
                console.warn('Unexpected response from server:', data);
            }
        } catch (error) {
            console.error('Error polling for updates:', error);
            this.retryCount++;
            
            // If we've exceeded max retries, disconnect and show error
            if (this.retryCount >= this.maxRetries) {
                console.error('Max retries exceeded. Disconnecting from real-time updates.');
                this.disconnect();
                // Notify user of connection issue
                if (typeof showNotification === 'function') {
                    showNotification('Connection to real-time updates lost. Please refresh the page.', 'error');
                }
                return;
            }
            
            // Exponential backoff for retries
            const delay = Math.min(5000 * Math.pow(2, this.retryCount), 30000); // Max 30 seconds
            console.log(`Retrying in ${delay}ms (attempt ${this.retryCount}/${this.maxRetries})`);
            await new Promise(resolve => setTimeout(resolve, delay));
        }
    }
    
    // Handle incoming updates
    handleUpdate(data) {
        // Notify all registered callbacks
        this.callbacks.forEach(callback => {
            try {
                callback(data, this.dashboardType);
            } catch (error) {
                console.error('Error in update callback:', error);
            }
        });
    }
    
    // Register a callback for updates
    onUpdate(callback) {
        if (typeof callback === 'function') {
            this.callbacks.push(callback);
        }
    }
    
    // Remove a callback
    removeCallback(callback) {
        const index = this.callbacks.indexOf(callback);
        if (index > -1) {
            this.callbacks.splice(index, 1);
        }
    }
    
    // Send a notification (for server-side updates)
    async sendNotification(notification) {
        try {
            const url = `${this.baseURL}/websocket-server.php`;
            console.log('Sending notification to:', url);
            
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(notification)
            });
            
            console.log('Notification response status:', response.status);
            
            // Check if response is OK before parsing JSON
            if (!response.ok) {
                const text = await response.text();
                console.error('Notification error response:', text.substring(0, 200));
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const contentType = response.headers.get('content-type');
            console.log('Notification content type:', contentType);
            
            if (!contentType || !contentType.includes('application/json')) {
                const text = await response.text();
                console.error('Notification response is not JSON:', text.substring(0, 200));
                throw new Error('Response is not JSON');
            }
            
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Error sending notification:', error);
            throw error;
        }
    }
}

// Create a global instance
console.log('=== CREATING GLOBAL REALTIME INSTANCE ===');
let baseURL = null;
if (typeof getApiBaseUrl === 'function') {
    baseURL = getApiBaseUrl();
} else {
    // Fallback to a more intelligent default based on the current URL
    const pathname = window.location.pathname;
    const origin = window.location.origin || 'http://localhost';
    // SPECIAL CASE FOR MFT DEPLOYMENT:
    // When the web server root is the MFT directory but accessed via /mft/ URL
    if (pathname.toLowerCase().startsWith('/mft/')) {
        baseURL = origin + '/MFT/api';
    } else if (pathname.toLowerCase().includes('/mft/')) {
        // Determine the actual case of the MFT directory from the URL
        const pathParts = pathname.split('/');
        for (let i = 0; i < pathParts.length; i++) {
            if (pathParts[i].toLowerCase() === 'mft') {
                baseURL = origin + '/MFT/api';
                break;
            }
        }
        // Fallback if we couldn't determine the case
        if (!baseURL) {
            baseURL = origin + '/MFT/api';
        }
    } else {
        baseURL = origin + '/MFT/api';
    }
}

// Fix for file:// protocol issues - ensure we have a valid HTTP/HTTPS URL
if (baseURL && baseURL.startsWith('file://')) {
    // If we're running from file://, we need to use localhost
    baseURL = 'http://localhost/MFT/api';
}

console.log('Global realtime instance baseURL:', baseURL);
const mftRealTime = new MFTRealTimeClient(baseURL);

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = MFTRealTimeClient;
}