// Pagination utility for MFT GROUP
// This script provides pagination functionality for tables and lists

/**
 * Initialize pagination component
 * @param {string} containerId - The ID of the container element for pagination
 * @param {Object} options - Configuration options
 * @returns {Object} Pagination instance
 */
function initializePagination(containerId, options) {
    // Default options
    const defaults = {
        currentPage: 1,
        totalPages: 1,
        itemsPerPage: 10,
        totalItems: 0,
        maxVisiblePages: 5,
        onPageChange: function(page) {
            console.log('Page changed to:', page);
        }
    };
    
    // Merge options with defaults
    const config = Object.assign({}, defaults, options);
    
    // Get container element
    const container = document.getElementById(containerId);
    if (!container) {
        console.error('Pagination container not found:', containerId);
        return null;
    }
    
    // Create pagination instance
    const pagination = {
        containerId: containerId,
        container: container,
        currentPage: config.currentPage,
        totalPages: config.totalPages,
        itemsPerPage: config.itemsPerPage,
        totalItems: config.totalItems,
        maxVisiblePages: config.maxVisiblePages,
        onPageChange: config.onPageChange,
        
        /**
         * Render pagination controls
         */
        render: function() {
            // Clear container
            this.container.innerHTML = '';
            
            // Don't render if there's only one page or no items
            if (this.totalPages <= 1) {
                return;
            }
            
            // Create pagination element
            const paginationElement = document.createElement('div');
            paginationElement.className = 'mft-pagination';
            
            // Previous button
            const prevButton = document.createElement('a');
            prevButton.className = 'page-link';
            prevButton.href = '#';
            prevButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
            prevButton.addEventListener('click', (e) => {
                e.preventDefault();
                if (this.currentPage > 1) {
                    this.goToPage(this.currentPage - 1);
                }
            });
            
            if (this.currentPage === 1) {
                prevButton.classList.add('disabled');
            }
            
            paginationElement.appendChild(prevButton);
            
            // Calculate visible page range
            let startPage, endPage;
            if (this.totalPages <= this.maxVisiblePages) {
                // Show all pages
                startPage = 1;
                endPage = this.totalPages;
            } else {
                // Calculate start and end pages
                const maxPagesBeforeCurrentPage = Math.floor(this.maxVisiblePages / 2);
                const maxPagesAfterCurrentPage = Math.ceil(this.maxVisiblePages / 2) - 1;
                
                if (this.currentPage <= maxPagesBeforeCurrentPage) {
                    // Current page is near the beginning
                    startPage = 1;
                    endPage = this.maxVisiblePages;
                } else if (this.currentPage + maxPagesAfterCurrentPage >= this.totalPages) {
                    // Current page is near the end
                    startPage = this.totalPages - this.maxVisiblePages + 1;
                    endPage = this.totalPages;
                } else {
                    // Current page is somewhere in the middle
                    startPage = this.currentPage - maxPagesBeforeCurrentPage;
                    endPage = this.currentPage + maxPagesAfterCurrentPage;
                }
            }
            
            // First page link (if not already shown)
            if (startPage > 1) {
                const firstPageLink = document.createElement('a');
                firstPageLink.className = 'page-link';
                firstPageLink.href = '#';
                firstPageLink.textContent = '1';
                firstPageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.goToPage(1);
                });
                paginationElement.appendChild(firstPageLink);
                
                // Ellipsis
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-ellipsis';
                    ellipsis.textContent = '...';
                    paginationElement.appendChild(ellipsis);
                }
            }
            
            // Page number links
            for (let i = startPage; i <= endPage; i++) {
                const pageLink = document.createElement('a');
                pageLink.className = 'page-link';
                if (i === this.currentPage) {
                    pageLink.classList.add('active');
                }
                pageLink.href = '#';
                pageLink.textContent = i;
                pageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.goToPage(i);
                });
                paginationElement.appendChild(pageLink);
            }
            
            // Last page link (if not already shown)
            if (endPage < this.totalPages) {
                // Ellipsis
                if (endPage < this.totalPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-ellipsis';
                    ellipsis.textContent = '...';
                    paginationElement.appendChild(ellipsis);
                }
                
                const lastPageLink = document.createElement('a');
                lastPageLink.className = 'page-link';
                lastPageLink.href = '#';
                lastPageLink.textContent = this.totalPages;
                lastPageLink.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.goToPage(this.totalPages);
                });
                paginationElement.appendChild(lastPageLink);
            }
            
            // Next button
            const nextButton = document.createElement('a');
            nextButton.className = 'page-link';
            nextButton.href = '#';
            nextButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
            nextButton.addEventListener('click', (e) => {
                e.preventDefault();
                if (this.currentPage < this.totalPages) {
                    this.goToPage(this.currentPage + 1);
                }
            });
            
            if (this.currentPage === this.totalPages) {
                nextButton.classList.add('disabled');
            }
            
            paginationElement.appendChild(nextButton);
            
            // Add to container
            this.container.appendChild(paginationElement);
        },
        
        /**
         * Go to specific page
         * @param {number} page - Page number to go to
         */
        goToPage: function(page) {
            if (page < 1 || page > this.totalPages || page === this.currentPage) {
                return;
            }
            
            this.currentPage = page;
            this.render();
            this.onPageChange(page);
        },
        
        /**
         * Update pagination with new data
         * @param {Object} data - New pagination data
         */
        update: function(data) {
            if (data.currentPage !== undefined) this.currentPage = data.currentPage;
            if (data.totalPages !== undefined) this.totalPages = data.totalPages;
            if (data.itemsPerPage !== undefined) this.itemsPerPage = data.itemsPerPage;
            if (data.totalItems !== undefined) this.totalItems = data.totalItems;
            
            this.render();
        }
    };
    
    // Initial render
    pagination.render();
    
    // Return pagination instance
    return pagination;
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { initializePagination };
}