// MFT GROUP Notification Service
// This script handles sending notifications to trigger real-time updates

class MFTNotificationService {
    constructor() {
        this.notificationEndpoints = [
            getApiUrl('websocket-server.php') // Long-polling endpoint
            // Add WebSocket endpoint here if implemented
        ];
    }
    
    // Send a notification that data has been updated
    async notifyDataUpdate(updateType, data = {}) {
        const notification = {
            type: updateType,
            timestamp: Date.now(),
            data: data
        };
        
        try {
            // Send notification to all endpoints
            const promises = this.notificationEndpoints.map(endpoint => 
                this.sendNotificationToEndpoint(endpoint, notification)
            );
            
            const results = await Promise.allSettled(promises);
            
            // Log results
            results.forEach((result, index) => {
                if (result.status === 'fulfilled') {
                    console.log(`Notification sent to ${this.notificationEndpoints[index]}:`, result.value);
                } else {
                    console.error(`Failed to send notification to ${this.notificationEndpoints[index]}:`, result.reason);
                }
            });
            
            return results;
        } catch (error) {
            console.error('Error sending notifications:', error);
            throw error;
        }
    }
    
    // Send notification to a specific endpoint
    async sendNotificationToEndpoint(endpoint, notification) {
        try {
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(notification)
            });
            
            const data = await response.json();
            return data;
        } catch (error) {
            console.error(`Error sending notification to ${endpoint}:`, error);
            throw error;
        }
    }
    
    // Notify about member updates
    async notifyMemberUpdate(memberId, updateDetails = {}) {
        return this.notifyDataUpdate('member_update', {
            memberId: memberId,
            ...updateDetails
        });
    }
    
    // Notify about contribution updates
    async notifyContributionUpdate(memberId, contributionId, updateDetails = {}) {
        return this.notifyDataUpdate('contribution_update', {
            memberId: memberId,
            contributionId: contributionId,
            ...updateDetails
        });
    }
    
    // Notify about loan updates
    async notifyLoanUpdate(memberId, loanId, updateDetails = {}) {
        return this.notifyDataUpdate('loan_update', {
            memberId: memberId,
            loanId: loanId,
            ...updateDetails
        });
    }
    
    // Notify about dashboard updates
    async notifyDashboardUpdate(dashboardType, updateDetails = {}) {
        return this.notifyDataUpdate('dashboard_update', {
            dashboardType: dashboardType,
            ...updateDetails
        });
    }
}

// Create a global instance
const mftNotifications = new MFTNotificationService();

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = MFTNotificationService;
}