// Mobile Enhancements for MFT GROUP Website

document.addEventListener('DOMContentLoaded', function() {
    // Handle touch devices specifically
    if ('ontouchstart' in window || navigator.maxTouchPoints > 0) {
        // Add touch-specific classes to body
        document.body.classList.add('touch-device');
        
        // Enhance buttons for better touch experience
        const buttons = document.querySelectorAll('.btn, .card, .nav-menu a, .sidebar-menu a');
        buttons.forEach(button => {
            // Add touch feedback
            button.addEventListener('touchstart', function() {
                this.classList.add('touch-active');
            });
            
            button.addEventListener('touchend', function() {
                setTimeout(() => {
                    this.classList.remove('touch-active');
                }, 150);
            });
            
            button.addEventListener('touchcancel', function() {
                this.classList.remove('touch-active');
            });
        });
        
        // Handle form inputs for better mobile experience
        const inputs = document.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            // Ensure inputs have appropriate sizing for touch
            if (!input.style.minHeight || parseInt(input.style.minHeight) < 44) {
                input.style.minHeight = '44px';
            }
            
            // Add focus enhancements
            input.addEventListener('focus', function() {
                this.classList.add('mobile-focused');
            });
            
            input.addEventListener('blur', function() {
                this.classList.remove('mobile-focused');
            });
        });
    }
    
    // Handle viewport height issues on mobile browsers
    function handleViewportHeight() {
        let vh = window.innerHeight * 0.01;
        document.documentElement.style.setProperty('--vh', `${vh}px`);
    }
    
    // Initial setup
    handleViewportHeight();
    
    // Update on resize
    window.addEventListener('resize', handleViewportHeight);
    
    // Update on orientation change
    window.addEventListener('orientationchange', function() {
        setTimeout(handleViewportHeight, 100);
    });
    
    // Improve scroll behavior on touch devices
    if ('scrollBehavior' in document.documentElement.style) {
        // Smooth scrolling for anchor links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function(e) {
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    e.preventDefault();
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });
    }
    
    // Handle keyboard visibility on mobile
    let initialViewportHeight = window.innerHeight;
    
    window.addEventListener('resize', function() {
        const currentViewportHeight = window.innerHeight;
        const keyboardHeight = initialViewportHeight - currentViewportHeight;
        
        // If keyboard is visible (keyboard height > 150px)
        if (keyboardHeight > 150) {
            document.body.classList.add('keyboard-visible');
            document.body.style.setProperty('--keyboard-height', `${keyboardHeight}px`);
        } else {
            document.body.classList.remove('keyboard-visible');
            document.body.style.removeProperty('--keyboard-height');
        }
    });
    
    // Prevent zoom on form focus for iOS
    const iOS = /iPad|iPhone|iPod/.test(navigator.userAgent);
    if (iOS) {
        const viewport = document.querySelector('meta[name="viewport"]');
        if (viewport) {
            const initialContent = viewport.getAttribute('content');
            
            document.addEventListener('focusin', function(e) {
                if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA' || e.target.tagName === 'SELECT') {
                    viewport.setAttribute('content', initialContent + ', maximum-scale=1');
                }
            });
            
            document.addEventListener('focusout', function(e) {
                if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA' || e.target.tagName === 'SELECT') {
                    viewport.setAttribute('content', initialContent);
                }
            });
        }
    }
    
    // Handle fast clicks on touch devices
    const touchElements = document.querySelectorAll('a, button, .btn, .card');
    touchElements.forEach(element => {
        let touchStartTime = 0;
        
        element.addEventListener('touchstart', function() {
            touchStartTime = new Date().getTime();
        });
        
        element.addEventListener('touchend', function(e) {
            const touchEndTime = new Date().getTime();
            const touchDuration = touchEndTime - touchStartTime;
            
            // If touch was very short, it's likely a tap
            if (touchDuration < 200) {
                // Prevent default to avoid double tap zoom
                e.preventDefault();
                
                // Trigger click event manually
                const clickEvent = new MouseEvent('click', {
                    bubbles: true,
                    cancelable: true,
                    view: window
                });
                e.target.dispatchEvent(clickEvent);
            }
        });
    });
});

// Utility function to detect mobile devices
function isMobileDevice() {
    return (typeof window.orientation !== "undefined") || (navigator.userAgent.indexOf('IEMobile') !== -1);
}

// Utility function to detect tablet devices
function isTabletDevice() {
    return isMobileDevice() && Math.min(screen.width, screen.height) >= 768;
}

// Utility function to detect phone devices
function isPhoneDevice() {
    return isMobileDevice() && !isTabletDevice();
}

// Export functions for use in other scripts
window.isMobileDevice = isMobileDevice;
window.isTabletDevice = isTabletDevice;
window.isPhoneDevice = isPhoneDevice;