// MFT GROUP Member Projects Real-Time Updates
// This script handles real-time updates for the member projects page

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Get member ID from localStorage or other source
    // In a real implementation, you would get this from authentication
    const memberId = getMemberId();
    
    if (memberId) {
        // Initialize real-time client
        mftRealTime.connect('member', memberId);
        
        // Register update callback
        mftRealTime.onUpdate(function(data, pageType) {
            if (pageType === 'member-projects') {
                updateMemberProjectsRealTime(data);
            }
        });
        
        // Load initial data
        loadProjectsData(memberId);
        
        console.log('Real-time updates started for member projects page');
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
    }
});

// Function to get member ID (this would be implemented based on your auth system)
function getMemberId() {
    // For now, we'll try to get it from localStorage
    // In a real implementation, this would come from your authentication system
    const memberId = localStorage.getItem('memberId');
    
    // If no member ID in localStorage, try to get from URL parameter
    if (!memberId) {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get('member_id');
    }
    
    return memberId;
}

// Function to load initial projects data
async function loadProjectsData(memberId) {
    try {
        // Validate member ID
        if (!memberId) {
            console.error('No member ID provided');
            showErrorState();
            return;
        }
        
        // Show loading state
        showLoadingState();
        
        // Fetch projects data from API
        console.log('Loading projects for member ID:', memberId);
        const projectsData = await mftApi.getMemberProjects(memberId);
        
        if (projectsData && !projectsData.error) {
            // Update the page with the fetched data
            updateProjectsList(projectsData);
        } else {
            console.error('Error in projects data response:', projectsData);
            // Show error state
            showErrorState();
        }
    } catch (error) {
        console.error('Error loading projects data:', error);
        showErrorState();
    }
}

// Function to show loading state
function showLoadingState() {
    // Update projects table with loading state
    const tableBody = document.getElementById('projects-table-body');
    if (tableBody) {
        tableBody.innerHTML = '<tr><td colspan="8" class="loading-message">Loading projects...</td></tr>';
    }
}

// Function to show error state
function showErrorState() {
    // Update projects table with error state
    const tableBody = document.getElementById('projects-table-body');
    if (tableBody) {
        tableBody.innerHTML = '<tr><td colspan="8" class="error-message">Error loading projects. Please try again later.</td></tr>';
    }
    
    // Log error for debugging
    console.error('Failed to load projects data. Check if member ID is valid and projects exist in database.');
}

// Function to update the member projects page with real-time data
function updateMemberProjectsRealTime(data) {
    console.log('Updating member projects with real-time data:', data);
    
    // Update projects information
    if (data.projects) {
        updateProjectsList(data.projects);
    }
}

// Function to update projects list
function updateProjectsList(data) {
    // Check if data is an object with a projects array
    let projects = [];
    if (data && typeof data === 'object') {
        if (Array.isArray(data)) {
            // Direct array
            projects = data;
        } else if (Array.isArray(data.projects)) {
            // Object with projects array
            projects = data.projects;
        }
    }
    
    // This would update the projects display with real-time data
    // In a real implementation, you would dynamically update the DOM elements
    console.log('Updating projects list with:', projects);
    
    // Update the projects display using the table layout
    renderProjects(projects);
}

// Function to render projects in the DOM (table layout)
function renderProjects(projects) {
    const tableBody = document.getElementById('projects-table-body');
    
    if (!projects || projects.length === 0) {
        tableBody.innerHTML = '<tr><td colspan="8" class="no-data-message">No projects found</td></tr>';
        return;
    }
    
    // Clear existing content
    tableBody.innerHTML = '';
    
    // Add new projects as table rows
    projects.forEach(project => {
        const row = document.createElement('tr');
        
        // Calculate progress percentage
        const targetAmount = parseFloat(project.target_amount || 0);
        const currentAmount = parseFloat(project.current_amount || 0);
        const progressPercentage = targetAmount > 0 ? Math.min(100, (currentAmount / targetAmount) * 100) : 0;
        
        // Format currency
        const formatCurrency = (amount) => {
            return 'Ksh ' + parseFloat(amount || 0).toLocaleString();
        };
        
        // Format date
        const formatDate = (dateString) => {
            if (!dateString || dateString === '0000-00-00') return 'N/A';
            const date = new Date(dateString);
            return isNaN(date.getTime()) ? 'N/A' : date.toLocaleDateString('en-US');
        };
        
        row.innerHTML = `
            <td>${project.name}</td>
            <td><span class="project-status ${project.member_project_status === 'Active' ? 'status-active' : 'status-completed'}">${project.member_project_status}</span></td>
            <td>${project.role || 'N/A'}</td>
            <td>${formatDate(project.start_date)}</td>
            <td>${formatDate(project.end_date)}</td>
            <td>
                <div class="progress-bar-container">
                    <div class="progress-bar" style="width: ${progressPercentage}%"></div>
                </div>
                <div class="progress-text">${progressPercentage.toFixed(1)}%</div>
            </td>
            <td>${formatCurrency(project.contribution_amount)}</td>
            <td>
                <div class="action-buttons">
                    <a href="projects.html?id=${project.id}" class="btn btn-primary btn-sm">View Details</a>
                </div>
            </td>
        `;
        tableBody.appendChild(row);
    });
}

// Helper function to format dates
function formatDate(dateString) {
    // Handle null, empty, or invalid dates
    if (!dateString || dateString === '0000-00-00') return 'N/A';
    
    const date = new Date(dateString);
    
    // Check if date is valid
    if (isNaN(date.getTime())) return 'N/A';
    
    return date.toLocaleDateString('en-US', { 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    });
}

// Function to manually refresh the projects page
function refreshMemberProjects() {
    const memberId = getMemberId();
    if (memberId) {
        loadProjectsData(memberId);
    }
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshMemberProjects();
    }
});