// MFT GROUP Member Management Real-Time Updates
// This script handles real-time updates for the member management dashboard

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize real-time client
    if (typeof mftRealTime !== 'undefined') {
        mftRealTime.connect('admin');
        
        // Register update callback
        mftRealTime.onUpdate(function(data, pageType) {
            if (pageType === 'admin') {
                updateMemberManagementRealTime(data);
            }
        });
        
        // Load initial data
        loadMemberData();
        
        console.log('Real-time updates started for member management dashboard');
    } else {
        console.warn('Real-time client not found. Real-time updates not started.');
        // Load initial data even without real-time
        loadMemberData();
    }
});

// Function to load initial member data
async function loadMemberData() {
    try {
        // Show loading states
        showLoadingStates();
        
        // Fetch data from API
        const members = await fetchMembers();
        const pendingApplications = await fetchPendingApplications();
        const projects = await fetchProjects();
        
        // Update the dashboard with fetched data
        if (members) {
            updateMemberTable(members);
        }
        
        if (pendingApplications) {
            updateApplicationsTable(pendingApplications);
        }
        
        if (projects) {
            updateProjectAssignments(projects);
        }
    } catch (error) {
        console.error('Error loading member data:', error);
        showErrorStates();
    }
}

// Function to show loading states
function showLoadingStates() {
    // Update member table
    const memberTableBody = document.getElementById('memberTableBody');
    if (memberTableBody) {
        memberTableBody.innerHTML = `
            <tr>
                <td colspan="7" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading members...</div>
                </td>
            </tr>`;
    }
    
    // Update applications table
    const applicationsTableBody = document.getElementById('applicationsTableBody');
    if (applicationsTableBody) {
        applicationsTableBody.innerHTML = `
            <tr>
                <td colspan="7" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading applications...</div>
                </td>
            </tr>`;
    }
    
    // Update renewals table
    const renewalsTableBody = document.querySelector('#renewals-tab .mft-table tbody');
    if (renewalsTableBody) {
        renewalsTableBody.innerHTML = `
            <tr>
                <td colspan="6" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading renewals...</div>
                </td>
            </tr>`;
    }
    
    // Update attendance table
    const attendanceTableBody = document.querySelector('#renewals-tab .mft-table:nth-child(2) tbody');
    if (attendanceTableBody) {
        attendanceTableBody.innerHTML = `
            <tr>
                <td colspan="4" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading attendance...</div>
                </td>
            </tr>`;
    }
    
    // Update project assignments
    const projectsContainer = document.getElementById('projectsCheckboxes');
    if (projectsContainer) {
        projectsContainer.innerHTML = '<div class="loading-state"><i class="fas fa-spinner"></i><div>Loading projects...</div></div>';
    }
}

// Function to show error states
function showErrorStates() {
    // Update member table
    const memberTableBody = document.getElementById('memberTableBody');
    if (memberTableBody) {
        memberTableBody.innerHTML = '<tr><td colspan="7" style="text-align: center; color: #F44336;">Error loading members</td></tr>';
    }
    
    // Update applications table
    const applicationsTableBody = document.getElementById('applicationsTableBody');
    if (applicationsTableBody) {
        applicationsTableBody.innerHTML = '<tr><td colspan="7" style="text-align: center; color: #F44336;">Error loading applications</td></tr>';
    }
    
    // Update renewals table
    const renewalsTableBody = document.querySelector('#renewals-tab .member-table tbody');
    if (renewalsTableBody) {
        renewalsTableBody.innerHTML = '<tr><td colspan="6" style="text-align: center; color: #F44336;">Error loading renewals</td></tr>';
    }
    
    // Update attendance table
    const attendanceTableBody = document.querySelector('#renewals-tab .member-table:nth-child(2) tbody');
    if (attendanceTableBody) {
        attendanceTableBody.innerHTML = '<tr><td colspan="4" style="text-align: center; color: #F44336;">Error loading attendance</td></tr>';
    }
    
    // Update project assignments
    const projectsContainer = document.getElementById('projectsCheckboxes');
    if (projectsContainer) {
        projectsContainer.innerHTML = '<div style="text-align: center; padding: 20px; color: #F44336;">Error loading projects</div>';
    }
}

// Function to fetch members
async function fetchMembers() {
    try {
        const response = await fetch(getApiUrl('members.php'));
        const data = await response.json();
        return data.members || [];
    } catch (error) {
        console.error('Error fetching members:', error);
        return null;
    }
}

// Function to fetch pending applications
async function fetchPendingApplications() {
    try {
        const response = await fetch(getApiUrl('members.php?pending=true'));
        const data = await response.json();
        return data.applications || [];
    } catch (error) {
        console.error('Error fetching pending applications:', error);
        return null;
    }
}

// Function to fetch projects
async function fetchProjects() {
    try {
        // Fetch real data from the API
        const response = await fetch(getApiUrl('projects.php'));
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        return data.projects || [];
    } catch (error) {
        console.error('Error fetching projects:', error);
        return null;
    }
}

// Function to update the member management page with real-time data
function updateMemberManagementRealTime(data) {
    console.log('Updating member management with real-time data:', data);
    
    // Update different sections based on the data received
    if (data.members) {
        updateMemberTable(data.members);
    }
    
    if (data.applications) {
        updateApplicationsTable(data.applications);
    }
    
    if (data.projects) {
        updateProjectAssignments(data.projects);
    }
    
    if (data.renewals) {
        updateRenewalsTable(data.renewals);
    }
    
    if (data.attendance) {
        updateAttendanceTable(data.attendance);
    }
    
    // Show notification about update
    if (typeof showNotification === 'function') {
        showNotification('Member data updated with latest information', 'success');
    }
}

// Function to update member table
function updateMemberTable(members) {
    const memberTableBody = document.getElementById('memberTableBody');
    if (!memberTableBody) return;
    
    if (!members || members.length === 0) {
        memberTableBody.innerHTML = `
            <tr>
                <td colspan="7" class="empty-state">
                    <i class="fas fa-users"></i>
                    <h3>No Members Found</h3>
                    <p>There are no members in the system yet.</p>
                </td>
            </tr>`;
        return;
    }
    
    // Clear existing rows
    memberTableBody.innerHTML = '';
    
    // Add new rows
    members.forEach(member => {
        const row = document.createElement('tr');
        const formattedDate = formatDate(member.registration_date);
        
        row.innerHTML = `
            <td>${member.member_id}</td>
            <td>${member.first_name} ${member.last_name}</td>
            <td>${member.phone || 'N/A'}</td>
            <td><span class="status-badge status-${member.status.toLowerCase()}">${member.status}</span></td>
            <td>${formattedDate}</td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline" title="View Profile" data-member-id="${member.id}">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-sm btn-primary" title="Edit" data-member-id="${member.id}">
                    <i class="fas fa-edit"></i>
                </button>
                <button class="btn btn-sm btn-${member.status === 'Active' ? 'danger' : 'success'}" title="${member.status === 'Active' ? 'Deactivate' : 'Activate'}" data-member-id="${member.id}">
                    <i class="fas fa-${member.status === 'Active' ? 'times' : 'check'}"></i>
                </button>
            </td>
        `;
        memberTableBody.appendChild(row);
    });
    
    // Reattach event listeners to action buttons
    updateActionButtons();
    
    // Add sorting functionality
    addSortingToTable('memberTableBody');
}

// Function to update applications table
function updateApplicationsTable(applications) {
    const applicationsTableBody = document.getElementById('applicationsTableBody');
    if (!applicationsTableBody) return;
    
    if (!applications || applications.length === 0) {
        applicationsTableBody.innerHTML = '<tr><td colspan="7" style="text-align: center;">No pending applications found</td></tr>';
        return;
    }
    
    // Clear existing rows
    applicationsTableBody.innerHTML = '';
    
    // Add new rows
    applications.forEach(application => {
        const row = document.createElement('tr');
        const formattedDate = formatDate(application.created_at);
        
        row.innerHTML = `
            <td>${application.member_id}</td>
            <td>${application.first_name} ${application.last_name}</td>
            <td>${application.phone || 'N/A'}</td>
            <td>${application.id_number || 'N/A'}</td>
            <td><span class="status-badge status-pending">Pending</span></td>
            <td>${formattedDate}</td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline" title="View Details" data-app-id="${application.id}">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-sm btn-primary" title="Approve" data-app-id="${application.id}">
                    <i class="fas fa-check"></i>
                </button>
                <button class="btn btn-sm btn-danger" title="Reject" data-app-id="${application.id}">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        `;
        applicationsTableBody.appendChild(row);
    });
    
    // Reattach event listeners to action buttons
    updateApplicationActionButtons();
}

// Function to update project assignments
function updateProjectAssignments(projects) {
    const projectsContainer = document.getElementById('projectsCheckboxes');
    if (!projectsContainer) return;
    
    if (!projects || projects.length === 0) {
        projectsContainer.innerHTML = '<div style="text-align: center; padding: 20px;">No projects available</div>';
        return;
    }
    
    // Clear existing content
    projectsContainer.innerHTML = '';
    
    // Add new project checkboxes
    projects.forEach(project => {
        const projectDiv = document.createElement('div');
        projectDiv.style.display = 'flex';
        projectDiv.style.alignItems = 'center';
        projectDiv.style.padding = '8px 0';
        projectDiv.style.borderBottom = '1px solid var(--gray-dark)';
        
        projectDiv.innerHTML = `
            <input type="checkbox" id="project${project.id}" value="${project.id}" style="margin-right: 10px;">
            <label for="project${project.id}" style="flex: 1;">${project.name}</label>
            <span style="color: var(--gray); font-size: 0.9rem;">Completion: ${formatDate(project.completion_date)} | Location: ${project.location}</span>
        `;
        projectsContainer.appendChild(projectDiv);
    });
}

// Function to update renewals table
function updateRenewalsTable(renewals) {
    const renewalsTableBody = document.querySelector('#renewals-tab .member-table tbody');
    if (!renewalsTableBody) return;
    
    if (!renewals || renewals.length === 0) {
        renewalsTableBody.innerHTML = '<tr><td colspan="6" style="text-align: center;">No renewals found</td></tr>';
        return;
    }
    
    // Clear existing rows
    renewalsTableBody.innerHTML = '';
    
    // Add new rows
    renewals.forEach(renewal => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${renewal.memberName}</td>
            <td>${renewal.membershipStatus}</td>
            <td>${formatDate(renewal.renewalDate)}</td>
            <td>KSh ${renewal.amount.toLocaleString()}</td>
            <td><span class="status-badge status-${renewal.status.toLowerCase()}">${renewal.status}</span></td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline" title="View Details">
                    <i class="fas fa-eye"></i>
                </button>
            </td>
        `;
        renewalsTableBody.appendChild(row);
    });
}

// Function to update attendance table
function updateAttendanceTable(attendance) {
    const attendanceTableBody = document.querySelector('#renewals-tab .member-table:nth-child(2) tbody');
    if (!attendanceTableBody) return;
    
    if (!attendance || attendance.length === 0) {
        attendanceTableBody.innerHTML = '<tr><td colspan="4" style="text-align: center;">No attendance records found</td></tr>';
        return;
    }
    
    // Clear existing rows
    attendanceTableBody.innerHTML = '';
    
    // Add new rows
    attendance.forEach(record => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${formatDate(record.meetingDate)}</td>
            <td>${record.meetingTitle}</td>
            <td>${record.attendance}</td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline" title="View Details">
                    <i class="fas fa-eye"></i>
                </button>
            </td>
        `;
        attendanceTableBody.appendChild(row);
    });
}

// Helper function to format dates
function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
        year: 'numeric', 
        month: 'short', 
        day: 'numeric' 
    });
}

// Function to update action buttons
function updateActionButtons() {
    const actionButtons = document.querySelectorAll('#memberTableBody .action-cell .btn');
    actionButtons.forEach(button => {
        // Remove existing event listeners by cloning
        const newButton = button.cloneNode(true);
        button.parentNode.replaceChild(newButton, button);
        
        newButton.addEventListener('click', function() {
            const title = this.getAttribute('title');
            const memberId = this.getAttribute('data-member-id');
            const memberName = this.closest('tr').querySelector('td:nth-child(2)').textContent;
            
            if (title === 'View Profile') {
                // Switch to member profile tab
                switchToTab('member-profile');
                // Update profile information with member details
                updateProfileDetails(memberId, memberName);
                // Reset to view mode
                resetToViewMode();
            } else if (title === 'Edit') {
                // Switch to member profile tab in edit mode
                switchToTab('member-profile');
                // Enable editing mode
                enableEditingMode(memberId, memberName);
            } else if (title === 'Deactivate') {
                if (confirm(`Are you sure you want to deactivate member ${memberName}?`)) {
                    alert(`Member ${memberName} deactivated successfully!`);
                    // Here you would typically make an API call to update the member status
                    this.closest('tr').querySelector('.status-badge').className = 'status-badge status-inactive';
                    this.closest('tr').querySelector('.status-badge').textContent = 'Inactive';
                    // Change button to Activate
                    this.title = 'Activate';
                    this.className = 'btn btn-sm btn-success';
                    this.innerHTML = '<i class="fas fa-check"></i>';
                    // Reattach event listeners
                    updateActionButtons();
                }
            } else if (title === 'Activate') {
                if (confirm(`Are you sure you want to activate member ${memberName}?`)) {
                    alert(`Member ${memberName} activated successfully!`);
                    // Here you would typically make an API call to update the member status
                    this.closest('tr').querySelector('.status-badge').className = 'status-badge status-active';
                    this.closest('tr').querySelector('.status-badge').textContent = 'Active';
                    // Change button to Deactivate
                    this.title = 'Deactivate';
                    this.className = 'btn btn-sm btn-danger';
                    this.innerHTML = '<i class="fas fa-times"></i>';
                    // Reattach event listeners
                    updateActionButtons();
                }
            }
        });
    });
}

// Function to update application action buttons
function updateApplicationActionButtons() {
    const actionButtons = document.querySelectorAll('#applicationsTableBody .action-cell .btn');
    actionButtons.forEach(button => {
        // Remove existing event listeners by cloning
        const newButton = button.cloneNode(true);
        button.parentNode.replaceChild(newButton, button);
        
        newButton.addEventListener('click', function() {
            const title = this.getAttribute('title');
            const appId = this.getAttribute('data-app-id');
            const appName = this.closest('tr').querySelector('td:nth-child(2)').textContent;
            
            if (title === 'View Details') {
                // Show application details (in a real app, this would show a modal with full details)
                alert(`Application details for ${appName} (${appId}) would be shown here.`);
            } else if (title === 'Approve') {
                approveApplication(appId);
            } else if (title === 'Reject') {
                rejectApplication(appId);
            }
        });
    });
}

// Function to switch to a specific tab
function switchToTab(tabName) {
    const tabs = document.querySelectorAll('.nav-tab');
    const tabContents = document.querySelectorAll('.tab-content');
    
    // Remove active class from all tabs and contents
    tabs.forEach(t => t.classList.remove('active'));
    tabContents.forEach(c => c.classList.remove('active'));
    
    // Add active class to clicked tab
    document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
    document.getElementById(`${tabName}-tab`).classList.add('active');
}

// Function to update profile details
function updateProfileDetails(memberId, memberName) {
    // This would typically fetch data from a server
    // For now, we'll just update the header
    document.querySelector('.profile-header h2').textContent = memberName;
    document.querySelector('.profile-detail:first-child strong').nextSibling.textContent = ' ' + memberId;
}

// Function to reset to view mode
function resetToViewMode() {
    // Hide save/cancel buttons
    document.getElementById('saveProfileBtn').style.display = 'none';
    document.getElementById('cancelEditBtn').style.display = 'none';
    // Show edit button
    document.getElementById('editProfileBtn').style.display = 'inline-block';
    
    // Disable form fields
    const formInputs = document.querySelectorAll('#member-profile-tab input, #member-profile-tab textarea, #member-profile-tab select');
    formInputs.forEach(input => {
        if (input.type !== 'file') {
            input.setAttribute('readonly', true);
            input.disabled = true;
        }
    });
}

// Function to enable editing mode
function enableEditingMode(memberId, memberName) {
    // Update profile header
    document.querySelector('.profile-header h2').textContent = memberName;
    document.querySelector('.profile-detail:first-child strong').nextSibling.textContent = ' ' + memberId;
    
    // Hide edit button
    document.getElementById('editProfileBtn').style.display = 'none';
    // Show save/cancel buttons
    document.getElementById('saveProfileBtn').style.display = 'inline-block';
    document.getElementById('cancelEditBtn').style.display = 'inline-block';
    
    // Enable form fields for editing
    const formInputs = document.querySelectorAll('#member-profile-tab input, #member-profile-tab textarea, #member-profile-tab select');
    formInputs.forEach(input => {
        if (input.type !== 'file') {
            input.removeAttribute('readonly');
            input.disabled = false;
        }
    });
}

// Function to approve a pending application
function approveApplication(appId) {
    // Show loading indicator
    const approveButton = document.querySelector(`[data-app-id="${appId}"][title="Approve"]`);
    const originalText = approveButton.innerHTML;
    approveButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
    approveButton.disabled = true;
    
    // Send approval request to server
    fetch(`api/members.php?id=${appId}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            status: 'Active'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.error) {
            throw new Error(data.error);
        }
        
        alert('Application approved successfully!');
        
        // Reload both tables
        loadMemberData();
    })
    .catch(error => {
        console.error('Error approving application:', error);
        alert('Error approving application: ' + error.message);
    })
    .finally(() => {
        // Restore approve button
        approveButton.innerHTML = originalText;
        approveButton.disabled = false;
    });
}

// Function to reject a pending application
function rejectApplication(appId) {
    if (!confirm('Are you sure you want to reject this application?')) {
        return;
    }
    
    // Show loading indicator
    const rejectButton = document.querySelector(`[data-app-id="${appId}"][title="Reject"]`);
    const originalText = rejectButton.innerHTML;
    rejectButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
    rejectButton.disabled = true;
    
    // Send delete request to server
    fetch(`api/members.php?id=${appId}`, {
        method: 'DELETE'
    })
    .then(response => response.json())
    .then(data => {
        if (data.error) {
            throw new Error(data.error);
        }
        
        alert('Application rejected successfully!');
        
        // Reload applications table
        loadMemberData();
    })
    .catch(error => {
        console.error('Error rejecting application:', error);
        alert('Error rejecting application: ' + error.message);
    })
    .finally(() => {
        // Restore reject button
        rejectButton.innerHTML = originalText;
        rejectButton.disabled = false;
    });
}

// Function to manually refresh the member management dashboard
function refreshMemberManagement() {
    loadMemberData();
}

// Function to add sorting to table
function addSortingToTable(tableBodyId) {
    const tableHeaders = document.querySelectorAll(`#${tableBodyId}`).closest('.mft-table').querySelectorAll('th.sortable-header');
    
    tableHeaders.forEach(header => {
        header.addEventListener('click', function() {
            const sortKey = this.getAttribute('data-sort');
            const isAscending = !this.classList.contains('asc');
            
            // Update header classes
            tableHeaders.forEach(h => h.classList.remove('asc', 'desc'));
            this.classList.toggle(isAscending ? 'asc' : 'desc', true);
            
            // Get table body and rows
            const tableBody = document.getElementById(tableBodyId);
            const rows = Array.from(tableBody.querySelectorAll('tr'));
            
            // Sort rows
            rows.sort((a, b) => {
                const aText = a.cells[Array.from(header.parentNode.children).indexOf(header)].textContent.trim();
                const bText = b.cells[Array.from(header.parentNode.children).indexOf(header)].textContent.trim();
                
                // Handle numeric values
                const aNum = parseFloat(aText.replace(/[^0-9.-]+/g, ''));
                const bNum = parseFloat(bText.replace(/[^0-9.-]+/g, ''));
                
                if (!isNaN(aNum) && !isNaN(bNum)) {
                    return isAscending ? aNum - bNum : bNum - aNum;
                }
                
                // Handle date values
                const aDate = new Date(aText);
                const bDate = new Date(bText);
                
                if (aDate !== 'Invalid Date' && bDate !== 'Invalid Date') {
                    return isAscending ? aDate - bDate : bDate - aDate;
                }
                
                // Handle text values
                return isAscending ? 
                    aText.localeCompare(bText, undefined, {numeric: true}) : 
                    bText.localeCompare(aText, undefined, {numeric: true});
            });
            
            // Re-append sorted rows
            rows.forEach(row => tableBody.appendChild(row));
        });
    });
}

// Function to show notification (fallback if not defined elsewhere)
function showNotification(message, type = 'info') {
    // Create notification element if it doesn't exist
    let notificationContainer = document.getElementById('realtime-notification');
    if (!notificationContainer) {
        notificationContainer = document.createElement('div');
        notificationContainer.id = 'realtime-notification';
        notificationContainer.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 8px;
            color: white;
            font-weight: 500;
            z-index: 10000;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateX(120%);
            transition: transform 0.3s ease;
        `;
        document.body.appendChild(notificationContainer);
    }
    
    // Set notification style based on type
    switch(type) {
        case 'success':
            notificationContainer.style.background = 'linear-gradient(135deg, #4CAF50, #2E7D32)';
            break;
        case 'error':
            notificationContainer.style.background = 'linear-gradient(135deg, #F44336, #D32F2F)';
            break;
        case 'warning':
            notificationContainer.style.background = 'linear-gradient(135deg, #FFC107, #FFA000)';
            break;
        default:
            notificationContainer.style.background = 'linear-gradient(135deg, #2196F3, #1976D2)';
    }
    
    // Set message and show
    notificationContainer.textContent = message;
    notificationContainer.style.transform = 'translateX(0)';
    
    // Auto-hide after 3 seconds
    setTimeout(() => {
        notificationContainer.style.transform = 'translateX(120%)';
    }, 3000);
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshMemberManagement();
    }
});