// MFT GROUP Member Dashboard Real-Time Updates
// This script handles real-time updates for the member dashboard

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Get member ID from localStorage or other source
    // In a real implementation, you would get this from authentication
    const memberId = getMemberId();
    
    if (memberId) {
        // Initialize real-time client with correct dashboard type
        mftRealTime.connect('member_dashboard', memberId);
        
        // Register update callback
        mftRealTime.onUpdate(function(data, dashboardType) {
            if (dashboardType === 'member_dashboard') {
                updateMemberDashboardRealTime(data);
            }
        });
        
        console.log('Real-time updates started for member dashboard');
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
    }
});

// Function to get member ID (this would be implemented based on your auth system)
function getMemberId() {
    // For now, we'll try to get it from localStorage
    // In a real implementation, this would come from your authentication system
    return localStorage.getItem('memberId') || null; // Return null instead of defaulting to '1'
}

// Function to update the member dashboard with real-time data
function updateMemberDashboardRealTime(data) {
    console.log('Updating member dashboard with real-time data:', data);
    
    // Handle different types of notifications
    if (data.type === 'contribution_update') {
        handleContributionUpdate(data.data);
    } else if (data.type === 'loan_update') {
        handleLoanUpdate(data.data);
    } else if (data.type === 'fine_update') {
        handleFineUpdate(data.data);
    } else if (data.type === 'settings_update') {
        handleSettingsUpdate(data.data);
    } else if (data.type === 'project_update') {
        handleProjectUpdate(data.data);
    } else if (data.type === 'member_notification') {
        handleMemberNotification(data.data);
    } else if (data.type === 'dashboard_update') {
        handleGeneralDashboardUpdate(data.data);
    } else {
        // Handle general dashboard updates
        handleGeneralDashboardUpdate(data);
    }
}

// Handle contribution updates
function handleContributionUpdate(data) {
    console.log('Processing contribution update:', data);
    
    // Update contributions card
    if (data.contributions && data.contributions.total_balance !== undefined) {
        updateContributionsCard({ contributions: data.contributions });
    } else if (data.contributions !== undefined) {
        updateContributionsCard({ total_balance: data.contributions });
    }
    
    // Show notification
    showNotification(`New contribution has been recorded`, 'success');
}

// Handle loan updates
function handleLoanUpdate(data) {
    console.log('Processing loan update:', data);
    
    // Update loans card
    if (data.loans && data.loans.outstanding_balance !== undefined) {
        updateLoansCard({ loans: data.loans });
    } else if (data.loans !== undefined) {
        updateLoansCard({ outstanding_balance: data.loans });
    }
    
    // Show notification
    let message = 'Loan status has been updated';
    if (data.action === 'applied') {
        message = `New loan application submitted by admin for KSH ${Number(data.loanData.amount).toLocaleString()}`;
    } else if (data.action === 'approved') {
        message = 'Your loan application has been approved';
    } else if (data.action === 'rejected') {
        message = 'Your loan application has been rejected';
    }
    
    showNotification(message, 'info');
    
    // Refresh loans section if on loans page
    if (typeof refreshLoansSection === 'function') {
        refreshLoansSection();
    }
}

// Handle fine updates
function handleFineUpdate(data) {
    console.log('Processing fine update:', data);
    
    // Update fines card
    if (data.fines && data.fines.total_fines !== undefined) {
        updateFinesCard({ fines: data.fines });
    } else if (data.fines !== undefined) {
        updateFinesCard({ total_fines: data.fines });
    }
    
    // Show notification
    showNotification(`Fine status has been updated`, 'info');
}

// Handle settings updates
function handleSettingsUpdate(data) {
    console.log('Processing settings update:', data);
    
    // In a real implementation, you would update UI elements based on changed settings
    showNotification('System settings have been updated', 'info');
}

// Handle project updates
function handleProjectUpdate(data) {
    console.log('Processing project update:', data);
    
    // Update projects card
    // In a real implementation, you would update project information
    showNotification(`Project has been updated`, 'info');
}

// Handle general dashboard updates
function handleGeneralDashboardUpdate(data) {
    console.log('Processing general dashboard update:', data);
    
    // Update member info in header
    if (data.member_info) {
        updateMemberHeader(data.member_info);
    }
    
    // Update contributions card
    if (data.contributions && data.contributions.total_balance !== undefined) {
        updateContributionsCard({ contributions: data.contributions });
    } else if (data.contributions !== undefined) {
        updateContributionsCard({ total_balance: data.contributions });
    }
    
    // Update loans card
    if (data.loans && data.loans.outstanding_balance !== undefined) {
        updateLoansCard({ loans: data.loans });
    } else if (data.loans !== undefined) {
        updateLoansCard({ outstanding_balance: data.loans });
    }
    
    // Update fines card
    if (data.fines && data.fines.total_fines !== undefined) {
        updateFinesCard({ fines: data.fines });
    } else if (data.fines !== undefined) {
        updateFinesCard({ total_fines: data.fines });
    }
    
    // Update missed payments card
    if (data.missed_payments) {
        updateMissedPaymentsCard(data.missed_payments.outstanding_balance, data.missed_payments.months_missed);
    }
    
    // Update projects card
    if (data.projects) {
        updateProjectsCard(data.projects);
    }
}

// Handle member-specific notifications
function handleMemberNotification(data) {
    console.log('Processing member notification:', data);
    
    // Show notification to member
    if (data.message) {
        showNotification(data.message, data.type || 'info');
    }
    
    // Handle specific notification types
    if (data.notificationType === 'loan_applied') {
        const loanAmount = data.principalAmount || data.loanAmount || 0;
        const formattedAmount = isNaN(Number(loanAmount)) ? 'KSH 0' : `KSH ${Number(loanAmount).toLocaleString()}`;
        showNotification(`Admin has applied for a loan on your behalf for ${formattedAmount}`, 'success');
        // In a real implementation, you would update the loans section
    } else if (data.notificationType === 'document_added') {
        showNotification('New document has been added to your account', 'info');
        // In a real implementation, you would refresh the documents section
    } else if (data.notificationType === 'meeting_reminder') {
        showNotification(`Reminder: ${data.meetingTitle} on ${data.meetingDate}`, 'info');
    }
}

// Function to show a notification to the user
function showNotification(message, type = 'info') {
    // Create a simple notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        background-color: ${type === 'success' ? '#4CAF50' : type === 'error' ? '#F44336' : '#2196F3'};
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        z-index: 10000;
        font-family: 'Inter', sans-serif;
    `;
    
    document.body.appendChild(notification);
    
    // Remove the notification after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 3000);
}

// Function to update member header information
function updateMemberHeader(memberInfo) {
    // Update greeting
    const greetingElement = document.getElementById('greeting');
    if (greetingElement) {
        // Get appropriate greeting based on time of day
        const greeting = getGreetingBasedOnTime();
        greetingElement.textContent = `${greeting}, ${memberInfo.first_name || 'Member'}`;
    }
    
    // Update member ID
    const memberIdElement = document.getElementById('memberIdValue');
    if (memberIdElement) {
        memberIdElement.textContent = memberInfo.member_id || 'MFT-XXXX-XXX';
    }
}

// Function to get appropriate greeting based on time of day
function getGreetingBasedOnTime() {
    const hours = new Date().getHours();
    
    if (hours >= 5 && hours < 12) {
        return 'Good morning';
    } else if (hours >= 12 && hours < 18) {
        return 'Good afternoon';
    } else {
        return 'Good evening';
    }
}

// Function to update contributions card
function updateContributionsCard(data) {
    console.log('Updating contributions card with:', data);
    
    // Get total balance from the correct field
    let totalBalance = 0;
    if (data.contributions && typeof data.contributions === 'object') {
        // Check if it's the dashboard API response format
        if (data.contributions.total_balance !== undefined) {
            totalBalance = parseFloat(data.contributions.total_balance) || 0;
        }
    } else if (data.total_balance !== undefined) {
        totalBalance = parseFloat(data.total_balance) || 0;
    }
    
    console.log('Total balance for contributions:', totalBalance);
    
    if (totalBalance !== undefined) {
        // Find all contributions cards
        const contributionCards = document.querySelectorAll('.dashboard-card.contributions');
        contributionCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(totalBalance).toLocaleString()}`;
            }
        });
    }
}

// Function to update loans card
function updateLoansCard(data) {
    console.log('Updating loans card with:', data);
    
    // Get outstanding balance from the correct field
    let outstandingBalance = 0;
    if (data.loans && typeof data.loans === 'object') {
        // Check if it's the dashboard API response format
        if (data.loans.outstanding_balance !== undefined) {
            outstandingBalance = parseFloat(data.loans.outstanding_balance) || 0;
        }
    } else if (data.outstanding_balance !== undefined) {
        outstandingBalance = parseFloat(data.outstanding_balance) || 0;
    }
    
    console.log('Outstanding balance for loans:', outstandingBalance);
    
    if (outstandingBalance !== undefined) {
        // Find all loans cards
        const loanCards = document.querySelectorAll('.dashboard-card.loans');
        loanCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(outstandingBalance).toLocaleString()}`;
            }
        });
    }
}

// Function to update fines card
function updateFinesCard(data) {
    console.log('Updating fines card with:', data);
    
    // Get total fines from the correct field
    let totalFines = 0;
    
    // Check if we have fines data as an object with total_fines field (from dashboard API)
    if (data.fines && typeof data.fines === 'object' && data.fines.total_fines !== undefined) {
        totalFines = parseFloat(data.fines.total_fines) || 0;
    }
    // Check if we have fines data as an object with total_fines field (from fines API response wrapped in object)
    else if (data.total_fines !== undefined) {
        totalFines = parseFloat(data.total_fines) || 0;
    }
    // Check if we have fines data as an array (from fines API - {fines: {count: 0, fines: []}})
    else if (data.fines && typeof data.fines === 'object' && Array.isArray(data.fines.fines)) {
        totalFines = data.fines.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    }
    // Check if we have fines data as an array directly (from fines API)
    else if (data.fines && Array.isArray(data.fines)) {
        totalFines = data.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    }
    
    console.log('Total fines:', totalFines);
    
    if (totalFines !== undefined) {
        // Find all fines cards
        const finesCards = document.querySelectorAll('.dashboard-card.fines');
        finesCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(totalFines).toLocaleString()}`;
            }
        });
    }
}

// Function to update projects card
function updateProjectsCard(data) {
    // Get projects data from the correct location
    let projects = [];
    if (data && Array.isArray(data)) {
        // Direct array from individual API
        projects = data;
    } else if (data && typeof data === 'object' && Array.isArray(data.projects)) {
        // From dashboard API
        projects = data.projects;
    }
    
    const projectCount = projects.length;
    
    // Update projects card
    const projectsCards = document.querySelectorAll('.dashboard-card.projects');
    projectsCards.forEach(card => {
        // Update card value (count of projects)
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = projectCount;
        }
        
        // Update card detail
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            detailElement.textContent = projectCount === 1 ? 'Project Assigned' : 'Projects Assigned';
        }
    });
}

// Function to update missed payments card
function updateMissedPaymentsCard(balance, monthsMissed) {
    // Update the missed payments card with real-time data
    const missedPaymentCards = document.querySelectorAll('.dashboard-card.events');
    missedPaymentCards.forEach(card => {
        // Update card value
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
        }
        
        // Update card detail with months missed information
        const detailElement = card.querySelector('.card-detail:last-child');
        if (detailElement) {
            detailElement.textContent = `${monthsMissed || 0} months missed`;
        }
    });
}

// Helper function to format project dates
function formatProjectDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const month = date.toLocaleDateString('en-US', { month: 'short' });
    const year = date.getFullYear();
    return `${month} ${year}`;
}

// Function to manually refresh the dashboard
function refreshMemberDashboard() {
    // Trigger an immediate update
    mftRealTime.pollForUpdates();
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshMemberDashboard();
    }
});