// MFT GROUP Fines Real-Time Updates
// This script handles real-time updates for the fines page

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Get member ID from localStorage or other source
    const memberId = getMemberId();
    
    if (memberId) {
        // Initialize real-time client
        mftRealTime.connect('member', memberId);
        
        // Register update callback
        mftRealTime.onUpdate(function(data, pageType) {
            if (pageType === 'fines') {
                updateFinesRealTime(data);
            }
        });
        
        // Load initial data
        loadFinesData(memberId);
        
        console.log('Real-time updates started for fines page');
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
    }
});

// Function to get member ID (this would be implemented based on your auth system)
function getMemberId() {
    // For now, we'll try to get it from localStorage
    // In a real implementation, this would come from your authentication system
    return localStorage.getItem('memberId') || '1'; // Default to member ID 1 for testing
}

// Function to load initial fines data
async function loadFinesData(memberId) {
    try {
        // Show loading state
        showLoadingState();
        
        // Fetch fines data from API
        const finesData = await mftApi.getMemberFines(memberId);
        
        if (finesData) {
            // Update the page with the fetched data
            updateFinesPage(finesData);
        } else {
            // Show error state
            showErrorState();
        }
    } catch (error) {
        console.error('Error loading fines data:', error);
        showErrorState();
    }
}

// Function to show loading state
function showLoadingState() {
    // Update summary cards with loading state
    const summaryCards = document.querySelectorAll('.summary-card .summary-value, .summary-card .summary-detail');
    summaryCards.forEach(card => {
        if (card.classList.contains('summary-value')) {
            card.textContent = '-';
        } else {
            card.textContent = 'Loading...';
        }
    });
    
    // Update statement info with loading state
    const infoValues = document.querySelectorAll('.info-value');
    infoValues.forEach(value => {
        value.textContent = '-';
    });
    
    // Update statement description
    const description = document.querySelector('.statement-description');
    if (description) {
        description.textContent = 'Loading statement information...';
    }
    
    // Update transactions table with loading state
    const transactionsTable = document.getElementById('fine-transactions');
    if (transactionsTable) {
        transactionsTable.innerHTML = '<tr><td colspan="6" style="text-align: center;">Loading fine transactions...</td></tr>';
    }
}

// Function to show error state
function showErrorState() {
    // Update summary cards with error state
    const summaryCards = document.querySelectorAll('.summary-card .summary-value, .summary-card .summary-detail');
    summaryCards.forEach(card => {
        if (card.classList.contains('summary-value')) {
            card.textContent = 'Error';
        } else {
            card.textContent = 'Failed to load data';
        }
    });
    
    // Update statement info with error state
    const infoValues = document.querySelectorAll('.info-value');
    infoValues.forEach(value => {
        value.textContent = 'Error';
    });
    
    // Update statement description
    const description = document.querySelector('.statement-description');
    if (description) {
        description.textContent = 'Error loading statement information. Please try again later.';
    }
    
    // Update transactions table with error state
    const transactionsTable = document.getElementById('fine-transactions');
    if (transactionsTable) {
        transactionsTable.innerHTML = '<tr><td colspan="6" style="text-align: center; color: #F44336;">Error loading fine transactions</td></tr>';
    }
}

// Function to update the fines page with real-time data
function updateFinesRealTime(data) {
    console.log('Updating fines with real-time data:', data);
    
    // Update fines information
    if (data.fines) {
        updateFinesPage(data.fines);
    }
}

// Function to update the fines page with data
function updateFinesPage(data) {
    // Update summary information
    if (data.summary) {
        updateSummaryCards(data.summary);
    }
    
    // Update statement information
    if (data.statement) {
        updateStatementInfo(data.statement);
    }
    
    // Update fine transactions
    if (data.transactions) {
        updateFineTransactions(data.transactions);
    }
}

// Function to update summary cards
function updateSummaryCards(summary) {
    // Update total fines incurred
    const incurredElement = document.querySelector('.summary-card:nth-child(1) .summary-value');
    if (incurredElement) {
        incurredElement.textContent = summary.totalIncurred ? `KSH ${summary.totalIncurred.toLocaleString()}` : 'KSH 0';
    }
    
    // Update total fines paid
    const paidElement = document.querySelector('.summary-card:nth-child(2) .summary-value');
    if (paidElement) {
        paidElement.textContent = summary.totalPaid ? `KSH ${summary.totalPaid.toLocaleString()}` : 'KSH 0';
    }
    
    // Update outstanding fines
    const outstandingElement = document.querySelector('.summary-card:nth-child(3) .summary-value');
    if (outstandingElement) {
        outstandingElement.textContent = summary.outstanding ? `KSH ${summary.outstanding.toLocaleString()}` : 'KSH 0';
    }
    
    // Update detail information
    const incurredDetail = document.querySelector('.summary-card:nth-child(1) .summary-detail');
    if (incurredDetail) {
        incurredDetail.textContent = summary.incurredPeriod || 'Financial Year 2024/2025';
    }
    
    const paidDetail = document.querySelector('.summary-card:nth-child(2) .summary-detail');
    if (paidDetail) {
        paidDetail.textContent = summary.paidPeriod || 'Financial Year 2024/2025';
    }
    
    const outstandingDetail = document.querySelector('.summary-card:nth-child(3) .summary-detail');
    if (outstandingDetail) {
        outstandingDetail.textContent = summary.outstandingDescription || 'Pending payment';
    }
}

// Function to update statement information
function updateStatementInfo(statement) {
    // Update statement period
    const periodElement = document.querySelector('.info-item:nth-child(1) .info-value');
    if (periodElement) {
        periodElement.textContent = statement.period || '-';
    }
    
    // Update member ID
    const memberIdElement = document.querySelector('.info-item:nth-child(2) .info-value');
    if (memberIdElement) {
        memberIdElement.textContent = statement.memberId || '-';
    }
    
    // Update member name
    const memberNameElement = document.querySelector('.info-item:nth-child(3) .info-value');
    if (memberNameElement) {
        memberNameElement.textContent = statement.memberName || '-';
    }
    
    // Update statement date
    const statementDateElement = document.querySelector('.info-item:nth-child(4) .info-value');
    if (statementDateElement) {
        statementDateElement.textContent = statement.date || '-';
    }
    
    // Update statement description
    const description = document.querySelector('.statement-description');
    if (description) {
        description.textContent = statement.description || 'This statement shows the fines incurred and paid during the financial year.';
    }
}

// Function to update fine transactions
function updateFineTransactions(transactions) {
    const transactionsTable = document.getElementById('fine-transactions');
    if (!transactionsTable) return;
    
    if (!transactions || transactions.length === 0) {
        transactionsTable.innerHTML = '<tr><td colspan="6" style="text-align: center;">No fine transactions found</td></tr>';
        return;
    }
    
    // Clear existing transactions
    transactionsTable.innerHTML = '';
    
    // Add new transactions
    transactions.forEach(transaction => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${transaction.date || '-'}</td>
            <td>${transaction.reason || '-'}</td>
            <td>${transaction.description || '-'}</td>
            <td class="negative">${transaction.amount ? `KSH ${transaction.amount.toLocaleString()}` : 'KSH 0'}</td>
            <td>${transaction.status || '-'}</td>
            <td>${transaction.paymentDate || '-'}</td>
        `;
        transactionsTable.appendChild(row);
    });
}

// Function to manually refresh the fines page
function refreshFines() {
    const memberId = getMemberId();
    if (memberId) {
        loadFinesData(memberId);
    }
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshFines();
    }
});