// MFT GROUP Dashboard Real-Time Updates
// This script handles real-time updates for the main member dashboard

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Get member ID from localStorage
    const memberId = localStorage.getItem('memberId');
    
    if (memberId) {
        // Initialize real-time client
        if (typeof mftRealTime !== 'undefined') {
            mftRealTime.connect('member_dashboard', memberId);
            
            // Register update callback
            mftRealTime.onUpdate(function(data, pageType) {
                if (pageType === 'member_dashboard') {
                    updateDashboardRealTime(data);
                }
            });
            
            console.log('Real-time updates started for member dashboard');
        } else {
            console.warn('Real-time client not found. Real-time updates not started.');
        }
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
    }
});

// Function to update the dashboard with real-time data
function updateDashboardRealTime(data) {
    console.log('Updating dashboard with real-time data:', data);
    
    // Update contributions information
    if (data.contributions) {
        updateContributionsInfo(data.contributions);
    }
    
    // Update fines information
    if (data.fines) {
        updateFinesInfo(data.fines);
    }
    
    // Update loans information
    if (data.loans) {
        updateLoansInfo(data.loans);
    }
    
    // Update dividends information
    if (data.dividends) {
        updateDividendsInfo(data.dividends);
    }
    
    // Update projects information
    if (data.projects) {
        updateProjectsInfo(data.projects);
    }
    
    // Update profile information
    if (data.member_info) {
        updateMemberInfo(data.member_info);
    }
    
    // Show notification about update
    if (typeof showNotification === 'function') {
        showNotification('Dashboard updated with latest information', 'success');
    }
}

// Function to update contributions information
function updateContributionsInfo(contributions) {
    try {
        const contributionsElement = document.getElementById('totalContributions');
        if (contributionsElement) {
            contributionsElement.textContent = 
                contributions.total_balance ? `KSH ${Number(contributions.total_balance).toLocaleString()}` : 'KSH 0';
        }
        
        // Update the card status with timestamp
        const contributionsStatus = document.querySelector('.contributions .card-status');
        if (contributionsStatus) {
            contributionsStatus.textContent = 'Updated just now';
            contributionsStatus.className = 'card-status status-active';
        }
    } catch (error) {
        console.error('Error updating contributions info:', error);
    }
}

// Function to update fines information
function updateFinesInfo(fines) {
    try {
        const finesElement = document.getElementById('pendingFines');
        if (finesElement) {
            finesElement.textContent = 
                fines.total_fines ? `KSH ${Number(fines.total_fines).toLocaleString()}` : 'KSH 0';
        }
        
        // Update the card status with timestamp
        const finesStatus = document.querySelector('.fines .card-status');
        if (finesStatus) {
            finesStatus.textContent = fines.total_fines > 0 ? `${fines.total_fines} pending` : 'No pending fines';
            finesStatus.className = fines.total_fines > 0 ? 'card-status status-warning' : 'card-status status-active';
        }
    } catch (error) {
        console.error('Error updating fines info:', error);
    }
}

// Function to update loans information
function updateLoansInfo(loans) {
    try {
        const loansElement = document.getElementById('activeLoans');
        if (loansElement) {
            loansElement.textContent = 
                loans.outstanding_balance ? `KSH ${Number(loans.outstanding_balance).toLocaleString()}` : 'KSH 0';
        }
        
        // Update the card status with timestamp
        const loansStatus = document.querySelector('.loans .card-status');
        if (loansStatus) {
            loansStatus.textContent = loans.active_loans > 0 ? `${loans.active_loans} payments due` : '0 payments due';
            loansStatus.className = loans.active_loans > 0 ? 'card-status status-warning' : 'card-status status-active';
        }
    } catch (error) {
        console.error('Error updating loans info:', error);
    }
}

// Function to update dividends information
function updateDividendsInfo(dividends) {
    try {
        const dividendsElement = document.getElementById('attendanceRate');
        if (dividendsElement) {
            dividendsElement.textContent = 
                dividends.total_dividends ? `KSH ${Number(dividends.total_dividends).toLocaleString()}` : 'KSH 0';
        }
        
        // Update the card status with timestamp
        const dividendsStatus = document.querySelector('.dividends .card-status');
        if (dividendsStatus) {
            dividendsStatus.textContent = 'Updated just now';
            dividendsStatus.className = 'card-status status-active';
        }
    } catch (error) {
        console.error('Error updating dividends info:', error);
    }
}

// Function to update projects information
function updateProjectsInfo(projects) {
    try {
        const projectsElement = document.getElementById('myProjectsCount');
        if (projectsElement) {
            projectsElement.textContent = projects.length || '0';
        }
        
        // Update the card status with timestamp
        const projectsStatus = document.querySelector('.projects .card-status');
        if (projectsStatus) {
            projectsStatus.textContent = 'Updated just now';
            projectsStatus.className = 'card-status status-active';
        }
    } catch (error) {
        console.error('Error updating projects info:', error);
    }
}

// Function to update member information
function updateMemberInfo(memberInfo) {
    try {
        // Update member ID
        const memberIdElement = document.getElementById('memberIdValue');
        if (memberIdElement) {
            memberIdElement.textContent = memberInfo.member_id || 'N/A';
        }
        
        // Update registration date
        const regDateElement = document.getElementById('registrationDateValue');
        if (regDateElement) {
            regDateElement.textContent = memberInfo.registration_date || 'N/A';
        }
        
        // Update member status
        const statusElement = document.getElementById('memberStatusValue');
        if (statusElement) {
            statusElement.textContent = memberInfo.status || 'Inactive';
            statusElement.className = 'info-card-value ' + (memberInfo.status === 'Active' ? 'status-active-card' : '');
        }
        
        // Update member name
        const nameElement = document.getElementById('memberName');
        if (nameElement) {
            nameElement.textContent = `${memberInfo.first_name || ''} ${memberInfo.last_name || ''}`.trim() || 'Member';
        }
    } catch (error) {
        console.error('Error updating member info:', error);
    }
}

// Function to show notification (fallback if not defined elsewhere)
function showNotification(message, type = 'info') {
    // Create notification element if it doesn't exist
    let notificationContainer = document.getElementById('realtime-notification');
    if (!notificationContainer) {
        notificationContainer = document.createElement('div');
        notificationContainer.id = 'realtime-notification';
        notificationContainer.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 8px;
            color: white;
            font-weight: 500;
            z-index: 10000;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateX(120%);
            transition: transform 0.3s ease;
        `;
        document.body.appendChild(notificationContainer);
    }
    
    // Set notification style based on type
    switch(type) {
        case 'success':
            notificationContainer.style.background = 'linear-gradient(135deg, #4CAF50, #2E7D32)';
            break;
        case 'error':
            notificationContainer.style.background = 'linear-gradient(135deg, #F44336, #D32F2F)';
            break;
        case 'warning':
            notificationContainer.style.background = 'linear-gradient(135deg, #FFC107, #FFA000)';
            break;
        default:
            notificationContainer.style.background = 'linear-gradient(135deg, #2196F3, #1976D2)';
    }
    
    // Set message and show
    notificationContainer.textContent = message;
    notificationContainer.style.transform = 'translateX(0)';
    
    // Auto-hide after 3 seconds
    setTimeout(() => {
        notificationContainer.style.transform = 'translateX(120%)';
    }, 3000);
}