// MFT GROUP Dashboard Data Loader
// This script fetches and displays member data on the dashboard

document.addEventListener('DOMContentLoaded', function() {
    console.log('=== DASHBOARD DATA LOADER INITIALIZED ===');
    
    // Get member ID from localStorage
    const memberId = localStorage.getItem('memberId');
    console.log('Member ID from localStorage:', memberId);
    
    // Add event listeners for action buttons
    addActionButtonListeners();
    
    // Keep trying to attach event listeners in case of timing issues
    const interval = setInterval(addActionButtonListeners, 1000);
    setTimeout(() => clearInterval(interval), 10000); // Stop after 10 seconds
    
    if (memberId) {
        console.log('Member ID found, loading member data');
        // Load member data
        loadMemberData(memberId);
        
        // Load dashboard data using optimized method
        loadDashboardDataOptimized(memberId);
    } else {
        console.error('Member ID not found in localStorage');
    }
});

// Add a fallback to ensure data is loaded even if DOMContentLoaded fires too early
if (document.readyState === 'loading') {
    // Document is still loading, normal flow will handle it
    console.log('Document is still loading');
} else {
    // Document has already loaded, check if we need to load data
    console.log('Document already loaded, checking if data needs to be loaded');
    const memberId = localStorage.getItem('memberId');
    if (memberId && !window.memberDataLoaded) {
        console.log('Loading data on fallback');
        loadMemberData(memberId);
        loadDashboardDataOptimized(memberId);
        window.memberDataLoaded = true;
    }
}

// Function to add event listeners for action buttons
function addActionButtonListeners() {
    console.log('=== BUTTON DEBUGGING START ===');
    
    // Test if we can access the document
    console.log('Document ready state:', document.readyState);
    
    // Test if we can find the buttons
    const makeContributionBtn = document.getElementById('makeContributionBtn');
    const applyLoanBtn = document.getElementById('applyLoanBtn');
    const topUpLoanBtn = document.getElementById('topUpLoanBtn');
    const viewDocumentsBtn = document.getElementById('viewDocumentsBtn');
    
    console.log('Button elements found:');
    console.log('Make Contribution:', makeContributionBtn);
    console.log('Apply Loan:', applyLoanBtn);
    console.log('Top Up Loan:', topUpLoanBtn);
    console.log('View Documents:', viewDocumentsBtn);
    
    // Make Contribution button
    if (makeContributionBtn) {
        console.log('Attaching event listener to Make Contribution button');
        makeContributionBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Make Contribution button clicked!');
            window.location.href = 'add-contribution.html';
        };
        console.log('Event listener attached to Make Contribution button');
    } else {
        console.log('Make Contribution button NOT found');
    }
    
    // Apply for Loan button
    if (applyLoanBtn) {
        console.log('Attaching event listener to Apply for Loan button');
        applyLoanBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Apply for Loan button clicked!');
            window.location.href = 'loan-application-modern.html';
        };
        console.log('Event listener attached to Apply for Loan button');
    } else {
        console.log('Apply for Loan button NOT found');
    }
    
    // Top Up Loan button
    if (topUpLoanBtn) {
        console.log('Attaching event listener to Top Up Loan button');
        topUpLoanBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Top Up Loan button clicked!');
            window.location.href = 'loan-application-modern.html?topup=true';
        };
        console.log('Event listener attached to Top Up Loan button');
    } else {
        console.log('Top Up Loan button NOT found');
    }
    
    // View Documents button
    if (viewDocumentsBtn) {
        console.log('Attaching event listener to View Documents button');
        viewDocumentsBtn.onclick = function(e) {
            e.preventDefault();
            console.log('View Documents button clicked!');
            window.location.href = 'documents.html';
        };
        console.log('Event listener attached to View Documents button');
    } else {
        console.log('View Documents button NOT found');
    }
    
    console.log('=== BUTTON DEBUGGING END ===');
}

// Handler functions for each button
function handleMakeContribution(e) {
    e.preventDefault();
    console.log('Make Contribution button clicked, redirecting to add-contribution.html');
    window.location.href = 'add-contribution.html';
}

function handleApplyLoan(e) {
    e.preventDefault();
    console.log('Apply for Loan button clicked, redirecting to loan-application-modern.html');
    window.location.href = 'loan-application-modern.html';
}

function handleTopUpLoan(e) {
    e.preventDefault();
    console.log('Top Up Loan button clicked, redirecting to loan-application-modern.html with topup parameter');
    window.location.href = 'loan-application-modern.html?topup=true';
}

function handleViewDocuments(e) {
    e.preventDefault();
    console.log('View Documents button clicked, redirecting to documents.html');
    window.location.href = 'documents.html';
}

// Function to load member data
async function loadMemberData(memberId) {
    try {
        console.log('Loading member data for memberId:', memberId);
        const memberData = await mftApi.getMemberData(memberId);
        console.log('Member data received:', memberData);
        
        if (memberData && !memberData.error) {
            // Store member data globally for later use
            window.currentMemberData = memberData;
            console.log('Stored member data in window.currentMemberData');
            
            // Update member info in header
            console.log('Calling updateMemberHeader');
            updateMemberHeader(memberData);
        } else {
            console.error('Failed to load member data:', memberData ? memberData.error : 'Unknown error');
        }
    } catch (error) {
        console.error('Error loading member data:', error);
    }
}

// Add a function to manually trigger member data loading for debugging
window.debugLoadMemberData = function() {
    const memberId = localStorage.getItem('memberId');
    if (memberId) {
        console.log('Manually loading member data for:', memberId);
        loadMemberData(memberId);
    } else {
        console.log('No memberId found in localStorage');
    }
};

// Cache for member dashboard data
const memberDashboardDataCache = {
    data: null,
    timestamp: 0,
    expirationTime: 5 * 60 * 1000 // 5 minutes cache expiration
};

// Function to load dashboard data
async function loadDashboardData(memberId) {
    console.log('Loading dashboard data for member ID:', memberId);
    
    // Check if we have valid cached data
    const now = Date.now();
    if (memberDashboardDataCache.data && (now - memberDashboardDataCache.timestamp) < memberDashboardDataCache.expirationTime) {
        console.log('Using cached member dashboard data');
        const cachedData = memberDashboardDataCache.data;
        
        // Update all cards with cached data
        updateContributionsCard(cachedData);
        updateLoansCard(cachedData);
        updateFinesCard(cachedData);
        updateProjectsCard(cachedData);
        updateProfileCard(cachedData.member_info || {});
        updateInboxCard([]); // Placeholder - would need actual messages API
        updateAttendanceRecordsCard({}); // Placeholder - would need actual attendance data
        updateMissedContributionsCard(cachedData.contributions || []);
        updateGuaranteedLoansCard(cachedData.loans || []);
        
        return;
    }
    
    try {
        // Load dashboard summary data
        const dashboardData = await mftApi.getMemberDashboardData(memberId);
        console.log('Dashboard data received:', dashboardData);
        if (dashboardData && !dashboardData.error) {
            // Cache the data
            memberDashboardDataCache.data = dashboardData;
            memberDashboardDataCache.timestamp = now;
            
            // Update all cards with dashboard data
            updateContributionsCard(dashboardData);
            updateLoansCard(dashboardData);
            updateFinesCard(dashboardData);
            updateProjectsCard(dashboardData);
            // Update new cards
            updateProfileCard(dashboardData.member_info || {});
            updateInboxCard([]); // Placeholder - would need actual messages API
            updateAttendanceRecordsCard({}); // Placeholder - would need actual attendance data
            updateMissedContributionsCard(dashboardData.contributions || []);
            updateGuaranteedLoansCard(dashboardData.loans || []);
        } else {
            // Fallback to individual APIs if dashboard API fails
            console.warn('Dashboard API failed, falling back to individual APIs');
            
            // Load contributions data
            const contributionsData = await mftApi.getMemberContributions(memberId);
            if (contributionsData && !contributionsData.error) {
                // Wrap the individual API response in the expected dashboard format
                updateContributionsCard({ contributions: contributionsData });
            }
            
            // Load loans data
            const loansData = await mftApi.getMemberLoans(memberId);
            if (loansData && !loansData.error) {
                // Wrap the individual API response in the expected dashboard format
                updateLoansCard({ loans: loansData });
            }
            
            // Load fines data
            const finesData = await mftApi.getMemberFines(memberId);
            if (finesData && !finesData.error) {
                // Wrap the individual API response in the expected dashboard format
                updateFinesCard({ fines: finesData });
            }
            
            // Load projects data
            const projectsData = await mftApi.getMemberProjects(memberId);
            if (projectsData && !projectsData.error) {
                updateProjectsCard(projectsData);
            }
            
            // Update new cards with individual data
            const memberData = await mftApi.getMemberData(memberId);
            if (memberData && !memberData.error) {
                updateProfileCard(memberData);
            }
            updateInboxCard([]); // Placeholder - would need actual messages API
            updateAttendanceRecordsCard({}); // Placeholder - would need actual attendance data
            updateMissedContributionsCard(contributionsData || []);
            updateGuaranteedLoansCard(loansData || []);
        }
    } catch (error) {
        console.error('Error loading dashboard data:', error);
    }
}

// Optimized version that reduces API calls by batching requests
async function loadDashboardDataOptimized(memberId) {
    console.log('Loading optimized dashboard data for member ID:', memberId);
    try {
        // Load all data in a single batch request
        const endpoint = `dashboard.php?member_id=${encodeURIComponent(memberId)}&type=member&action=getAllData`;
        const url = getApiUrl ? getApiUrl(endpoint) : `/api/${endpoint}`;
        console.log('Fetching all dashboard data from:', url);
        
        const response = await fetch(url);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            throw new Error('Response is not JSON');
        }
        
        const data = await response.json();
        console.log('All dashboard data received:', data);
        
        // Update all cards with the received data
        if (data.contributions) {
            updateContributionsCard({ contributions: data.contributions });
        }
        
        if (data.loans) {
            updateLoansCard({ loans: data.loans });
        }
        
        if (data.fines) {
            updateFinesCard({ fines: data.fines });
        }
        
        if (data.projects) {
            updateProjectsCard(data.projects);
        }
        
        if (data.member_info) {
            updateProfileCard(data.member_info);
        }
        
        // Update placeholder cards
        updateInboxCard([]); // Placeholder - would need actual messages API
        updateAttendanceRecordsCard({}); // Placeholder - would need actual attendance data
        
        if (data.contributions) {
            updateMissedContributionsCard(data.contributions);
        }
        
        if (data.loans) {
            updateGuaranteedLoansCard(data.loans);
        }
        
    } catch (error) {
        console.error('Error loading optimized dashboard data:', error);
        // Fallback to original method
        console.log('Falling back to original data loading method');
        await loadDashboardData(memberId);
    }
}

// Function to update member header information
function updateMemberHeader(memberData) {
    console.log('=== UPDATING MEMBER HEADER ===');
    console.log('Member data received:', memberData);
    
    // Retry mechanism to ensure elements are available
    let retries = 0;
    const maxRetries = 10;
    const retryInterval = 500; // 500ms
    
    function tryUpdateElements() {
        retries++;
        console.log(`Attempt ${retries} to update elements`);
        
        // Update greeting prefix with time-based greeting
        const greetingPrefixElement = document.getElementById('greetingPrefix');
        console.log('Greeting prefix element:', greetingPrefixElement);
        
        // Update member name
        const memberNameElement = document.getElementById('memberName');
        console.log('Member name element:', memberNameElement);
        
        // Check if both elements are available
        if (greetingPrefixElement && memberNameElement && memberData.first_name) {
            console.log('Both elements found, updating content');
            
            // Update greeting prefix with time-based greeting
            const greeting = getGreetingBasedOnTime();
            console.log('Time-based greeting:', greeting);
            greetingPrefixElement.textContent = greeting + ',';
            console.log('Updated greeting prefix to:', greetingPrefixElement.textContent);
            
            // Use full name (first name and last name)
            const fullName = memberData.first_name + (memberData.last_name ? ' ' + memberData.last_name : '');
            console.log('Full name:', fullName);
            
            // Log the current text content before updating
            console.log('Current member name text:', memberNameElement.textContent);
            
            memberNameElement.textContent = fullName;
            console.log('Updated member name to:', memberNameElement.textContent);
            
            // Verify the update was successful
            console.log('Verification - member name is now:', memberNameElement.textContent);
            
            // Update other elements
            updateOtherElements(memberData);
            
            return true; // Success
        } else {
            console.log('Elements not ready or data missing');
            if (!greetingPrefixElement) console.log('Greeting prefix element not found');
            if (!memberNameElement) console.log('Member name element not found');
            if (!memberData.first_name) console.log('First name not available in member data');
            
            if (retries < maxRetries) {
                console.log(`Retrying in ${retryInterval}ms...`);
                setTimeout(tryUpdateElements, retryInterval);
            } else {
                console.error('Failed to update elements after maximum retries');
            }
            return false; // Not successful yet
        }
    }
    
    // Start the update process
    tryUpdateElements();
}

// Function to update other member info elements
function updateOtherElements(memberData) {
    console.log('Updating other elements with data:', memberData);
    
    // Update member ID
    const memberIdElement = document.getElementById('memberIdValue');
    if (memberIdElement && memberData.member_id) {
        memberIdElement.textContent = memberData.member_id;
    }
    
    // Update registration date
    const registrationDateElement = document.getElementById('registrationDateValue');
    if (registrationDateElement && memberData.registration_date) {
        // Format the date nicely
        const registrationDate = new Date(memberData.registration_date);
        registrationDateElement.textContent = registrationDate.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
    }
    
    // Update status
    const statusElement = document.getElementById('memberStatusValue');
    if (statusElement && memberData.status) {
        statusElement.textContent = memberData.status;
        
        // Add color based on status
        statusElement.className = 'info-card-value status-active-card';
        if (memberData.status.toLowerCase() === 'active') {
            statusElement.style.color = '#4CAF50'; // Green
        } else if (memberData.status.toLowerCase() === 'pending') {
            statusElement.style.color = '#FFC107'; // Yellow
        } else if (memberData.status.toLowerCase() === 'suspended') {
            statusElement.style.color = '#F44336'; // Red
        } else {
            statusElement.style.color = '#2196F3'; // Blue for other statuses
        }
    }
    
    // Update last login
    const lastLoginElement = document.getElementById('lastLoginValue');
    if (lastLoginElement && memberData.last_login) {
        // Format the date nicely
        const lastLogin = new Date(memberData.last_login);
        lastLoginElement.textContent = lastLogin.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    } else if (lastLoginElement) {
        lastLoginElement.textContent = 'Never';
    }
    
    // Note: IP address is no longer displayed on the dashboard per requirements
    // but is still logged in the database for security purposes
    
    // Load and display member photo
    loadMemberPhoto(memberData.member_id);
}

// Add a function to manually trigger header update for debugging
window.debugUpdateMemberHeader = function() {
    if (window.currentMemberData) {
        console.log('Manually updating member header with:', window.currentMemberData);
        updateMemberHeader(window.currentMemberData);
    } else {
        console.log('No member data available in window.currentMemberData');
    }
};

// Function to load and display member photo
async function loadMemberPhoto(memberId) {
    try {
        // Fetch member photo information
        const response = await fetch(`api/member-photo.php?member_id=${memberId}`);
        const data = await response.json();
        
        const memberPhoto = document.getElementById('memberPhoto');
        const noPhotoMessage = document.getElementById('noPhotoMessage');
        
        if (data.success && data.photo_url) {
            // Photo exists, display it
            memberPhoto.src = data.photo_url;
            memberPhoto.style.display = 'block';
            noPhotoMessage.style.display = 'none';
        } else {
            // No photo available
            memberPhoto.style.display = 'none';
            noPhotoMessage.style.display = 'block';
        }
    } catch (error) {
        console.error('Error loading member photo:', error);
        // Show no photo message on error
        const memberPhoto = document.getElementById('memberPhoto');
        const noPhotoMessage = document.getElementById('noPhotoMessage');
        memberPhoto.style.display = 'none';
        noPhotoMessage.style.display = 'block';
    }
}

// Function to get appropriate greeting based on time of day
function getGreetingBasedOnTime() {
    const hours = new Date().getHours();
    
    if (hours >= 5 && hours < 12) {
        return 'Good morning';
    } else if (hours >= 12 && hours < 18) {
        return 'Good afternoon';
    } else {
        return 'Good evening';
    }
}

// Function to update contributions card
function updateContributionsCard(data) {
    console.log('Updating contributions card with data:', data);
    
    // Get total balance from the correct field
    let totalBalance = 0;
    if (data.contributions && typeof data.contributions === 'object') {
        // Check if it's the dashboard API response format
        if (data.contributions.total_balance !== undefined) {
            totalBalance = parseFloat(data.contributions.total_balance) || 0;
        } 
        // Check if it's the contributions API response format
        else if (data.total_balance !== undefined) {
            totalBalance = parseFloat(data.total_balance) || 0;
        }
        // Calculate from contributions array if needed
        else if (Array.isArray(data.contributions)) {
            totalBalance = data.contributions.reduce((sum, contribution) => {
                return sum + (contribution.status === 'Confirmed' ? parseFloat(contribution.amount) || 0 : 0);
            }, 0);
        }
    } else if (data.total_balance !== undefined) {
        totalBalance = parseFloat(data.total_balance) || 0;
    }
    
    console.log('Calculated total balance:', totalBalance);
    
    // Update contributions card
    const contributionCards = document.querySelectorAll('.dashboard-card.contributions');
    contributionCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(totalBalance).toLocaleString()}`;
        }
    });
}

// Function to update loans card
function updateLoansCard(data) {
    console.log('Updating loans card with data:', data);
    
    // Get outstanding balance from the correct field
    let outstandingBalance = 0;
    if (data.loans && typeof data.loans === 'object') {
        // Check if it's the dashboard API response format
        if (data.loans.outstanding_balance !== undefined) {
            outstandingBalance = parseFloat(data.loans.outstanding_balance) || 0;
        } 
        // Check if it's the loans API response format
        else if (data.outstanding_balance !== undefined) {
            outstandingBalance = parseFloat(data.outstanding_balance) || 0;
        }
        // Calculate from loans array if needed
        else if (Array.isArray(data.loans)) {
            outstandingBalance = data.loans.reduce((sum, loan) => {
                return sum + (loan.status === 'Active' ? parseFloat(loan.principal_amount) || 0 : 0);
            }, 0);
        }
    } else if (data.outstanding_balance !== undefined) {
        outstandingBalance = parseFloat(data.outstanding_balance) || 0;
    }
    
    console.log('Calculated outstanding balance:', outstandingBalance);
    
    // Update loans card
    const loanCards = document.querySelectorAll('.dashboard-card.loans');
    loanCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(outstandingBalance).toLocaleString()}`;
        }
    });
}

// Function to update fines card
function updateFinesCard(data) {
    console.log('Updating fines card with data:', data);
    
    // Calculate total fines
    let totalFines = 0;
    
    // Check if we have fines data as an object with total_fines field (from dashboard API)
    if (data.fines && typeof data.fines === 'object' && data.fines.total_fines !== undefined) {
        totalFines = parseFloat(data.fines.total_fines) || 0;
    } 
    // Check if we have fines data as an object with total_fines field (from fines API response wrapped in object)
    else if (data.total_fines !== undefined) {
        totalFines = parseFloat(data.total_fines) || 0;
    }
    // Check if we have fines data as an array (from fines API - {fines: {count: 0, fines: []}})
    else if (data.fines && typeof data.fines === 'object' && Array.isArray(data.fines.fines)) {
        totalFines = data.fines.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    }
    // Check if we have fines data as an array directly (from fines API)
    else if (data.fines && Array.isArray(data.fines)) {
        totalFines = data.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    } 
    // Check if we have total_fines directly (fallback)
    else if (data.total_fines !== undefined) {
        totalFines = parseFloat(data.total_fines) || 0;
    }
    
    console.log('Fines cards found:', document.querySelectorAll('.dashboard-card.fines').length, 'Total fines:', totalFines);
    
    // Update fines card
    const finesCards = document.querySelectorAll('.dashboard-card.fines');
    finesCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(totalFines).toLocaleString()}`;
        }
    });
}

// Function to update projects card
function updateProjectsCard(data) {
    // Get projects data from the correct location
    let projects = [];
    if (data.projects && Array.isArray(data.projects)) {
        // Direct array from individual API
        projects = data.projects;
    } else if (data.projects && typeof data.projects === 'object' && Array.isArray(data.projects.projects)) {
        // From dashboard API
        projects = data.projects.projects;
    } else if (data.projects && typeof data.projects === 'object' && !Array.isArray(data.projects)) {
        // If data.projects is an object with project data (new dashboard API format)
        projects = Array.isArray(data.projects) ? data.projects : [];
    }
    
    const projectCount = projects.length;
    
    // Update projects card
    const projectsCards = document.querySelectorAll('.dashboard-card.projects');
    projectsCards.forEach(card => {
        // Update card value (count of projects)
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = projectCount;
        }
        
        // Update card detail
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            detailElement.textContent = projectCount === 1 ? 'Project Assigned' : 'Projects Assigned';
        }
    });
    
    // Also update the specific "My Projects" card
    const myProjectsCountElement = document.getElementById('myProjectsCount');
    if (myProjectsCountElement) {
        myProjectsCountElement.textContent = projectCount;
    }
}

// Function to update profile card
function updateProfileCard(profileData) {
    // Update profile card
    const profileCards = document.querySelectorAll('.dashboard-card.profile');
    profileCards.forEach(card => {
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            if (profileData.verified) {
                detailElement.textContent = 'Account verified';
            } else {
                detailElement.textContent = 'Account not verified';
            }
        }
        
        // Update profile status value
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = profileData.status || 'Active';
        }
        
        // Update the specific profile status element
        const profileStatusElement = document.getElementById('profileStatus');
        if (profileStatusElement) {
            profileStatusElement.textContent = profileData.status || 'Active';
        }
    });
}

// Function to update inbox card
function updateInboxCard(messages) {
    const unreadCount = messages && Array.isArray(messages) ? 
        messages.filter(msg => !msg.read).length : 0;
    
    // Update inbox card
    const inboxCards = document.querySelectorAll('.dashboard-card.inbox');
    inboxCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = unreadCount;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${unreadCount} new`;
            if (unreadCount > 0) {
                statusElement.className = 'card-status status-warning';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific unread messages element
    const unreadMessagesElement = document.getElementById('unreadMessages');
    if (unreadMessagesElement) {
        unreadMessagesElement.textContent = unreadCount;
    }
}

// Function to update attendance records card (replaces dividends)
function updateAttendanceRecordsCard(attendanceData) {
    // Calculate attendance rate (simplified logic)
    let attendanceRate = '83.3%';
    let totalMeetings = 24;
    let attendedMeetings = 20;
    
    if (attendanceData && typeof attendanceData === 'object') {
        // In a real implementation, this would calculate from actual data
        // For now, we're using placeholder values
    }
    
    // Update attendance records card
    const attendanceCards = document.querySelectorAll('.dashboard-card.dividends');
    attendanceCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = attendanceRate;
        }
        
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            detailElement.textContent = 'Meeting Attendance';
        }
    });
    
    // Also update the specific attendance rate element
    const attendanceRateElement = document.getElementById('attendanceRate');
    if (attendanceRateElement) {
        attendanceRateElement.textContent = attendanceRate;
    }
}

// Function to update missed contributions card
function updateMissedContributionsCard(contributions) {
    // Calculate missed contributions (simplified logic)
    let missedAmount = 0;
    let monthsMissed = 0;
    
    if (contributions && Array.isArray(contributions)) {
        // Filter for missed/unpaid contributions
        const missedContributions = contributions.filter(contribution => 
            contribution.status === 'Missed' || contribution.status === 'Unpaid');
        
        // Calculate total missed amount
        missedAmount = missedContributions.reduce((sum, contribution) => 
            sum + (parseFloat(contribution.amount) || 0), 0);
        
        // For months missed, we would need more detailed data
        monthsMissed = missedContributions.length;
    }
    
    // Update missed contributions card
    const missedCards = document.querySelectorAll('.dashboard-card.missed');
    missedCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(missedAmount || 0).toLocaleString()}`;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${monthsMissed || 0} months`;
            if (monthsMissed > 0) {
                statusElement.className = 'card-status status-danger';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific missed contributions element
    const missedContributionsElement = document.getElementById('missedContributions');
    if (missedContributionsElement) {
        missedContributionsElement.textContent = `KSH ${Number(missedAmount || 0).toLocaleString()}`;
    }
}

// Function to update guaranteed loans card
function updateGuaranteedLoansCard(loans) {
    // Calculate guaranteed loans amount (simplified logic)
    let guaranteedAmount = 0;
    let activeGuarantees = 0;
    
    if (loans && Array.isArray(loans)) {
        // Filter for loans where the member is a guarantor
        const guaranteedLoans = loans.filter(loan => 
            loan.guarantor_id && loan.status === 'Active');
        
        // Calculate total guaranteed amount
        guaranteedAmount = guaranteedLoans.reduce((sum, loan) => 
            sum + (parseFloat(loan.principal_amount) || 0), 0);
        
        // Count active guarantees
        activeGuarantees = guaranteedLoans.length;
    }
    
    // Update guaranteed loans card
    const guaranteedCards = document.querySelectorAll('.dashboard-card.guaranteed');
    guaranteedCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(guaranteedAmount || 0).toLocaleString()}`;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${activeGuarantees} active`;
            if (activeGuarantees > 0) {
                statusElement.className = 'card-status status-warning';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific guaranteed loans element
    const guaranteedLoansElement = document.getElementById('guaranteedLoans');
    if (guaranteedLoansElement) {
        guaranteedLoansElement.textContent = `KSH ${Number(guaranteedAmount || 0).toLocaleString()}`;
    }
}

// Helper function to format project dates
function formatProjectDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const month = date.toLocaleDateString('en-US', { month: 'short' });
    const year = date.getFullYear();
    return `${month} ${year}`;
}