// MFT GROUP Contributions Real-Time Updates
// This script handles real-time updates for the contributions page

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Get member ID from localStorage or other source
    const memberId = getMemberId();
    
    console.log('Contributions page loaded. Member ID:', memberId);
    
    if (memberId) {
        // Initialize real-time client
        mftRealTime.connect('member', memberId);
        
        // Register update callback
        mftRealTime.onUpdate(function(data, pageType) {
            if (pageType === 'member') {
                updateContributionsRealTime(data);
            }
        });
        
        // Load initial data
        loadContributionsData(memberId);
        
        console.log('Real-time updates started for contributions page');
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
        showErrorState();
    }
});

// Function to get member ID (this would be implemented based on your auth system)
function getMemberId() {
    // For now, we'll try to get it from localStorage
    // In a real implementation, this would come from your authentication system
    const memberId = localStorage.getItem('memberId') || null; // Return null instead of defaulting to '1'
    console.log('Retrieved member ID from localStorage:', memberId);
    return memberId;
}

// Function to load initial contributions data
async function loadContributionsData(memberId) {
    try {
        console.log('Loading contributions data for member:', memberId);
        
        // Show loading state
        showLoadingState();
        
        // Fetch contributions data from API
        const contributionsData = await mftApi.getMemberContributions(memberId);
        
        console.log('Received contributions data:', contributionsData);
        
        if (contributionsData) {
            // Check if we have an error
            if (contributionsData.error) {
                console.error('API Error:', contributionsData.error);
                showErrorState();
                return;
            }
            
            // Update the page with the fetched data
            updateContributionsPage(contributionsData);
        } else {
            // Show error state
            showErrorState();
        }
    } catch (error) {
        console.error('Error loading contributions data:', error);
        showErrorState();
    }
}

// Function to show loading state
function showLoadingState() {
    // Update summary cards with loading state
    const summaryCards = document.querySelectorAll('.summary-card .summary-value, .summary-card .summary-detail');
    summaryCards.forEach(card => {
        if (card.classList.contains('summary-value')) {
            card.textContent = '-';
        } else {
            card.textContent = 'Loading...';
        }
    });
    
    // Update records table with loading state
    const recordsTable = document.getElementById('contribution-records');
    if (recordsTable) {
        recordsTable.innerHTML = '<tr><td colspan="5" style="text-align: center;">Loading contribution records...</td></tr>';
    }
}

// Function to show error state
function showErrorState() {
    // Update summary cards with error state
    const summaryCards = document.querySelectorAll('.summary-card .summary-value, .summary-card .summary-detail');
    summaryCards.forEach(card => {
        if (card.classList.contains('summary-value')) {
            card.textContent = 'Error';
        } else {
            card.textContent = 'Failed to load data';
        }
    });
    
    // Update records table with error state
    const recordsTable = document.getElementById('contribution-records');
    if (recordsTable) {
        recordsTable.innerHTML = '<tr><td colspan="5" style="text-align: center; color: #F44336;">Error loading contribution records</td></tr>';
    }
}

// Function to update the contributions page with real-time data
function updateContributionsRealTime(data) {
    console.log('Updating contributions with real-time data:', data);
    
    try {
        // Handle the new data structure from websocket-server.php
        if (data && data.status === 'updated' && data.data) {
            // This is a full update from the long-polling mechanism
            // Extract contributions data from the member dashboard data
            if (data.data.contributions) {
                // Create a properly formatted data object for updateContributionsPage
                const contributionsData = {
                    total_balance: data.data.contributions.total_balance || 0
                };
                updateContributionsPage(contributionsData);
            }
        } else if (data && data.member_info !== undefined) {
            // This is the member dashboard data structure from real-time updates
            // We need to extract contributions data from it
            if (data.contributions) {
                const contributionsData = {
                    total_balance: data.contributions.total_balance || 0
                };
                updateContributionsPage(contributionsData);
            }
        } else if (data && data.contributions !== undefined) {
            // This is the member dashboard data structure where contributions data is directly available
            // Extract contributions data from it
            const contributionsData = {
                total_balance: data.contributions.total_balance || 0
            };
            updateContributionsPage(contributionsData);
        } else if (data && (data.total_balance !== undefined || data.current_balance !== undefined)) {
            // This is the simplified format with just total_balance
            updateContributionsPage(data);
        } else if (data) {
            // Handle direct data structure
            updateContributionsPage(data);
        }
        // If data is null/undefined, we don't update anything to prevent errors
    } catch (error) {
        console.error('Error processing real-time update:', error);
        // Don't show error state for real-time updates, just log the error
        // The existing data should remain displayed
    }
}

// Function to update the contributions page with data
function updateContributionsPage(data) {
    console.log('Updating contributions page with data:', data);
    
    // Handle different response formats
    let summaryData, recordsData;
    
    try {
        // Check if this is the new format with summary and records
        if (data && data.summary && data.records) {
            console.log('Matched format: new format with summary and records');
            summaryData = data.summary;
            recordsData = data.records;
        } 
        // Check if this is the old format with contributions array directly
        else if (data && data.contributions && Array.isArray(data.contributions)) {
            console.log('Matched format: old format with contributions array directly');
            // Calculate summary from contributions
            summaryData = calculateSummaryFromContributions(data.contributions);
            // Use the total_balance from the API response if available
            if (data.total_balance !== undefined) {
                summaryData.current_balance = data.total_balance;
            }
            recordsData = data.contributions;
        }
        // Check if data itself is the contributions array
        else if (data && Array.isArray(data)) {
            console.log('Matched format: data itself is contributions array');
            summaryData = calculateSummaryFromContributions(data);
            recordsData = data;
        }
        // Handle the real-time update format from websocket-server.php
        else if (data && data.contributions && data.contributions.contributions !== undefined) {
            console.log('Matched format: real-time update format from websocket-server.php');
            // This is the format from the initial API call
            summaryData = {
                current_balance: data.total_balance || data.contributions.total_balance || 0,
                total_contributions: data.total_balance || data.contributions.total_balance || 0,
                balance_date: new Date().toISOString().split('T')[0]
            };
            recordsData = data.contributions.contributions || [];
        }
        // Handle the simplified format with just total_balance
        else if (data && (data.total_balance !== undefined || data.current_balance !== undefined)) {
            console.log('Matched format: simplified format with just total_balance');
            // This is the format from real-time updates
            summaryData = {
                current_balance: parseFloat(data.total_balance) || parseFloat(data.current_balance) || 0,
                total_contributions: parseFloat(data.total_balance) || parseFloat(data.current_balance) || 0,
                balance_date: new Date().toISOString().split('T')[0]
            };
            // We don't have records data in this format, so we'll keep the existing records
            recordsData = null; // Don't update records
        }
        // Handle error case
        else {
            console.error('Unexpected data format:', data);
            if (data && typeof data === 'object') {
                console.log('Data keys:', Object.keys(data));
            }
            // Don't show error state here, just return to keep existing data
            return;
        }
        
        // Update summary information
        if (summaryData) {
            updateSummaryCards(summaryData);
        }
        
        // Update contribution records only if we have new records data
        if (recordsData && Array.isArray(recordsData)) {
            updateContributionRecords(recordsData);
        }
    } catch (error) {
        console.error('Error updating contributions page:', error);
        // Don't show error state here, just log the error to keep existing data
    }
}

// Function to calculate summary data from contributions array
function calculateSummaryFromContributions(contributions) {
    if (!contributions || !Array.isArray(contributions)) return {};
    
    let totalContributions = 0;
    let currentBalance = 0;
    
    contributions.forEach(contribution => {
        const amount = parseFloat(contribution.amount) || 0;
        if (contribution.status && contribution.status.toLowerCase() === 'confirmed') {
            totalContributions += amount;
            currentBalance += amount;
        }
    });
    
    return {
        total_contributions: totalContributions,
        current_balance: currentBalance,
        balance_date: new Date().toISOString().split('T')[0]
    };
}

// Function to update summary cards
function updateSummaryCards(summary) {
    console.log('Updating summary cards with:', summary);
    
    try {
        // Update current balance
        const balanceElement = document.querySelector('.summary-card:nth-child(1) .summary-value');
        if (balanceElement) {
            const balance = summary.current_balance || summary.currentBalance || 0;
            balanceElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
        }
        
        // Update total contributions
        const totalElement = document.querySelector('.summary-card:nth-child(2) .summary-value');
        if (totalElement) {
            const total = summary.total_contributions || summary.totalContributions || 0;
            totalElement.textContent = `KSH ${Number(total).toLocaleString()}`;
        }
        
        // Update next due date (automatically calculated to 1st of next month)
        const dueDateElement = document.querySelector('.summary-card:nth-child(3) .summary-value');
        if (dueDateElement) {
            dueDateElement.textContent = calculateNextDueDate();
        }
        
        // Update detail information
        const balanceDetail = document.querySelector('.summary-card:nth-child(1) .summary-detail');
        if (balanceDetail) {
            const date = summary.balance_date || summary.balanceDate || 'Today';
            balanceDetail.textContent = `As of ${date}`;
        }
        
        const totalDetail = document.querySelector('.summary-card:nth-child(2) .summary-detail');
        if (totalDetail) {
            totalDetail.textContent = 'Total confirmed contributions';
        }
        
        const dueDateDetail = document.querySelector('.summary-card:nth-child(3) .summary-detail');
        if (dueDateDetail) {
            dueDateDetail.textContent = 'Monthly contribution due on 1st';
        }
    } catch (error) {
        console.error('Error updating summary cards:', error);
        // Don't show error state here, just log the error
    }
}

// Function to calculate the next due date (1st of next month)
function calculateNextDueDate() {
    const today = new Date();
    const nextMonth = new Date(today.getFullYear(), today.getMonth() + 1, 1);
    return nextMonth.toLocaleDateString('en-US', { 
        month: 'long', 
        day: 'numeric', 
        year: 'numeric' 
    });
}

// Function to update contribution records
function updateContributionRecords(records) {
    console.log('Updating contribution records with:', records);
    
    try {
        const recordsTable = document.getElementById('contribution-records');
        if (!recordsTable) {
            console.error('Contribution records table not found');
            return;
        }
        
        if (!records || records.length === 0) {
            recordsTable.innerHTML = '<tr><td colspan="5" style="text-align: center;">No contribution records found</td></tr>';
            return;
        }
        
        // Clear existing records
        recordsTable.innerHTML = '';
        
        // Add new records
        records.forEach(record => {
            const row = document.createElement('tr');
            const amount = parseFloat(record.amount) || 0;
            const balance = record.balance ? parseFloat(record.balance) : 0;
            
            row.innerHTML = `
                <td>${record.contribution_date || record.date || '-'}</td>
                <td>${record.description || 'Monthly contribution'}</td>
                <td>${record.id || '-'}</td>
                <td class="${amount > 0 ? 'positive' : 'negative'}">KSH ${Math.abs(amount).toLocaleString()}</td>
                <td>KSH ${balance.toLocaleString()}</td>
            `;
            recordsTable.appendChild(row);
        });
    } catch (error) {
        console.error('Error updating contribution records:', error);
        // Don't show error state here, just log the error
    }
}

// Function to manually refresh the contributions page
function refreshContributions() {
    const memberId = getMemberId();
    if (memberId) {
        loadContributionsData(memberId);
    }
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshContributions();
    }
});