// MFT GROUP Admin Settings JavaScript
// This script handles the frontend interaction with the settings API

class MFTAdminSettings {
    constructor() {
        this.apiBaseURL = getApiBaseUrl();
        this.settingsCache = {};
        this.isSaving = false;
        this.isInitialized = false;
        this.bankAccounts = [];
    }
    
    // Initialize the settings panel
    async init() {
        console.log('Initializing Admin Settings...');
        
        try {
            // Load all settings
            await this.loadAllSettings();
        } catch (error) {
            console.error('Error during settings initialization:', error);
            // Continue with initialization even if settings fail to load
        }
        
        // Set up event listeners
        this.setupEventListeners();
        
        // Set up real-time updates
        this.setupRealTimeUpdates();
        
        // Load bank accounts
        this.loadBankAccounts();
        
        this.isInitialized = true;
        console.log('Admin Settings initialized');
    }
    
    // Load all settings from the API
    async loadAllSettings() {
        try {
            console.log('Loading settings from API...');
            const response = await fetch(`${this.apiBaseURL}/settings.php`);
            console.log('Settings API response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            console.log('Settings API response:', result);
            
            if (result.hasOwnProperty('settings') && Array.isArray(result.settings)) {
                try {
                    // Convert array to object for easier lookup
                    this.settingsCache = {};
                    result.settings.forEach(setting => {
                        // Store just the value and description for easier access
                        this.settingsCache[setting.setting_key] = {
                            setting_key: setting.setting_key,
                            setting_value: setting.setting_value,
                            description: setting.description
                        };
                    });
                    console.log('Settings cache updated:', this.settingsCache);
                    
                    // Wrap populateSettingsForm in try/catch to handle any errors
                    try {
                        this.populateSettingsForm();
                        console.log('Settings loaded and form populated successfully');
                    } catch (populateError) {
                        console.error('Error populating settings form:', populateError);
                        console.error('Error details:', {
                            message: populateError.message,
                            stack: populateError.stack
                        });
                        // Don't throw the error, just log it
                        this.showNotification('Failed to populate settings form: ' + (populateError.message || 'Unknown error'), 'error');
                    }
                } catch (processingError) {
                    console.error('Error processing settings data:', processingError);
                    console.error('Error details:', {
                        message: processingError.message,
                        stack: processingError.stack
                    });
                    // Don't throw the error, just log it
                    this.showNotification('Failed to process settings data: ' + (processingError.message || 'Unknown error'), 'error');
                }
            } else {
                console.warn('No settings found or invalid response format');
                // Initialize empty cache and populate form
                this.settingsCache = {};
                
                // Wrap populateSettingsForm in try/catch to handle any errors
                try {
                    this.populateSettingsForm();
                    console.log('Settings form initialized with empty values');
                } catch (populateError) {
                    console.error('Error populating settings form with empty values:', populateError);
                    console.error('Error details:', {
                        message: populateError.message,
                        stack: populateError.stack
                    });
                    // Don't throw the error, just log it
                    this.showNotification('Failed to populate settings form with empty values: ' + (populateError.message || 'Unknown error'), 'error');
                }
            }
        } catch (error) {
            console.error('Error loading settings:', error);
            console.error('Error details:', {
                message: error.message,
                stack: error.stack
            });
            
            // Show user-friendly error message with more details
            const errorMessage = error.message || 'Unknown error occurred while loading settings';
            this.showNotification('Error loading settings: ' + errorMessage, 'error');
            // Don't re-throw the error to prevent the generic error message
        }
    }
    
    // Populate the settings form with loaded values
    populateSettingsForm() {
        console.log('Populating settings form with cache:', this.settingsCache);
        
        // Check if settingsCache is valid
        if (!this.settingsCache || typeof this.settingsCache !== 'object') {
            console.warn('Settings cache is invalid or empty');
            return;
        }
        
        // Iterate through all settings in the cache
        for (const [key, setting] of Object.entries(this.settingsCache)) {
            try {
                // Validate setting object
                if (!setting || typeof setting !== 'object') {
                    console.warn('Invalid setting object for key:', key, setting);
                    continue;
                }
                
                // Find the corresponding input element
                const inputElement = document.getElementById(key);
                if (inputElement) {
                    console.log('Populating input:', key, 'with value:', setting.setting_value);
                    
                    // Skip file input elements as they cannot be programmatically set for security reasons
                    if (inputElement.type === 'file') {
                        console.log('Skipping file input:', key, '(cannot be programmatically set)');
                        continue;
                    }
                    
                    // Handle different input types
                    try {
                        if (inputElement.type === 'checkbox') {
                            // For checkboxes, compare with true-like values
                            inputElement.checked = setting.setting_value === true || 
                                                 setting.setting_value === 'true' || 
                                                 setting.setting_value === '1' ||
                                                 setting.setting_value === 1;
                            console.log('Set checkbox', key, 'to', inputElement.checked);
                        } else if (inputElement.type === 'select-one') {
                            // For select elements, set the value
                            inputElement.value = setting.setting_value;
                            console.log('Set select', key, 'to', inputElement.value);
                        } else if (inputElement.type === 'color') {
                            // For color inputs, ensure proper format
                            inputElement.value = setting.setting_value || '#000000';
                            console.log('Set color', key, 'to', inputElement.value);
                        } else if (inputElement.type === 'date') {
                            // For date inputs, set the value directly
                            inputElement.value = setting.setting_value || '';
                            console.log('Set date', key, 'to', inputElement.value);
                        } else {
                            // For text, number, and other inputs
                            inputElement.value = setting.setting_value || '';
                            console.log('Set input', key, 'to', inputElement.value);
                        }
                    } catch (inputError) {
                        console.error('Error setting value for input:', key, inputError);
                        console.error('Input element:', inputElement);
                        console.error('Setting value:', setting.setting_value);
                        // Continue with other settings even if one fails
                    }
                } else {
                    // Also handle textarea elements
                    const textareaElement = document.querySelector(`textarea#${key}`);
                    if (textareaElement) {
                        try {
                            textareaElement.value = setting.setting_value || '';
                            console.log('Set textarea', key, 'to', textareaElement.value);
                        } catch (textareaError) {
                            console.error('Error setting value for textarea:', key, textareaError);
                            console.error('Textarea element:', textareaElement);
                            console.error('Setting value:', setting.setting_value);
                            // Continue with other settings even if one fails
                        }
                    } else {
                        console.log('No input or textarea found for setting:', key);
                    }
                }
            } catch (error) {
                console.error('Error processing setting:', key, error);
                console.error('Setting:', setting);
                // Continue with other settings even if one fails
            }
        }
        
        console.log('Settings form population completed');
    }
    
    // Set up event listeners for form elements
    setupEventListeners() {
        // Set up save buttons with section data
        const saveButtons = document.querySelectorAll('.btn-primary[data-section]');
        saveButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const section = button.getAttribute('data-section');
                this.handleSaveSettingsBySection(section);
            });
        });
        
        // Set up reset buttons with section data
        const resetButtons = document.querySelectorAll('.btn-secondary[data-section]');
        resetButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const section = button.getAttribute('data-section');
                this.handleResetSettingsBySection(section);
            });
        });
        
        // Set up tab navigation
        const tabs = document.querySelectorAll('.settings-tab');
        tabs.forEach(tab => {
            tab.addEventListener('click', (e) => {
                this.switchTab(e);
            });
        });
        
        // Set up real-time input updates
        const inputs = document.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            // For immediate feedback, update on change
            input.addEventListener('change', (e) => {
                this.handleInputChange(e);
            });
            
            // For continuous feedback, update on input (for text fields)
            if (input.type === 'text' || input.type === 'textarea' || input.type === 'number') {
                input.addEventListener('input', (e) => {
                    this.handleInputChange(e);
                });
            }
        });
        
        // Set up photo upload buttons
        this.setupPhotoUploadButtons();
        
        // Set up banking functionality
        const addBankAccountBtn = document.getElementById('addBankAccount');
        if (addBankAccountBtn) {
            addBankAccountBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.addNewBankAccount();
            });
        }
    }
    
    // Set up photo upload functionality
    setupPhotoUploadButtons() {
        // Get all photo upload buttons
        const uploadButtons = document.querySelectorAll('.photo-upload-btn');
        
        uploadButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const targetInputId = button.getAttribute('data-target');
                const fileInput = document.getElementById(targetInputId + 'Upload');
                
                if (fileInput) {
                    // Trigger file selection
                    fileInput.click();
                    
                    // Handle file selection
                    fileInput.addEventListener('change', (event) => {
                        const file = event.target.files[0];
                        if (file) {
                            this.handlePhotoUpload(file, targetInputId);
                        }
                    }, { once: true }); // Use once to prevent multiple event listeners
                }
            });
        });
    }
    
    // Handle photo upload
    async handlePhotoUpload(file, targetInputId) {
        // Validate file type
        if (!file.type.startsWith('image/')) {
            this.showNotification('Please select a valid image file (JPEG, PNG, GIF)', 'error');
            return;
        }
        
        // Validate file size (max 5MB)
        if (file.size > 5 * 1024 * 1024) {
            this.showNotification('File size exceeds 5MB limit', 'error');
            return;
        }
        
        try {
            // Show uploading status
            const targetInput = document.getElementById(targetInputId);
            const originalValue = targetInput.value;
            targetInput.value = 'Uploading...';
            targetInput.disabled = true;
            
            // Create FormData for file upload
            const formData = new FormData();
            formData.append('profile_photo', file);
            formData.append('setting_key', targetInputId);
            
            // Upload to server
            const response = await fetch(`${this.apiBaseURL}/upload-official-photo.php`, {
                method: 'POST',
                body: formData
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.path) {
                    // Set the uploaded photo path as the value
                    targetInput.value = result.path;
                    targetInput.disabled = false;
                    
                    // Show success notification
                    this.showNotification('Photo uploaded successfully!', 'success');
                    
                    // Trigger change event to ensure the value is saved
                    targetInput.dispatchEvent(new Event('change'));
                } else {
                    throw new Error(result.error || 'Upload failed');
                }
            } else {
                const errorResult = await response.json();
                throw new Error(errorResult.error || 'Upload failed');
            }
            
        } catch (error) {
            console.error('Error uploading photo:', error);
            this.showNotification('Error uploading photo: ' + error.message, 'error');
            
            // Reset input
            const targetInput = document.getElementById(targetInputId);
            targetInput.disabled = false;
        }
    }

    // Handle saving settings by section
    async handleSaveSettingsBySection(sectionName) {
        // Prevent multiple simultaneous saves
        if (this.isSaving) {
            this.showNotification('Please wait, saving in progress...', 'warning');
            return;
        }
        
        this.isSaving = true;
        
        try {
            // Get the save button and status element for this section
            const saveButton = document.querySelector(`.btn-primary[data-section="${sectionName}"]`);
            const statusElement = document.getElementById(`${sectionName}-status`);
            
            let originalText = '';
            if (saveButton) {
                originalText = saveButton.innerHTML;
                saveButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
                saveButton.disabled = true;
            }
            
            if (statusElement) {
                statusElement.style.display = 'block';
                statusElement.className = 'save-status loading';
                statusElement.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving settings...';
            }
            
            // Collect settings from the specified section
            const sectionElement = document.getElementById(`${sectionName}-tab`);
            if (!sectionElement) {
                throw new Error(`Section ${sectionName} not found`);
            }
            
            const settingsArray = this.collectSettingsFromSection(sectionElement);
            
            // Special handling for officials section to ensure all settings exist
            if (sectionName === 'officials') {
                console.log('Initializing officials settings before saving...');
                try {
                    const initResponse = await fetch(`${this.apiBaseURL}/init-officials-settings-api.php`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        }
                    });
                    
                    const initResult = await initResponse.json();
                    console.log('Officials settings initialization result:', initResult);
                    
                    if (initResult.status === 'success') {
                        console.log('Officials settings initialized successfully');
                    } else {
                        console.warn('Failed to initialize officials settings:', initResult.error);
                    }
                } catch (initError) {
                    console.warn('Error initializing officials settings:', initError);
                    // Continue with save operation even if initialization fails
                }
            }
            
            // Convert array to object for easier handling
            const settings = {};
            settingsArray.forEach(setting => {
                settings[setting.setting_key] = setting;
            });
            
            console.log('Collected settings:', settings);
            
            // Update each setting individually with better error handling
            const updateResults = [];
            for (const setting of Object.values(settings)) {
                try {
                    const result = await this.updateSetting(setting.setting_key, setting.setting_value, setting.description);
                    updateResults.push({
                        key: setting.setting_key,
                        success: result,
                        error: result ? null : 'Failed to update setting'
                    });
                } catch (error) {
                    updateResults.push({
                        key: setting.setting_key,
                        success: false,
                        error: error.message || 'Unknown error'
                    });
                }
            }
            
            // Check results and provide detailed feedback
            const failedUpdates = updateResults.filter(result => !result.success);
            const successfulUpdates = updateResults.filter(result => result.success);
            
            if (failedUpdates.length === 0) {
                // All updates were successful
                if (saveButton) {
                    saveButton.innerHTML = '<i class="fas fa-check"></i> Settings Saved!';
                }
                
                if (statusElement) {
                    statusElement.className = 'save-status success';
                    statusElement.innerHTML = `<i class="fas fa-check"></i> Settings saved successfully! (${successfulUpdates.length} of ${updateResults.length} settings saved)`;
                }
                
                // Notify real-time system of update
                this.notifySettingsUpdate(Object.keys(settings));
                
                // Show notification
                this.showNotification(`Settings saved successfully! (${successfulUpdates.length} of ${updateResults.length} settings saved)`, 'success');
                
                // Reset button and status after delay
                setTimeout(() => {
                    if (saveButton) {
                        saveButton.innerHTML = originalText;
                        saveButton.disabled = false;
                    }
                    
                    if (statusElement) {
                        setTimeout(() => {
                            statusElement.style.display = 'none';
                        }, 3000);
                    }
                }, 2000);
            } else {
                // Some updates failed
                const errorDetails = failedUpdates.map(f => `${f.key}: ${f.error}`).join(', ');
                const successCount = successfulUpdates.length;
                const totalCount = updateResults.length;
                
                console.error('Some settings failed to save:', failedUpdates);
                
                if (saveButton) {
                    saveButton.innerHTML = '<i class="fas fa-exclamation-circle"></i> Save Partially Failed';
                }
                
                if (statusElement) {
                    statusElement.className = 'save-status error';
                    statusElement.innerHTML = `<i class="fas fa-exclamation-circle"></i> Partially saved: ${successCount}/${totalCount} settings. Failed: ${errorDetails}`;
                }
                
                this.showNotification(`Settings partially saved: ${successCount}/${totalCount} settings. Check console for details.`, 'warning');
                
                // Reset button after delay
                setTimeout(() => {
                    if (saveButton) {
                        saveButton.innerHTML = originalText;
                        saveButton.disabled = false;
                    }
                    
                    if (statusElement) {
                        setTimeout(() => {
                            statusElement.style.display = 'none';
                        }, 5000);
                    }
                }, 3000);
            }
        } catch (error) {
            console.error('Error saving settings:', error);
            
            // Show error message
            this.showNotification('Error saving settings: ' + (error.message || 'Unknown error occurred'), 'error');
            
            // Update UI with error
            const saveButtons = document.querySelectorAll(`.btn-primary[data-section]`);
            saveButtons.forEach(button => {
                button.innerHTML = '<i class="fas fa-exclamation-circle"></i> Save Failed';
                button.disabled = false;
                
                setTimeout(() => {
                    button.innerHTML = '<i class="fas fa-save"></i> Save Settings';
                }, 3000);
            });
            
            // Update status elements
            const statusElements = document.querySelectorAll('.save-status');
            statusElements.forEach(element => {
                element.className = 'save-status error';
                element.innerHTML = '<i class="fas fa-exclamation-circle"></i> Error saving settings: ' + (error.message || 'Unknown error occurred');
                
                setTimeout(() => {
                    element.style.display = 'none';
                }, 5000);
            });
        } finally {
            this.isSaving = false;
        }
    }
    
    // Handle resetting settings by section
    handleResetSettingsBySection(sectionName) {
        // Confirm reset action
        if (confirm('Are you sure you want to reset these settings to their default values?')) {
            // Reset all inputs in the current section to their default values
            const sectionElement = document.getElementById(`${sectionName}-tab`);
            const inputs = sectionElement.querySelectorAll('input, select, textarea');
            
            inputs.forEach(input => {
                // Reset to default values based on type
                if (input.type === 'checkbox') {
                    // Check if there's a default checked state in the original HTML
                    input.checked = input.hasAttribute('checked');
                } else if (input.type === 'select-one') {
                    // Reset to first option
                    input.selectedIndex = 0;
                } else {
                    // For text, number, and other inputs, reset to placeholder or empty
                    input.value = input.placeholder || '';
                }
            });
            
            // Show reset confirmation
            const statusElement = document.getElementById(`${sectionName}-status`);
            if (statusElement) {
                statusElement.style.display = 'block';
                statusElement.className = 'save-status success';
                statusElement.innerHTML = '<i class="fas fa-undo"></i> Settings reset to default values!';
                
                setTimeout(() => {
                    statusElement.style.display = 'none';
                }, 3000);
            }
            
            this.showNotification('Settings reset to default values', 'success');
        }
    }
    
    // Collect settings from a specific section
    collectSettingsFromSection(sectionElement) {
        console.log('Collecting settings from section:', sectionElement);
        
        if (!sectionElement) {
            console.error('No section provided to collectSettingsFromSection');
            return [];
        }
        
        const settings = [];
        const inputs = sectionElement.querySelectorAll('input, select, textarea');
        
        console.log('Found inputs:', inputs.length);
        
        inputs.forEach(input => {
            // Skip buttons and other non-setting elements
            if (input.type === 'button' || input.type === 'submit' || input.type === 'reset') {
                console.log('Skipping button/input:', input.type);
                return;
            }
            
            // Use the element ID as the setting key
            if (input.id) {
                // Special handling for officials settings to ensure they're collected properly
                if (input.id.startsWith('chairman') || 
                    input.id.startsWith('assistantChairman') || 
                    input.id.startsWith('secretary') || 
                    input.id.startsWith('organisingSecretary') || 
                    input.id.startsWith('treasurer') || 
                    input.id.startsWith('loansOfficer')) {
                    
                    // For officials, we want to collect all fields
                    if (input.type === 'checkbox') {
                        settings.push({
                            setting_key: input.id,
                            setting_value: input.checked,
                            description: input.getAttribute('data-description') || input.id
                        });
                        console.log('Collected officials checkbox setting:', input.id, '=', input.checked);
                    } else {
                        settings.push({
                            setting_key: input.id,
                            setting_value: input.value,
                            description: input.getAttribute('data-description') || input.id
                        });
                        console.log('Collected officials setting:', input.id, '=', input.value);
                    }
                } else {
                    // Handle regular settings
                    if (input.type === 'checkbox') {
                        settings.push({
                            setting_key: input.id,
                            setting_value: input.checked,
                            description: input.getAttribute('data-description') || input.id
                        });
                        console.log('Collected checkbox setting:', input.id, '=', input.checked);
                    } else {
                        settings.push({
                            setting_key: input.id,
                            setting_value: input.value,
                            description: input.getAttribute('data-description') || input.id
                        });
                        console.log('Collected setting:', input.id, '=', input.value);
                    }
                }
            } else {
                console.log('Input without ID skipped:', input);
            }
        });
        
        console.log('Collected settings array:', settings);
        return settings;
    }
    
    // Handle input changes for real-time updates
    handleInputChange(event) {
        const input = event.target;
        if (input.id) {
            // Update cache
            if (input.type === 'checkbox') {
                this.settingsCache[input.id] = { 
                    setting_key: input.id, 
                    setting_value: input.checked,
                    description: input.getAttribute('data-description') || input.id
                };
            } else {
                this.settingsCache[input.id] = { 
                    setting_key: input.id, 
                    setting_value: input.value,
                    description: input.getAttribute('data-description') || input.id
                };
            }
            
            // Notify of change (for real-time preview)
            this.notifySettingChange(input.id, input.type === 'checkbox' ? input.checked : input.value);
        }
    }
    
    // Switch between tabs
    switchTab(event) {
        const tab = event.target;
        const tabName = tab.getAttribute('data-tab');
        
        // Remove active class from all tabs
        document.querySelectorAll('.settings-tab').forEach(t => {
            t.classList.remove('active');
        });
        
        // Add active class to clicked tab
        tab.classList.add('active');
        
        // Hide all tab content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.classList.remove('active');
        });
        
        // Show the selected tab content
        const tabContent = document.getElementById(`${tabName}-tab`);
        if (tabContent) {
            tabContent.classList.add('active');
        }
    }
    
    // Notify of a setting change (for real-time preview)
    notifySettingChange(key, value) {
        // Dispatch a custom event that other components can listen for
        const event = new CustomEvent('settingChanged', {
            detail: { key, value }
        });
        document.dispatchEvent(event);
    }
    
    // Notify of settings update (for real-time synchronization)
    notifySettingsUpdate(keys) {
        // Dispatch a custom event for settings updates
        const event = new CustomEvent('settingsUpdated', {
            detail: { keys }
        });
        document.dispatchEvent(event);
        
        // Also notify the real-time client if available
        if (typeof mftRealTime !== 'undefined') {
            mftRealTime.sendNotification({
                type: 'settings_update',
                settings: keys,
                timestamp: Date.now()
            }).catch(error => {
                console.error('Error sending real-time notification:', error);
            });
        }
    }
    
    // Set up real-time updates
    setupRealTimeUpdates() {
        // Listen for real-time updates if the real-time client is available
        if (typeof mftRealTime !== 'undefined') {
            mftRealTime.onUpdate((data, dashboardType) => {
                if (data.type === 'settings_update') {
                    console.log('Received settings update notification:', data);
                    // Reload settings when they're updated elsewhere
                    this.loadAllSettings();
                }
            });
        }
        
        // Listen for our own custom events
        document.addEventListener('settingsUpdated', (event) => {
            console.log('Settings updated:', event.detail.keys);
            // You could update UI elements here based on the updated settings
        });
        
        document.addEventListener('settingChanged', (event) => {
            console.log('Setting changed:', event.detail.key, event.detail.value);
            // You could update UI elements in real-time here
        });
    }
    
    // Update a specific setting value
    async updateSetting(key, value, description = null) {
        try {
            console.log('Updating setting:', { key, value, description });
            
            // First, check if the setting exists
            const checkResponse = await fetch(`${this.apiBaseURL}/settings.php?key=${encodeURIComponent(key)}`);
            
            if (checkResponse.status === 404) {
                // Setting doesn't exist, create it
                const createData = {
                    setting_key: key,
                    setting_value: value,
                    description: description || key
                };
                
                console.log('Creating new setting with data:', createData);
                
                const createResponse = await fetch(`${this.apiBaseURL}/settings.php`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    body: JSON.stringify(createData)
                });
                
                const createResult = await createResponse.json();
                console.log('Create response:', createResponse.status, createResult);
                
                if (createResult.status === 'success' || createResponse.status === 201) {
                    // Update cache
                    this.settingsCache[key] = { 
                        setting_key: key, 
                        setting_value: value,
                        description: description || key
                    };
                    return true;
                } else {
                    console.error('Failed to create setting:', createResult);
                    // If it's a "setting already exists" error, try updating instead
                    if (createResult.error && createResult.error.includes('already exists')) {
                        return await this.updateExistingSetting(key, value, description);
                    }
                    // Show specific error to user
                    this.showNotification(`Failed to create setting "${key}": ${createResult.error || 'Unknown error'}`, 'error');
                    return false;
                }
            } else if (checkResponse.ok) {
                // Setting exists, update it
                return await this.updateExistingSetting(key, value, description);
            } else {
                const errorResult = await checkResponse.json();
                console.error('Failed to check setting existence:', errorResult);
                // Try to create the setting anyway
                return await this.createSetting(key, value, description);
            }
        } catch (error) {
            console.error('Error updating setting:', error);
            this.showNotification(`Error updating setting "${key}": ${error.message || 'Network error'}`, 'error');
            // Try to create the setting as a fallback
            return await this.createSetting(key, value, description);
        }
    }
    
    // Create a new setting
    async createSetting(key, value, description = null) {
        try {
            const createData = {
                setting_key: key,
                setting_value: value,
                description: description || key
            };
            
            console.log('Attempting to create setting with data:', createData);
            
            const createResponse = await fetch(`${this.apiBaseURL}/settings.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify(createData)
            });
            
            const createResult = await createResponse.json();
            console.log('Create setting response:', createResponse.status, createResult);
            
            if (createResult.status === 'success' || createResponse.status === 201) {
                // Update cache
                this.settingsCache[key] = { 
                    setting_key: key, 
                    setting_value: value,
                    description: description || key
                };
                return true;
            } else {
                console.error('Failed to create setting:', createResult);
                this.showNotification(`Failed to create setting "${key}": ${createResult.error || 'Unknown error'}`, 'error');
                return false;
            }
        } catch (error) {
            console.error('Error creating setting:', error);
            this.showNotification(`Error creating setting "${key}": ${error.message || 'Network error'}`, 'error');
            return false;
        }
    }
    
    // Update an existing setting
    async updateExistingSetting(key, value, description = null) {
        try {
            const updateData = {
                setting_value: value,
                description: description || key
            };
            
            console.log('Updating existing setting:', key, 'with data:', updateData);
            
            const updateResponse = await fetch(`${this.apiBaseURL}/settings.php?key=${encodeURIComponent(key)}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify(updateData)
            });
            
            const updateResult = await updateResponse.json();
            console.log('Update setting response:', updateResponse.status, updateResult);
            
            if (updateResult.status === 'success' || updateResponse.status === 200) {
                // Update cache
                this.settingsCache[key] = { 
                    setting_key: key, 
                    setting_value: value,
                    description: description || key
                };
                
                // Notify of update
                this.notifySettingsUpdate([key]);
                
                return true;
            } else {
                console.error('Failed to update setting:', updateResult);
                this.showNotification(`Failed to update setting "${key}": ${updateResult.error || 'Unknown error'}`, 'error');
                return false;
            }
        } catch (error) {
            console.error('Error updating existing setting:', error);
            this.showNotification(`Error updating setting "${key}": ${error.message || 'Network error'}`, 'error');
            return false;
        }
    }

    // Show notification to user
    showNotification(message, type = 'info') {
        // Create a simple notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 5px;
            color: white;
            background-color: ${type === 'success' ? '#4CAF50' : type === 'error' ? '#F44336' : type === 'warning' ? '#FFC107' : '#2196F3'};
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
            z-index: 10000;
            font-family: 'Inter', sans-serif;
            font-size: 14px;
            max-width: 300px;
        `;
        
        // Add to document
        document.body.appendChild(notification);
        
        // Remove after delay
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 5000);
    }
    
    // Get a specific setting value
    getSetting(key, defaultValue = null) {
        if (this.settingsCache[key]) {
            return this.settingsCache[key].setting_value;
        }
        return defaultValue;
    }
    
    // Load bank accounts from settings
    loadBankAccounts() {
        // This would typically fetch from the database
        // For now, we'll initialize with sample data
        this.bankAccounts = [
            {
                id: 'primary',
                type: 'bank',
                name: 'Primary Account',
                bankName: 'Kingdom Bank',
                accountNumber: '3061803729002',
                accountName: 'MFT GROUP',
                branch: 'Kikuyu Branch',
                swiftCode: 'CIFIKENAXXX'
            },
            {
                id: 'mpesa',
                type: 'mpesa',
                name: 'M-Pesa Paybill',
                paybillNumber: '123456',
                accountNumber: 'MFT'
            }
        ];
        
        // Render the bank accounts in the UI
        this.renderBankAccounts();
    }
    
    // Render bank accounts in the UI
    renderBankAccounts() {
        const container = document.getElementById('bankAccountsContainer');
        if (!container) return;
        
        // Clear container
        container.innerHTML = '';
        
        // Add each account
        this.bankAccounts.forEach(account => {
            container.appendChild(this.createBankAccountElement(account));
        });
    }
    
    // Create a bank account element
    createBankAccountElement(account) {
        const accountCard = document.createElement('div');
        accountCard.className = 'bank-account-card';
        accountCard.style.cssText = 'background: rgba(255,92,27,0.1); border: 1px solid var(--primary); border-radius: 8px; padding: 20px; margin-bottom: 20px;';
        
        if (account.type === 'bank') {
            accountCard.innerHTML = `
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                    <h4><i class="fas fa-piggy-bank"></i> ${account.name}</h4>
                    <div>
                        <button class="btn btn-outline edit-account" style="padding: 5px 10px; margin-right: 10px;" data-id="${account.id}">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                        <button class="btn btn-outline delete-account" style="padding: 5px 10px; color: #F44336; border-color: #F44336;" data-id="${account.id}">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-col">
                        <div class="form-group">
                            <label>Bank Name</label>
                            <input type="text" class="form-control" value="${account.bankName}" data-field="bankName" data-id="${account.id}" readonly>
                        </div>
                    </div>
                    <div class="form-col">
                        <div class="form-group">
                            <label>Account Number</label>
                            <input type="text" class="form-control" value="${account.accountNumber}" data-field="accountNumber" data-id="${account.id}" readonly>
                        </div>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-col">
                        <div class="form-group">
                            <label>Account Name</label>
                            <input type="text" class="form-control" value="${account.accountName}" data-field="accountName" data-id="${account.id}" readonly>
                        </div>
                    </div>
                    <div class="form-col">
                        <div class="form-group">
                            <label>Branch</label>
                            <input type="text" class="form-control" value="${account.branch}" data-field="branch" data-id="${account.id}" readonly>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label>Swift Code</label>
                    <input type="text" class="form-control" value="${account.swiftCode}" data-field="swiftCode" data-id="${account.id}" readonly>
                </div>
            `;
        } else if (account.type === 'mpesa') {
            accountCard.innerHTML = `
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                    <h4><i class="fas fa-mobile-alt"></i> ${account.name}</h4>
                    <div>
                        <button class="btn btn-outline edit-account" style="padding: 5px 10px; margin-right: 10px;" data-id="${account.id}">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                        <button class="btn btn-outline delete-account" style="padding: 5px 10px; color: #F44336; border-color: #F44336;" data-id="${account.id}">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-col">
                        <div class="form-group">
                            <label>Paybill Number</label>
                            <input type="text" class="form-control" value="${account.paybillNumber}" data-field="paybillNumber" data-id="${account.id}" readonly>
                        </div>
                    </div>
                    <div class="form-col">
                        <div class="form-group">
                            <label>Account Number</label>
                            <input type="text" class="form-control" value="${account.accountNumber}" data-field="accountNumber" data-id="${account.id}" readonly>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Add event listeners for edit and delete buttons
        const editButtons = accountCard.querySelectorAll('.edit-account');
        editButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                this.editBankAccount(account.id);
            });
        });
        
        const deleteButtons = accountCard.querySelectorAll('.delete-account');
        deleteButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                this.deleteBankAccount(account.id);
            });
        });
        
        return accountCard;
    }
    
    // Add a new bank account
    addNewBankAccount() {
        // Create a new bank account
        const accountType = prompt('Enter account type (bank/mpesa):', 'bank');
        if (!accountType || (accountType !== 'bank' && accountType !== 'mpesa')) return;
        
        const newAccount = {
            id: 'account-' + Date.now(),
            type: accountType,
            name: accountType === 'bank' ? 'New Bank Account' : 'New M-Pesa Account'
        };
        
        if (accountType === 'bank') {
            newAccount.bankName = '';
            newAccount.accountNumber = '';
            newAccount.accountName = '';
            newAccount.branch = '';
            newAccount.swiftCode = '';
        } else {
            newAccount.paybillNumber = '';
            newAccount.accountNumber = '';
        }
        
        this.bankAccounts.push(newAccount);
        this.renderBankAccounts();
    }
    
    // Edit a bank account
    editBankAccount(accountId) {
        // Find all input fields for this account and make them editable
        const inputs = document.querySelectorAll(`input[data-id="${accountId}"]`);
        inputs.forEach(input => {
            input.removeAttribute('readonly');
            input.style.border = '1px solid var(--primary)';
        });
        
        // Change the edit button to save
        const editButton = document.querySelector(`.edit-account[data-id="${accountId}"]`);
        if (editButton) {
            editButton.innerHTML = '<i class="fas fa-save"></i> Save';
            editButton.classList.remove('edit-account');
            editButton.classList.add('save-account');
            editButton.onclick = () => {
                this.saveBankAccount(accountId);
            };
        }
    }
    
    // Save a bank account
    saveBankAccount(accountId) {
        // Find all input fields for this account and make them readonly again
        const inputs = document.querySelectorAll(`input[data-id="${accountId}"]`);
        inputs.forEach(input => {
            input.setAttribute('readonly', 'readonly');
            input.style.border = '1px solid var(--gray-dark)';
        });
        
        // Change the save button back to edit
        const saveButton = document.querySelector(`.save-account[data-id="${accountId}"]`);
        if (saveButton) {
            saveButton.innerHTML = '<i class="fas fa-edit"></i> Edit';
            saveButton.classList.remove('save-account');
            saveButton.classList.add('edit-account');
            saveButton.onclick = () => {
                this.editBankAccount(accountId);
            };
        }
        
        // In a real implementation, this would save to the database
        this.showNotification('Bank account saved successfully!', 'success');
    }
    
    // Delete a bank account
    deleteBankAccount(accountId) {
        if (confirm('Are you sure you want to delete this bank account?')) {
            this.bankAccounts = this.bankAccounts.filter(account => account.id !== accountId);
            this.renderBankAccounts();
            this.showNotification('Bank account deleted successfully!', 'success');
        }
    }
    
    // Check if settings are initialized
    isSettingsInitialized() {
        return this.isInitialized;
    }
}

// Initialize the admin settings when the DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // The initialization is now handled in the HTML file
});

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = MFTAdminSettings;
}