// MFT GROUP Admin Real-Time Service
// This script handles sending real-time notifications from the admin dashboard

class MFTAdminRealTimeService {
    constructor() {
        this.apiBaseURL = getApiBaseUrl();
        this.notificationEndpoint = '/websocket-server.php';
    }
    
    /**
     * Send a notification to all connected clients
     * @param {string} type - The type of notification
     * @param {object} data - The data to send with the notification
     * @param {string} target - The target audience ('admin', 'member_dashboard', 'all')
     * @param {number} memberId - Optional member ID for member-specific notifications
     */
    async sendNotification(type, data, target = 'all', memberId = null) {
        try {
            const notificationData = {
                type: type,
                data: data,
                target: target
            };
            
            if (memberId) {
                notificationData.member_id = memberId;
            }
            
            const response = await fetch(`${this.apiBaseURL}${this.notificationEndpoint}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(notificationData)
            });
            
            const result = await response.json();
            
            if (result.status === 'notification_received') {
                console.log('Notification sent successfully:', type);
                return true;
            } else {
                console.error('Failed to send notification:', result.error || 'Unknown error');
                return false;
            }
        } catch (error) {
            console.error('Error sending notification:', error);
            return false;
        }
    }
    
    /**
     * Notify about contribution updates
     * @param {number} memberId - The member ID
     * @param {number} contributionId - The contribution ID
     * @param {number} amount - The contribution amount
     */
    async notifyContributionUpdate(memberId, contributionId, amount) {
        return await this.sendNotification('contribution_update', {
            member_id: memberId,
            contribution_id: contributionId,
            amount: amount,
            timestamp: Date.now()
        }, 'member_dashboard', memberId);
    }
    
    /**
     * Notify about loan updates
     * @param {number} memberId - The member ID
     * @param {number} loanId - The loan ID
     * @param {number} amount - The loan amount
     */
    async notifyLoanUpdate(memberId, loanId, amount) {
        return await this.sendNotification('loan_update', {
            member_id: memberId,
            loan_id: loanId,
            amount: amount,
            timestamp: Date.now()
        }, 'member_dashboard', memberId);
    }
    
    /**
     * Notify about member updates
     * @param {number} memberId - The member ID
     * @param {string} action - The action performed (created, updated, deleted)
     */
    async notifyMemberUpdate(memberId, action) {
        return await this.sendNotification('member_update', {
            member_id: memberId,
            action: action,
            timestamp: Date.now()
        }, 'admin');
    }
    
    /**
     * Notify about project updates
     * @param {number} projectId - The project ID
     * @param {string} action - The action performed (created, updated, deleted)
     */
    async notifyProjectUpdate(projectId, action) {
        return await this.sendNotification('project_update', {
            project_id: projectId,
            action: action,
            timestamp: Date.now()
        }, 'all');
    }
    
    /**
     * Notify about settings updates
     * @param {array} settingsKeys - The keys of the updated settings
     */
    async notifySettingsUpdate(settingsKeys) {
        return await this.sendNotification('settings_update', {
            settings: settingsKeys,
            timestamp: Date.now()
        }, 'all');
    }
    
    /**
     * Notify about dashboard updates
     * @param {string} dashboardType - The type of dashboard ('admin', 'member')
     */
    async notifyDashboardUpdate(dashboardType) {
        return await this.sendNotification('dashboard_update', {
            dashboard_type: dashboardType,
            timestamp: Date.now()
        }, dashboardType);
    }
}

// Create a global instance
const mftAdminRealTime = new MFTAdminRealTimeService();

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = MFTAdminRealTimeService;
}