// MFT GROUP Admin Navigation Handler
// This script handles the sidebar navigation for the admin dashboard

console.log('=== ADMIN NAVIGATION LOADING ===');

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM Content Loaded - Initializing admin navigation');
    initializeAdminNavigation();
    // NEW: Hide navigation items based on permissions
    // DISABLED: All admins should see all modules
    // hideNavItemsBasedOnPermissions();
});

// Function to initialize admin navigation
function initializeAdminNavigation() {
    console.log('Initializing admin navigation');
    
    // Add click event listeners to all navigation links
    const navLinks = document.querySelectorAll('a[data-page]');
    console.log('Found', navLinks.length, 'navigation links');
    
    if (navLinks.length === 0) {
        console.warn('No navigation links found');
        return;
    }
    
    navLinks.forEach(link => {
        // Remove any existing listeners to prevent duplicates
        link.removeEventListener('click', handleNavigationClick);
        
        // Add click event listener
        link.addEventListener('click', handleNavigationClick);
    });
    
    console.log('Navigation listeners attached successfully');
}

// Function to handle navigation click events
function handleNavigationClick(e) {
    console.log('External JavaScript: Link clicked');
    // If the link already has a proper href (not just #), let it work normally
    const href = this.getAttribute('href');
    if (href && href !== '#' && !href.startsWith('#')) {
        console.log('External JavaScript: Letting link with href work normally:', href);
        // Allow the default behavior for links with real href values
        return;
    }
    
    console.log('External JavaScript: Preventing default and handling with JavaScript');
    // For links with # href, use the JavaScript navigation
    e.preventDefault();
    const page = this.getAttribute('data-page');
    const subtab = this.getAttribute('data-subtab');
    
    console.log('Navigation link clicked:', page, subtab);
    
    // NEW: Check permissions before navigating to restricted pages
    if (page !== 'dashboard' && !hasPagePermission(page)) {
        alert('You do not have permission to access this page. Contact your system administrator.');
        return;
    }
    
    navigateToPage(page, subtab);
}

// Function to navigate to a specific page
function navigateToPage(page, subtab) {
    console.log('Navigating to page:', page, 'subtab:', subtab);
    
    try {
        // Construct the URL based on the page
        let url;
        switch (page) {
            case 'dashboard':
                url = 'admin-dashboard.html';
                break;
            case 'members':
                if (subtab) {
                    switch (subtab) {
                        case 'add-member':
                            url = 'add_member.php';
                            break;
                        case 'import-member':
                            url = 'import_member.html'; // Placeholder
                            break;
                        case 'export-members':
                            url = 'export_members.html'; // Placeholder
                            break;
                        default:
                            url = 'member-management.html';
                    }
                } else {
                    url = 'member-management.html';
                }
                break;
            case 'contributions':
                url = 'contributions-management.html';
                break;
            case 'loans':
                url = 'loans-management.html';
                break;
            case 'fines':
                url = 'fines.html';
                break;
            case 'projects':
                url = 'admin-projects.html';
                break;
            case 'meetings':
                url = 'meetings.html';
                break;
            case 'documents':
                // Changed from 'documents.html' to 'admin_documents.html'
                url = 'admin_documents.html';
                break;
            case 'audit':
                url = 'audit-trail.html';
                break;
            case 'expenses':
                url = 'expenses.html';
                break;
            case 'organization':
                url = 'organization-details.html';
                break;
            case 'reports':
                url = 'admin-reports.html';
                break;
            case 'settings':
                url = 'admin-settings.html';
                break;
            default:
                console.warn('Unknown page:', page);
                url = 'admin-dashboard.html';
        }
        
        console.log('Redirecting to:', url);
        
        // Redirect to the page
        window.location.href = url;
        
    } catch (error) {
        console.error('Error navigating to page:', page, error);
    }
}

// NEW: Function to check if user has permission for a specific page
// DISABLED: All admins should have access to all pages
function hasPagePermission(page) {
    // All admins can access all pages
    return true;
    
    /*
    // Get user role from session storage
    try {
        const adminData = sessionStorage.getItem('adminData');
        if (adminData) {
            const admin = JSON.parse(adminData);
            const role = admin.role || 'Member';
            
            // Define role permissions for pages
            const rolePermissions = {
                'Admin': ['all'],
                'Treasurer': ['dashboard', 'members', 'contributions', 'loans', 'fines', 'projects', 'audit', 'expenses', 'reports'],
                'Secretary': ['dashboard', 'members', 'contributions', 'projects', 'meetings', 'documents', 'reports'],
                'Chairman': ['dashboard', 'members', 'projects', 'meetings', 'audit', 'reports'],
                'Assistant Chairman': ['dashboard', 'members', 'projects', 'meetings', 'audit', 'reports'],
                'Assistant Secretary': ['dashboard', 'members', 'projects', 'meetings', 'documents', 'reports'],
                'Loans Officer': ['dashboard', 'members', 'loans', 'audit', 'reports'],
                'Member': ['dashboard']
            };
            
            const permissions = rolePermissions[role] || ['dashboard'];
            
            // Admin has access to everything
            if (permissions.includes('all')) {
                return true;
            }
            
            return permissions.includes(page);
        }
    } catch (e) {
        console.error('Error checking page permissions:', e);
    }
    
    // Default to only dashboard access
    return page === 'dashboard';
    */
}

// NEW: Function to hide navigation items based on user permissions
function hideNavItemsBasedOnPermissions() {
    try {
        const navLinks = document.querySelectorAll('a[data-page]');
        
        navLinks.forEach(link => {
            const page = link.getAttribute('data-page');
            
            // Always show dashboard
            if (page === 'dashboard') {
                return;
            }
            
            // Hide links based on permissions
            if (!hasPagePermission(page)) {
                link.style.display = 'none';
            }
        });
    } catch (error) {
        console.error('Error hiding navigation items:', error);
    }
}

// Make functions available globally
window.initializeAdminNavigation = initializeAdminNavigation;
window.hasPagePermission = hasPagePermission; // Export for use in other scripts

console.log('=== ADMIN NAVIGATION LOADED ===');