// MFT GROUP Admin Dashboard Real-Time Updates
// This script handles real-time updates for the admin dashboard

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize real-time client
    mftRealTime.connect('admin');
    
    // Register update callback
    mftRealTime.onUpdate(function(data, dashboardType) {
        if (dashboardType === 'admin') {
            updateAdminDashboardRealTime(data);
        }
    });
    
    console.log('Real-time updates started for admin dashboard');
    
    // Initialize real-time features
    initializeRealTimeFeatures();
});

// Function to initialize real-time features
function initializeRealTimeFeatures() {
    // Create real-time indicator
    if (!document.getElementById('realtimeIndicator')) {
        const indicator = document.createElement('div');
        indicator.id = 'realtimeIndicator';
        indicator.className = 'realtime-indicator connecting';
        indicator.innerHTML = '<div class="pulse"></div> Connecting...';
        document.body.appendChild(indicator);
        
        // Update real-time indicator after connection
        setTimeout(() => {
            indicator.className = 'realtime-indicator';
            indicator.innerHTML = '<div class="pulse"></div> Live Updates';
            showNotification('Real-time updates connected', 'success');
        }, 2000);
    }
}

// Function to show a notification to the user
function showNotification(message, type = 'info') {
    // Check if notification already exists
    const existingNotification = document.querySelector('.notification');
    if (existingNotification) {
        existingNotification.remove();
    }
    
    // Create a notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 80px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        background-color: ${type === 'success' ? '#4CAF50' : type === 'error' ? '#F44336' : '#2196F3'};
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        z-index: 10000;
        font-family: 'Inter', sans-serif;
        transform: translateX(120%);
        transition: transform 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Trigger reflow to enable transition
    notification.offsetHeight;
    notification.style.transform = 'translateX(0)';
    
    // Remove the notification after 3 seconds
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

// Function to update the admin dashboard with real-time data
function updateAdminDashboardRealTime(data) {
    console.log('Updating admin dashboard with real-time data:', data);
    
    // Handle different types of notifications
    if (data.type === 'member_update') {
        handleMemberUpdate(data.data);
    } else if (data.type === 'contribution_update') {
        handleContributionUpdate(data.data);
    } else if (data.type === 'loan_update') {
        handleLoanUpdate(data.data);
    } else if (data.type === 'fine_update') {
        handleFineUpdate(data.data);
    } else if (data.type === 'settings_update') {
        handleSettingsUpdate(data.data);
    } else if (data.type === 'project_update') {
        handleProjectUpdate(data.data);
    } else if (data.type === 'dashboard_update') {
        handleGeneralDashboardUpdate(data.data);
    } else {
        // Handle general dashboard updates
        handleGeneralDashboardUpdate(data);
    }
}

// Handle member updates
function handleMemberUpdate(data) {
    console.log('Processing member update:', data);
    
    // Update members card
    if (data.action) {
        updateMembersCard(data.action);
    }
    
    // Show notification
    showNotification(`Member ${data.action} successfully`, 'success');
}

// Handle contribution updates
function handleContributionUpdate(data) {
    console.log('Processing contribution update:', data);
    
    // Update contributions card
    if (data.amount !== undefined) {
        updateContributionsCard(data.amount);
    }
    
    // Show notification
    showNotification(`New contribution recorded: KSH ${Number(data.amount).toLocaleString()}`, 'success');
}

// Handle loan updates
function handleLoanUpdate(data) {
    console.log('Processing loan update:', data);
    
    // Update loans card
    if (data.amount !== undefined) {
        updateLoansCard(data.amount);
    }
    
    // Show notification
    if (data.action === 'new_loan') {
        showNotification(`New loan applied for member: KSH ${Number(data.amount).toLocaleString()}`, 'success');
    } else {
        showNotification(`Loan ${data.status}: KSH ${Number(data.amount).toLocaleString()}`, 'info');
    }
    
    // Refresh loans section if on loans page
    if (typeof refreshLoansSection === 'function') {
        refreshLoansSection();
    }
}

// Handle fine updates
function handleFineUpdate(data) {
    console.log('Processing fine update:', data);
    
    // Update fines card
    if (data.amount !== undefined) {
        updateFinesCard(data.amount);
    }
    
    // Show notification
    showNotification(`Fine ${data.status}: KSH ${Number(data.amount).toLocaleString()}`, 'info');
}

// Handle settings updates
function handleSettingsUpdate(data) {
    console.log('Processing settings update:', data);
    
    // In a real implementation, you would update UI elements based on changed settings
    showNotification('System settings have been updated', 'info');
}

// Handle project updates
function handleProjectUpdate(data) {
    console.log('Processing project update:', data);
    
    // Update projects card
    if (data.action) {
        updateProjectsCard(data.action);
    }
    
    // Show notification
    showNotification(`Project ${data.action} successfully`, 'info');
}

// Handle general dashboard updates
function handleGeneralDashboardUpdate(data) {
    console.log('Processing general dashboard update:', data);
    
    // Update admin info in header
    if (data.admin_info) {
        updateAdminHeader(data.admin_info);
    }
    
    // Update contributions card
    if (data.contributions && data.contributions.total_balance !== undefined) {
        updateContributionsCard(data.contributions.total_balance);
    }
    
    // Update loans card
    if (data.loans && data.loans.outstanding_balance !== undefined) {
        updateLoansCard(data.loans.outstanding_balance);
    }
    
    // Update fines card
    if (data.fines && data.fines.total_fines !== undefined) {
        updateFinesCard(data.fines.total_fines);
    }
    
    // Update members card
    if (data.members) {
        updateMembersCard(data.members);
    }
    
    // Update projects card
    if (data.projects) {
        updateProjectsCard(data.projects);
    }
    
    // Update bank balance card
    if (data.bank_balance !== undefined) {
        updateBankBalanceCard(data.bank_balance);
    }
}

// Function to update admin header information
function updateAdminHeader(adminInfo) {
    // Update greeting
    const greetingElement = document.getElementById('greeting');
    if (greetingElement) {
        // Get appropriate greeting based on time of day
        const greeting = getGreetingBasedOnTime();
        greetingElement.textContent = `${greeting}, ${adminInfo.name || 'Admin'}`;
    }
    
    // Update admin name
    const adminNameElement = document.getElementById('adminName');
    if (adminNameElement) {
        adminNameElement.textContent = adminInfo.name || 'Admin User';
    }
}

// Function to get appropriate greeting based on time of day
function getGreetingBasedOnTime() {
    const hours = new Date().getHours();
    
    if (hours >= 5 && hours < 12) {
        return 'Good morning';
    } else if (hours >= 12 && hours < 18) {
        return 'Good afternoon';
    } else {
        return 'Good evening';
    }
}

// Function to update contributions card
function updateContributionsCard(balance) {
    if (balance !== undefined) {
        // Find all contributions cards
        const contributionCards = document.querySelectorAll('.dashboard-card.contributions');
        contributionCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update loans card
function updateLoansCard(balance) {
    if (balance !== undefined) {
        // Find all loans cards
        const loanCards = document.querySelectorAll('.dashboard-card.loans');
        loanCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update fines card
function updateFinesCard(balance) {
    if (balance !== undefined) {
        // Find all fines cards
        const finesCards = document.querySelectorAll('.dashboard-card.fines');
        finesCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to update members card
function updateMembersCard(membersData) {
    // Update the members card with real-time data
    const membersCards = document.querySelectorAll('.dashboard-card.members');
    membersCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            // If membersData is a number, use it directly
            // If it's an object with count, use that
            const count = typeof membersData === 'number' ? membersData : 
                         membersData.count !== undefined ? membersData.count : 
                         membersData.total !== undefined ? membersData.total : 
                         null;
            
            if (count !== null) {
                valueElement.textContent = count;
            }
        }
    });
}

// Function to update projects card
function updateProjectsCard(projectsData) {
    // Update the projects card with real-time data
    const projectsCards = document.querySelectorAll('.dashboard-card.projects');
    projectsCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            // If projectsData is a number, use it directly
            // If it's an object with count, use that
            const count = typeof projectsData === 'number' ? projectsData : 
                         projectsData.count !== undefined ? projectsData.count : 
                         projectsData.total !== undefined ? projectsData.total : 
                         null;
            
            if (count !== null) {
                valueElement.textContent = count;
            }
        }
    });
}

// Function to update bank balance card
function updateBankBalanceCard(balance) {
    if (balance !== undefined) {
        // Find all bank balance cards
        const bankCards = document.querySelectorAll('.dashboard-card.bank');
        bankCards.forEach(card => {
            const valueElement = card.querySelector('.card-value');
            if (valueElement) {
                valueElement.textContent = `KSH ${Number(balance).toLocaleString()}`;
            }
        });
    }
}

// Function to manually refresh the dashboard
function refreshAdminDashboard() {
    // Trigger an immediate update
    mftRealTime.pollForUpdates();
}

// Function to send a notification to a specific member
function sendMemberNotification(memberId, notificationData) {
    // Create notification object
    const notification = {
        type: 'member_notification',
        target: 'member',
        targetId: memberId,
        data: notificationData,
        timestamp: new Date().toISOString()
    };
    
    // Send notification through real-time client
    if (typeof mftRealTime !== 'undefined' && typeof mftRealTime.sendMemberNotification === 'function') {
        mftRealTime.sendMemberNotification(memberId, notification)
            .then(response => {
                console.log('Member notification sent successfully:', response);
            })
            .catch(error => {
                console.error('Error sending member notification:', error);
            });
    } else {
        // Fallback to general notification
        if (typeof mftRealTime !== 'undefined' && typeof mftRealTime.sendNotification === 'function') {
            mftRealTime.sendNotification(notification)
                .then(response => {
                    console.log('Notification sent successfully:', response);
                })
                .catch(error => {
                    console.error('Error sending notification:', error);
                });
        }
    }
}

// Function to broadcast a notification to all members
function broadcastMemberNotification(notificationData) {
    // Create notification object
    const notification = {
        type: 'broadcast_notification',
        target: 'all_members',
        data: notificationData,
        timestamp: new Date().toISOString()
    };
    
    // Send notification through real-time client
    if (typeof mftRealTime !== 'undefined' && typeof mftRealTime.sendNotification === 'function') {
        mftRealTime.sendNotification(notification)
            .then(response => {
                console.log('Broadcast notification sent successfully:', response);
            })
            .catch(error => {
                console.error('Error sending broadcast notification:', error);
            });
    }
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshAdminDashboard();
    }
});