<?php
// Script to create a test loan and approve it to verify agreement generation
require_once 'database/config.php';

echo "=== Creating and Approving Test Loan ===\n\n";

try {
    $pdo = getDatabaseConnection();
    
    // Create a test loan
    echo "Creating a test loan...\n";
    
    // Get an active member to associate with the loan
    $member_sql = "SELECT id, member_id, first_name, last_name FROM members WHERE status = 'Active' LIMIT 1";
    $member_stmt = $pdo->prepare($member_sql);
    $member_stmt->execute();
    $member = $member_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$member) {
        echo "No active members found. Cannot create test loan.\n";
        exit(1);
    }
    
    echo "Using member: " . $member['member_id'] . " (" . $member['first_name'] . " " . $member['last_name'] . ")\n";
    
    // Generate a unique reference
    $reference = "LOAN-TEST-" . date('YmdHis') . "-" . strtoupper(substr(md5(uniqid()), 0, 4));
    
    // Insert test loan
    $insert_sql = "INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) 
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $insert_stmt = $pdo->prepare($insert_sql);
    $result = $insert_stmt->execute([
        $reference,
        $member['id'],
        'Emergency',
        10000.00,  // principal_amount
        12.00,     // interest_rate
        11200.00,  // total_amount (10000 + 12% interest)
        12,        // repayment_period
        933.33,    // monthly_payment
        'Test loan for agreement generation verification',
        date('Y-m-d'),
        'Pending'
    ]);
    
    if ($result) {
        $loan_id = $pdo->lastInsertId();
        echo "Test loan created successfully!\n";
        echo "Loan ID: " . $loan_id . "\n";
        echo "Reference: " . $reference . "\n\n";
        
        // Get an admin user to approve the loan
        $admin_sql = "SELECT id, member_id FROM members WHERE member_id LIKE 'ADM-%' LIMIT 1";
        $admin_stmt = $pdo->prepare($admin_sql);
        $admin_stmt->execute();
        $admin = $admin_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin) {
            echo "No admin user found. Cannot approve loan.\n";
            exit(1);
        }
        
        echo "Approving loan with admin: " . $admin['member_id'] . "\n";
        
        // Update the loan status to Approved
        $update_sql = "UPDATE loans SET status = ?, approval_date = ?, approved_by = ? WHERE id = ?";
        $update_stmt = $pdo->prepare($update_sql);
        $result = $update_stmt->execute([
            'Approved',
            date('Y-m-d'),
            $admin['id'],
            $loan_id
        ]);
        
        if ($result) {
            echo "Loan approved successfully!\n\n";
            
            // Now call the agreement generation function directly
            echo "Generating loan agreement...\n";
            
            // Define the agreement generation function
            function generateLoanAgreementForLoan($loanId) {
                global $pdo;
                
                try {
                    // Get loan details with member information
                    $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id, m.id_number, m.phone, m.address 
                            FROM loans l
                            JOIN members m ON l.member_id = m.id
                            WHERE l.id = ?";
                    
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$loanId]);
                    $loan = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$loan) {
                        error_log("Loan not found for agreement generation: " . $loanId);
                        return false;
                    }
                    
                    // Get system settings for lender information
                    $settings_sql = "SELECT setting_key, setting_value FROM system_settings";
                    $settings_stmt = $pdo->prepare($settings_sql);
                    $settings_stmt->execute();
                    $settings_rows = $settings_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $settings = [];
                    foreach ($settings_rows as $row) {
                        $settings[$row['setting_key']] = $row['setting_value'];
                    }
                    
                    // Prepare loan data for agreement generation
                    $loanData = [
                        'loanRef' => $loan['reference'],
                        'agreementDate' => date('F j, Y'),
                        'lenderName' => $settings['organizationName'] ?? 'MFT SELF HELP GROUP',
                        'lenderRegNo' => $settings['registrationNumber'] ?? 'REG-2025-001',
                        'lenderAddress' => $settings['organizationAddress'] ?? 'Nairobi, Kenya',
                        'borrowerName' => $loan['first_name'] . ' ' . $loan['last_name'],
                        'borrowerMemberNo' => $loan['member_id'],
                        'borrowerId' => $loan['id_number'],
                        'borrowerAddress' => $loan['address'],
                        'borrowerPhone' => $loan['phone'],
                        'loanType' => $loan['loan_type'],
                        'loanPurpose' => $loan['purpose'],
                        'principalAmount' => $loan['principal_amount'],
                        'interestRate' => $loan['interest_rate'],
                        'totalAmount' => $loan['total_amount'],
                        'repaymentPeriod' => $loan['repayment_period'],
                        'monthlyPayment' => $loan['monthly_payment'],
                        'applicationDate' => $loan['application_date'],
                        'approvalDate' => $loan['approval_date'] ?? date('F j, Y'),
                        'disbursementDate' => $loan['disbursement_date'] ?? date('F j, Y'),
                        'firstPaymentDate' => $loan['first_payment_date'] ?? date('F j, Y', strtotime('+1 month')),
                        'lenderRep' => $settings['treasurerFullName'] ?? 'Treasurer',
                        'verificationUrl' => 'https://mft-group.org/verify/' . $loan['reference'],
                        'schedule' => [] // In a real implementation, this would be calculated
                    ];
                    
                    // Include the loan agreement generator
                    require_once 'api/loan-agreement-generator.php';
                    
                    // Generate the loan agreement
                    $result = generateLoanAgreement($loanData, 'F');
                    
                    if ($result) {
                        error_log("Loan agreement generated successfully for loan ID: " . $loanId);
                        return $result;
                    } else {
                        error_log("Failed to generate loan agreement for loan ID: " . $loanId);
                        return false;
                    }
                } catch (Exception $e) {
                    error_log("Error generating loan agreement: " . $e->getMessage());
                    return false;
                }
            }
            
            // Generate the agreement
            $agreement_path = generateLoanAgreementForLoan($loan_id);
            
            if ($agreement_path) {
                echo "SUCCESS: Loan agreement generated!\n";
                echo "Agreement saved at: " . $agreement_path . "\n\n";
                
                // Verify the file exists
                if (file_exists($agreement_path)) {
                    $file_size = filesize($agreement_path);
                    echo "File verification:\n";
                    echo "- Path: " . $agreement_path . "\n";
                    echo "- Size: " . $file_size . " bytes\n";
                    echo "- Created: " . date("F j, Y, g:i a", filemtime($agreement_path)) . "\n";
                } else {
                    echo "WARNING: Agreement file not found at expected location.\n";
                }
            } else {
                echo "FAILED: Could not generate loan agreement.\n";
            }
        } else {
            echo "Failed to approve loan.\n";
        }
    } else {
        echo "Failed to create test loan.\n";
    }
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
}

echo "\n=== Test Complete ===\n";
?>