<?php
// MFT GROUP Validate M-PESA Payment API
// This API validates M-PESA reference codes and records payments

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Initialize database connection
global $pdo;
$pdo = getDatabaseConnection();

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handlePostRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handlePostRequest() {
    global $pdo;
    
    // Validate M-PESA payment
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Required fields
    $required_fields = ['reference_code', 'member_id'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        // Check if the reference code exists in the payments table
        $payment_sql = "SELECT p.*, m.id as member_numeric_id FROM payments p 
                        JOIN members m ON p.member_id = m.id 
                        WHERE p.reference = ? AND m.member_id = ?";
        $payment_stmt = $pdo->prepare($payment_sql);
        $payment_stmt->execute([$input['reference_code'], $input['member_id']]);
        $payment = $payment_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$payment) {
            // Check if this is a pending payment that needs to be validated
            // In a real implementation, you would integrate with the M-PESA API to verify the transaction
            // For now, we'll simulate a successful validation for demonstration purposes
            
            // Let's assume the reference code format is PAY-YYYY-XXXXXX
            if (strpos($input['reference_code'], 'PAY-') === 0) {
                // Simulate successful validation
                // In a real implementation, you would:
                // 1. Call M-PESA API to verify the transaction
                // 2. Get transaction details (amount, date, etc.)
                // 3. Update or create payment records accordingly
                
                // For demonstration, we'll create a dummy payment record
                $pdo->beginTransaction();
                
                try {
                    // Get member's numeric ID
                    $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
                    $member_stmt = $pdo->prepare($member_check_sql);
                    $member_stmt->execute([$input['member_id']]);
                    $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$member_result) {
                        http_response_code(404);
                        echo json_encode(['error' => 'Member not found']);
                        return;
                    }
                    
                    $member_numeric_id = $member_result['id'];
                    
                    // Create a payment record
                    $payment_sql = "INSERT INTO payments (reference, member_id, account_id, amount, payment_method, description, status, processed_by, processed_at) 
                                    VALUES (?, ?, ?, ?, ?, ?, 'Completed', ?, ?)";
                    $payment_stmt = $pdo->prepare($payment_sql);
                    $payment_stmt->execute([
                        $input['reference_code'],
                        $member_numeric_id,
                        1, // Assuming savings account ID is 1
                        1000, // Default amount for demonstration
                        'M-PESA',
                        'Validated payment',
                        $member_numeric_id,
                        date('Y-m-d H:i:s')
                    ]);
                    
                    $payment_id = $pdo->lastInsertId();
                    
                    // Create a contribution record
                    $contribution_sql = "INSERT INTO contributions 
                                        (member_id, payment_id, amount, contribution_date, period_month, period_year, description, status, confirmed_by, confirmed_at) 
                                        VALUES (?, ?, ?, ?, ?, ?, 'Validated payment', 'Confirmed', ?, ?)";
                    $contribution_stmt = $pdo->prepare($contribution_sql);
                    $contribution_stmt->execute([
                        $member_numeric_id,
                        $payment_id,
                        1000, // Default amount for demonstration
                        date('Y-m-d'), // Today's date
                        date('n'), // Current month
                        date('Y'), // Current year
                        'Validated payment',
                        $member_numeric_id,
                        date('Y-m-d H:i:s')
                    ]);
                    
                    $pdo->commit();
                    
                    // Return success response
                    http_response_code(200);
                    echo json_encode([
                        'success' => true,
                        'message' => 'Payment validated successfully and recorded',
                        'payment_id' => $payment_id,
                        'amount' => 1000
                    ]);
                } catch (Exception $e) {
                    $pdo->rollback();
                    throw $e;
                }
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Invalid M-Pesa reference code. Please confirm and try again.']);
            }
        } else {
            // Payment already exists
            if ($payment['status'] === 'Completed') {
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'message' => 'Payment already validated and recorded',
                    'payment_id' => $payment['id'],
                    'amount' => $payment['amount']
                ]);
            } else {
                // Update payment status to Completed
                $update_sql = "UPDATE payments SET status = 'Completed', processed_at = ? WHERE id = ?";
                $update_stmt = $pdo->prepare($update_sql);
                $update_stmt->execute([date('Y-m-d H:i:s'), $payment['id']]);
                
                // Also update related contribution if exists
                $update_contrib_sql = "UPDATE contributions SET status = 'Confirmed', confirmed_at = ? WHERE payment_id = ?";
                $update_contrib_stmt = $pdo->prepare($update_contrib_sql);
                $update_contrib_stmt->execute([date('Y-m-d H:i:s'), $payment['id']]);
                
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'message' => 'Payment validated and status updated successfully',
                    'payment_id' => $payment['id'],
                    'amount' => $payment['amount']
                ]);
            }
        }
    } catch (Exception $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollback();
        }
        throw $e;
    }
}
?>