<?php
// MFT GROUP Official Photo Upload API
// This API handles profile photo uploads for officials in settings

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Function to create directory if it doesn't exist
function createDirectoryIfNotExists($path) {
    if (!file_exists($path)) {
        mkdir($path, 0755, true);
    }
    return is_writable($path);
}

// Function to validate image file
function validateImageFile($file) {
    // Check if file was uploaded without errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['valid' => false, 'error' => 'File upload error'];
    }
    
    // Check file size (max 5MB)
    if ($file['size'] > 5 * 1024 * 1024) {
        return ['valid' => false, 'error' => 'File size must be less than 5MB'];
    }
    
    // Check file type
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
    if (!in_array($file['type'], $allowedTypes)) {
        return ['valid' => false, 'error' => 'Only JPEG, PNG, and GIF files are allowed'];
    }
    
    // Verify it's actually an image
    $imageInfo = getimagesize($file['tmp_name']);
    if ($imageInfo === false) {
        return ['valid' => false, 'error' => 'File is not a valid image'];
    }
    
    return ['valid' => true];
}

// Function to generate unique filename
function generateUniqueFilename($originalName, $settingKey) {
    $extension = pathinfo($originalName, PATHINFO_EXTENSION);
    $timestamp = time();
    return "official_{$settingKey}_{$timestamp}." . strtolower($extension);
}

try {
    // Check if request method is POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        exit;
    }
    
    // Check if setting key is provided
    $settingKey = isset($_POST['setting_key']) ? $_POST['setting_key'] : null;
    if (!$settingKey) {
        http_response_code(400);
        echo json_encode(['error' => 'Setting key is required']);
        exit;
    }
    
    // Check if file was uploaded
    if (!isset($_FILES['profile_photo']) || $_FILES['profile_photo']['error'] === UPLOAD_ERR_NO_FILE) {
        http_response_code(400);
        echo json_encode(['error' => 'No file uploaded']);
        exit;
    }
    
    // Validate the uploaded file
    $validation = validateImageFile($_FILES['profile_photo']);
    if (!$validation['valid']) {
        http_response_code(400);
        echo json_encode(['error' => $validation['error']]);
        exit;
    }
    
    // Create Officials Photos directory
    $uploadDir = __DIR__ . '/../Officials Photos';
    if (!createDirectoryIfNotExists($uploadDir)) {
        http_response_code(500);
        echo json_encode(['error' => 'Unable to create upload directory']);
        exit;
    }
    
    // Generate unique filename
    $filename = generateUniqueFilename($_FILES['profile_photo']['name'], $settingKey);
    $uploadPath = $uploadDir . '/' . $filename;
    
    // Move uploaded file to destination
    if (move_uploaded_file($_FILES['profile_photo']['tmp_name'], $uploadPath)) {
        // Return success response with the path
        $relativePath = 'Officials Photos/' . $filename;
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Official photo uploaded successfully',
            'filename' => $filename,
            'path' => $relativePath
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to move uploaded file']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}
?>