<?php
// MFT GROUP System Settings API
// This API provides system settings data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    error_log("OPTIONS request handled");
    http_response_code(200);
    exit;
}

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

// Debug logging
error_log("Settings API called with method: $method");
if (isset($_GET['key'])) {
    error_log("Settings API called with key: " . $_GET['key']);
}

try {
    switch ($method) {
        case 'GET':
            error_log("Handling GET request");
            handleGetRequest();
            break;
        case 'POST':
            error_log("Handling POST request");
            handlePostRequest();
            break;
        case 'PUT':
            error_log("Handling PUT request");
            handlePutRequest();
            break;
        default:
            error_log("Method not allowed: $method");
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Settings API error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    error_log("handleGetRequest called");
    if (isset($_GET['key'])) {
        // Get specific setting by key
        getSetting($_GET['key']);
    } else {
        // Get all settings
        getSettings();
    }
}

function getSetting($key) {
    error_log("getSetting called with key: $key");
    $sql = "SELECT id, setting_key, setting_value, description, created_at, updated_at
            FROM system_settings
            WHERE setting_key = ?";
    
    $setting = fetchSingleRow($sql, [$key]);
    
    if ($setting) {
        error_log("Setting found: " . $setting['setting_key']);
        http_response_code(200);
        echo json_encode($setting);
    } else {
        error_log("Setting not found: $key");
        http_response_code(404);
        echo json_encode(['error' => 'Setting not found']);
    }
}

function getSettings() {
    error_log("getSettings called");
    $sql = "SELECT id, setting_key, setting_value, description, created_at, updated_at
            FROM system_settings
            ORDER BY setting_key";
    
    $settings = executeQuery($sql);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($settings),
        'settings' => $settings
    ]);
}

function handlePostRequest() {
    error_log("handlePostRequest called");
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error_log("Invalid JSON data in POST request");
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['setting_key', 'setting_value'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            error_log("Missing required field in POST request: $field");
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Check if setting already exists
    $check_sql = "SELECT id FROM system_settings WHERE setting_key = ?";
    $existing = fetchSingleRow($check_sql, [$input['setting_key']]);
    
    if ($existing) {
        error_log("Setting already exists in POST request: " . $input['setting_key']);
        http_response_code(400);
        echo json_encode(['error' => 'Setting with this key already exists']);
        return;
    }
    
    // Insert setting
    $sql = "INSERT INTO system_settings (setting_key, setting_value, description) 
            VALUES (?, ?, ?)";
    
    $params = [
        $input['setting_key'],
        $input['setting_value'],
        $input['description'] ?? null
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $setting_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Setting created successfully',
            'setting_id' => $setting_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handlePutRequest() {
    error_log("handlePutRequest called");
    
    if (!isset($_GET['key'])) {
        error_log("Setting key is required for PUT request");
        http_response_code(400);
        echo json_encode(['error' => 'Setting key is required']);
        return;
    }
    
    $key = $_GET['key'];
    error_log("PUT request for key: $key");
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error_log("Invalid JSON data in PUT request");
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    error_log("PUT request data: " . print_r($input, true));
    
    // Prepare update data
    $update_fields = [];
    $update_params = [];
    
    // Add setting_value if provided and not null
    if (isset($input['setting_value']) && $input['setting_value'] !== null) {
        $update_fields[] = "setting_value = ?";
        $update_params[] = $input['setting_value'];
        error_log("Adding setting_value to update: " . $input['setting_value']);
    }
    
    // Add description if provided and not null
    if (isset($input['description']) && $input['description'] !== null) {
        $update_fields[] = "description = ?";
        $update_params[] = $input['description'];
        error_log("Adding description to update: " . $input['description']);
    }
    
    // Check if we have fields to update
    if (empty($update_fields)) {
        error_log("No fields to update in PUT request");
        http_response_code(400);
        echo json_encode(['error' => 'No fields to update']);
        return;
    }
    
    // Build the SQL query
    $sql = "UPDATE system_settings SET " . implode(", ", $update_fields) . " WHERE setting_key = ?";
    $update_params[] = $key;
    
    error_log("SQL query: $sql");
    error_log("SQL params: " . print_r($update_params, true));
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $update_params);
        error_log("Rows affected: $rows_affected");
        
        if ($rows_affected === 0) {
            // Check if the setting actually exists
            $check_sql = "SELECT id FROM system_settings WHERE setting_key = ?";
            $existing = fetchSingleRow($check_sql, [$key]);
            if (!$existing) {
                error_log("Setting not found in database: $key");
                http_response_code(404);
                echo json_encode(['error' => 'Setting not found']);
                rollbackTransaction();
                return;
            } else {
                error_log("Setting exists but no rows were affected: $key");
                http_response_code(200); // Still return success since no changes were needed
                echo json_encode(['message' => 'No changes were needed']);
                rollbackTransaction();
                return;
            }
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Setting updated successfully'
        ]);
        error_log("Setting updated successfully: $key");
    } catch (Exception $e) {
        error_log("Error updating setting: " . $e->getMessage());
        try {
            rollbackTransaction();
        } catch (Exception $rollbackException) {
            error_log("Rollback error: " . $rollbackException->getMessage());
        }
        throw $e;
    }
}
?>