<?php
// MFT GROUP SMS Notification Service
// This script provides functionality to send SMS notifications to members

require_once __DIR__ . '/../database/config.php';

/**
 * Send SMS notification to a member
 * @param string $phoneNumber Member's phone number
 * @param string $message SMS message content
 * @return array Result with success status and message
 */
function sendSMS($phoneNumber, $message) {
    // In a real implementation, you would integrate with an SMS gateway service
    // Examples include:
    // - Twilio
    // - Nexmo/Vonage
    // - Africa's Talking
    // - Local Kenyan SMS gateways
    
    // For demonstration purposes, we'll just log the SMS
    error_log("SMS to $phoneNumber: $message");
    
    // Simulate SMS sending
    // In a real implementation, you would make an API call to your SMS provider
    try {
        // Example using a fictional SMS API:
        /*
        $smsProviderUrl = "https://api.smsprovider.com/send";
        $apiKey = "YOUR_API_KEY";
        
        $data = [
            'api_key' => $apiKey,
            'to' => $phoneNumber,
            'message' => $message,
            'sender_id' => 'MFTGROUP'
        ];
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $result = file_get_contents($smsProviderUrl, false, $context);
        
        if ($result === FALSE) {
            throw new Exception('Failed to send SMS');
        }
        
        return ['success' => true, 'message' => 'SMS sent successfully'];
        */
        
        // For now, just return success
        return ['success' => true, 'message' => 'SMS would be sent in production'];
    } catch (Exception $e) {
        error_log("Error sending SMS: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to send SMS: ' . $e->getMessage()];
    }
}

/**
 * Send fine notification SMS to member
 * @param int $memberId Member ID
 * @param float $fineAmount Fine amount
 * @param string $reason Fine reason
 * @param float $outstandingFines Total outstanding fines
 * @return array Result with success status and message
 */
function sendFineNotificationSMS($memberId, $fineAmount, $reason, $outstandingFines) {
    try {
        // Get member details
        $pdo = getDatabaseConnection();
        $sql = "SELECT first_name, last_name, phone FROM members WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$memberId]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member || !$member['phone']) {
            return ['success' => false, 'message' => 'Member not found or phone number missing'];
        }
        
        // Format the SMS message
        $message = "Dear {$member['first_name']}, a fine of Ksh " . number_format($fineAmount, 2) . 
                  " has been recorded for $reason. Your total outstanding fines: Ksh " . 
                  number_format($outstandingFines, 2) . ". MFT GROUP";
        
        // Send the SMS
        return sendSMS($member['phone'], $message);
    } catch (Exception $e) {
        error_log("Error sending fine notification SMS: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to send fine notification: ' . $e->getMessage()];
    }
}

/**
 * Send loan notification SMS to member
 * @param int $memberId Member ID
 * @param string $loanType Loan type
 * @param string $status Loan status (approved/rejected)
 * @param array $details Additional details
 * @return array Result with success status and message
 */
function sendLoanNotificationSMS($memberId, $loanType, $status, $details = []) {
    try {
        // Get member details
        $pdo = getDatabaseConnection();
        $sql = "SELECT first_name, last_name, phone FROM members WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$memberId]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member || !$member['phone']) {
            return ['success' => false, 'message' => 'Member not found or phone number missing'];
        }
        
        // Format the SMS message based on status
        if ($status === 'approved') {
            $amount = isset($details['principal_amount']) ? number_format($details['principal_amount'], 2) : 'N/A';
            $message = "Dear {$member['first_name']}, your $loanType loan application for Ksh $amount has been APPROVED. MFT GROUP";
        } else if ($status === 'rejected') {
            $reason = isset($details['reason']) ? $details['reason'] : 'Not specified';
            $message = "Dear {$member['first_name']}, your $loanType loan application has been REJECTED. Reason: $reason. MFT GROUP";
        } else {
            $message = "Dear {$member['first_name']}, there is an update on your $loanType loan application. MFT GROUP";
        }
        
        // Send the SMS
        return sendSMS($member['phone'], $message);
    } catch (Exception $e) {
        error_log("Error sending loan notification SMS: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to send loan notification: ' . $e->getMessage()];
    }
}

// Handle API requests - only when accessed directly, not when included
if (isset($_SERVER['SCRIPT_NAME']) && basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    header("Content-Type: application/json");
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: POST");
    header("Access-Control-Allow-Headers: Content-Type");
    
    $method = $_SERVER['REQUEST_METHOD'];
    
    if ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid JSON data']);
            exit;
        }
        
        $action = isset($input['action']) ? $input['action'] : '';
        
        switch ($action) {
            case 'send_fine_notification':
                if (!isset($input['member_id']) || !isset($input['amount']) || !isset($input['reason']) || !isset($input['outstanding_fines'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Missing required parameters']);
                    exit;
                }
                
                $result = sendFineNotificationSMS(
                    $input['member_id'], 
                    $input['amount'], 
                    $input['reason'], 
                    $input['outstanding_fines']
                );
                
                if ($result['success']) {
                    http_response_code(200);
                    echo json_encode(['message' => $result['message']]);
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => $result['message']]);
                }
                break;
                
            case 'send_loan_notification':
                if (!isset($input['member_id']) || !isset($input['loan_type']) || !isset($input['status'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Missing required parameters']);
                    exit;
                }
                
                $result = sendLoanNotificationSMS(
                    $input['member_id'], 
                    $input['loan_type'], 
                    $input['status'], 
                    $input['details'] ?? []
                );
                
                if ($result['success']) {
                    http_response_code(200);
                    echo json_encode(['message' => $result['message']]);
                } else {
                    http_response_code(500);
                    echo json_encode(['error' => $result['message']]);
                }
                break;
                
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid action']);
                break;
        }
        exit(); // Exit after handling API request
    } else {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        exit(); // Exit after handling API request
    }
}
?>