<?php
// MFT GROUP Projects API
// This API provides project data and handles project assignments for members

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['member_id'])) {
        // Get projects assigned to a specific member
        getMemberProjects($_GET['member_id']);
    } else if (isset($_GET['id'])) {
        // Get specific project
        getProject($_GET['id']);
    } else {
        // Get all projects
        getProjects();
    }
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    if (isset($input['action']) && $input['action'] === 'assign') {
        // Assign projects to member
        assignProjectsToMember($input);
    } else {
        // Create new project
        createProject($input);
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Project ID is required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    updateProject($_GET['id'], $input);
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Project ID is required']);
        return;
    }
    
    deleteProject($_GET['id']);
}

function getProjects() {
    // Get filter parameters
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $search = isset($_GET['search']) ? $_GET['search'] : null;
    $date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
    $date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;
    
    // Build WHERE clause
    $where_conditions = [];
    $params = [];
    
    if ($status && $status !== 'all') {
        $where_conditions[] = "status = ?";
        $params[] = $status;
    }
    
    if ($search && $search !== 'all') {
        $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
        $search_param = '%' . $search . '%';
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    // Add date range filters (using completion_date instead of start_date)
    if ($date_from) {
        $where_conditions[] = "completion_date >= ?";
        $params[] = $date_from;
    }
    
    if ($date_to) {
        $where_conditions[] = "completion_date <= ?";
        $params[] = $date_to;
    }
    
    $where_clause = '';
    if (!empty($where_conditions)) {
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
    }
    
    $sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price,
                   created_at, updated_at
            FROM projects 
            $where_clause
            ORDER BY completion_date DESC";
    
    $projects = executeQuery($sql, $params);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($projects),
        'projects' => $projects
    ]);
}

function getProject($id) {
    // Check if member_id is provided in the request
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    
    if ($member_id) {
        // Get project with member-specific information
        getProjectForMember($id, $member_id);
        return;
    }
    
    $sql = "SELECT id, name, description, location, completion_date, status, total_shares, share_price,
                   created_at, updated_at
            FROM projects 
            WHERE id = ?";
    
    $project = fetchSingleRow($sql, [$id]);
    
    if ($project) {
        http_response_code(200);
        echo json_encode($project);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Project not found']);
    }
}

function getProjectForMember($project_id, $member_id) {
    // Validate member ID
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    // Check if the member_id is numeric (database ID) or a reference ID
    $actual_member_id = $member_id;
    if (!is_numeric($member_id)) {
        // It's a reference ID, look up the actual database ID
        $member_lookup_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_lookup_sql, [$member_id]);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Use the actual database ID
        $actual_member_id = $member_result['id'];
    }
    
    // Get project with member-specific information
    // Fixed field names to match the database schema
    $sql = "SELECT p.id, p.name, p.description, p.location, p.completion_date, p.status, p.total_shares, p.share_price,
                   mp.status as role, mp.amount_contributed as contribution_amount, mp.join_date as joined_date, mp.status as member_project_status,
                   p.created_at, p.updated_at
            FROM projects p
            LEFT JOIN member_projects mp ON p.id = mp.project_id AND mp.member_id = ?
            WHERE p.id = ?";
    
    $project = fetchSingleRow($sql, [$actual_member_id, $project_id]);
    
    if ($project) {
        http_response_code(200);
        echo json_encode($project);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Project not found']);
    }
}

function getMemberProjects($member_id) {
    // Validate member ID
    if (!$member_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    // Check if the member_id is numeric (database ID) or a reference ID
    if (!is_numeric($member_id)) {
        // It's a reference ID, look up the actual database ID
        $member_lookup_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_result = fetchSingleRow($member_lookup_sql, [$member_id]);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Use the actual database ID
        $member_id = $member_result['id'];
    }
    
    // Fixed field names to match the database schema
    $sql = "SELECT p.id, p.name, p.description, p.location, p.completion_date, p.status as status, p.total_shares, p.share_price,
                   mp.status as role, mp.amount_contributed as contribution_amount, mp.join_date as joined_date, mp.status as member_project_status,
                   p.created_at, p.updated_at
            FROM projects p
            JOIN member_projects mp ON p.id = mp.project_id
            WHERE mp.member_id = ?
            ORDER BY p.completion_date DESC";
    
    try {
        $projects = executeQuery($sql, [$member_id]);
        
        // Transform the data to match what the frontend expects
        $transformed_projects = array_map(function($project) {
            return [
                'id' => $project['id'],
                'name' => $project['name'],
                'description' => $project['description'],
                'location' => $project['location'],
                'completion_date' => $project['completion_date'],
                'status' => $project['status'],
                'role' => $project['role'],
                'contribution_amount' => $project['contribution_amount'],
                'joined_date' => $project['joined_date'],
                'member_project_status' => $project['member_project_status'],
                'total_shares' => $project['total_shares'],
                'share_price' => $project['share_price'],
                'created_at' => $project['created_at'],
                'updated_at' => $project['updated_at']
            ];
        }, $projects);
        
        http_response_code(200);
        echo json_encode([
            'count' => count($transformed_projects),
            'projects' => $transformed_projects
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

function createProject($input) {
    // Validate required fields
    $required_fields = ['name'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // For now, we'll use a default admin user ID (you may want to get this from authentication)
    $default_user_id = 1; // This should be the authenticated user ID in a real implementation
    
    $sql = "INSERT INTO projects (name, description, location, start_date, end_date, completion_date, total_shares, share_price, target_amount, current_amount, status, manager_id, created_by) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $params = [
        $input['name'],
        $input['description'] ?? null,
        $input['location'] ?? null,
        $input['start_date'] ?? null,
        $input['end_date'] ?? null,
        $input['completion_date'] ?? null,
        $input['total_shares'] ?? 0,
        $input['share_price'] ?? 0.00,
        $input['target_amount'] ?? 0.00,
        $input['current_amount'] ?? 0.00,
        $input['status'] ?? 'Planning',
        $input['manager_id'] ?? $default_user_id,
        $input['created_by'] ?? $default_user_id
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $project_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Project created successfully',
            'project_id' => $project_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function updateProject($id, $input) {
    // Build dynamic update query based on provided fields
    $updates = [];
    $params = [];
    
    // Add fields that can be updated
    $updatable_fields = [
        'name', 'description', 'location', 'start_date', 'end_date', 
        'completion_date', 'total_shares', 'share_price', 
        'target_amount', 'current_amount', 'status'
    ];
    
    foreach ($updatable_fields as $field) {
        if (isset($input[$field])) {
            $updates[] = "$field = ?";
            $params[] = $input[$field];
        }
    }
    
    // Only proceed if there are fields to update
    if (empty($updates)) {
        http_response_code(400);
        echo json_encode(['error' => 'No valid fields to update']);
        return;
    }
    
    // Add the ID for the WHERE clause
    $params[] = $id;
    
    $sql = "UPDATE projects SET " . implode(', ', $updates) . ", updated_at = CURRENT_TIMESTAMP WHERE id = ?";
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Project updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function deleteProject($id) {
    try {
        beginTransaction();
        
        // Delete member_project associations first (due to foreign key constraints)
        $sql1 = "DELETE FROM member_projects WHERE project_id = ?";
        executeNonQuery($sql1, [$id]);
        
        // Delete the project
        $sql2 = "DELETE FROM projects WHERE id = ?";
        executeNonQuery($sql2, [$id]);
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Project deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function assignProjectsToMember($input) {
    if (!isset($input['member_id']) || !isset($input['project_ids'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID and project IDs are required']);
        return;
    }
    
    $member_id = $input['member_id'];
    $project_ids = $input['project_ids'];
    $joined_date = $input['joined_date'] ?? date('Y-m-d');
    $role = $input['role'] ?? 'Investor';
    $contribution_amount = $input['contribution_amount'] ?? 0.00;
    
    try {
        beginTransaction();
        
        // Assign new projects
        foreach ($project_ids as $project_id) {
            // Check if assignment already exists
            $check_sql = "SELECT id FROM member_projects WHERE member_id = ? AND project_id = ?";
            $existing = fetchSingleRow($check_sql, [$member_id, $project_id]);
            
            if ($existing) {
                // Update existing assignment
                $update_sql = "UPDATE member_projects SET 
                              role = ?, 
                              contribution_amount = ?, 
                              joined_date = ?, 
                              status = 'Active',
                              updated_at = CURRENT_TIMESTAMP
                              WHERE member_id = ? AND project_id = ?";
                executeNonQuery($update_sql, [$role, $contribution_amount, $joined_date, $member_id, $project_id]);
            } else {
                // Create new assignment
                $insert_sql = "INSERT INTO member_projects (member_id, project_id, role, contribution_amount, joined_date, status) 
                               VALUES (?, ?, ?, ?, ?, ?)";
                executeNonQuery($insert_sql, [$member_id, $project_id, $role, $contribution_amount, $joined_date, 'Active']);
            }
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Projects assigned to member successfully',
            'member_id' => $member_id,
            'assigned_projects' => count($project_ids)
        ]);
        
        // Trigger real-time update to member dashboard
        triggerMemberDashboardUpdate($member_id, $project_ids);
        
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function triggerMemberDashboardUpdate($member_id, $project_ids) {
    // In a real implementation, this would send a WebSocket message or similar
    // to update the member's dashboard in real-time
    error_log("Triggering real-time update for member $member_id with projects: " . json_encode($project_ids));
}
?>