<?php
/**
 * PDF Utility Functions using mPDF
 * This file provides general PDF generation capabilities for the MFT application
 */

// Try to include mPDF library
$mpdfAutoloadPath = __DIR__ . '/../vendor/autoload.php';
if (file_exists($mpdfAutoloadPath)) {
    require_once $mpdfAutoloadPath;
}

/**
 * Generate a PDF from HTML content
 * 
 * @param string $html HTML content to convert to PDF
 * @param string $filename Name of the PDF file (without extension)
 * @param array $options mPDF configuration options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return mixed PDF output or file path if saved to file
 */
function generatePdfFromHtml($html, $filename = 'document', $options = [], $output_mode = 'D') {
    try {
        // Check if mPDF is available
        if (!class_exists('\Mpdf\Mpdf')) {
            // Try alternative PDF generation methods
            return generatePdfFallback($html, $filename, $options, $output_mode);
        }
        
        // Default configuration
        $default_options = [
            'format' => 'A4',
            'margin_left' => 15,
            'margin_right' => 15,
            'margin_top' => 20,
            'margin_bottom' => 20,
            'margin_header' => 10,
            'margin_footer' => 10
        ];
        
        // Merge user options with defaults
        $config = array_merge($default_options, $options);
        
        // Create new mPDF instance
        $mpdfClass = '\Mpdf\Mpdf';
        $mpdf = new $mpdfClass($config);
        
        // Set document properties
        $mpdf->SetTitle($filename);
        $mpdf->SetAuthor('MFT GROUP');
        
        // Add header and footer if provided in options
        if (isset($options['header'])) {
            $mpdf->SetHTMLHeader($options['header']);
        }
        
        if (isset($options['footer'])) {
            $mpdf->SetHTMLFooter($options['footer']);
        }
        
        // Write HTML content to PDF
        $mpdf->WriteHTML($html);
        
        // Handle different output modes
        switch ($output_mode) {
            case 'D': // Download
                $mpdf->Output($filename . '.pdf', 'D');
                break;
            case 'I': // Inline
                $mpdf->Output($filename . '.pdf', 'I');
                break;
            case 'F': // File
                $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
                $mpdf->Output($file_path, 'F');
                return $file_path;
                break;
            default:
                $mpdf->Output($filename . '.pdf', 'D');
        }
        
        return true;
    } catch (Exception $e) {
        error_log('PDF Generation Error: ' . $e->getMessage());
        // Try fallback method
        return generatePdfFallback($html, $filename, $options, $output_mode);
    }
}

/**
 * Fallback PDF generation method when mPDF is not available
 * 
 * @param string $html HTML content to convert to PDF
 * @param string $filename Name of the PDF file (without extension)
 * @param array $options Configuration options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return mixed PDF output or file path if saved to file
 */
function generatePdfFallback($html, $filename = 'document', $options = [], $output_mode = 'D') {
    try {
        // Try to use TCPDF as fallback
        $tcpdfPath = __DIR__ . '/../lib/tcpdf/SimpleTCPDF.php';
        if (file_exists($tcpdfPath)) {
            require_once $tcpdfPath;
            
            if (class_exists('SimpleTCPDF')) {
                $tcpdfClass = 'SimpleTCPDF';
                $pdf = new $tcpdfClass();
                $pdf->AddPage();
                $pdf->WriteHTML($html);
                $pdf_content = $pdf->Output('', 'S');
                
                switch ($output_mode) {
                    case 'D': // Download
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                        
                    case 'I': // Inline
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: inline; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                        
                    case 'F': // File
                        $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
                        $dir = dirname($file_path);
                        if (!file_exists($dir)) {
                            mkdir($dir, 0777, true);
                        }
                        file_put_contents($file_path, $pdf_content);
                        return $file_path;
                        
                    default:
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                }
            }
        }
        
        // If TCPDF is not available, create a simple text-based PDF
        return createSimpleTextPdf($html, $filename, $output_mode);
    } catch (Exception $e) {
        error_log('PDF Fallback Generation Error: ' . $e->getMessage());
        throw new Exception('Failed to generate PDF with any available method: ' . $e->getMessage());
    }
}

/**
 * Create a simple text-based PDF as a last resort
 * @param string $html HTML content
 * @param string $filename Name of the PDF file (without extension)
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return mixed PDF output or file path if saved to file
 */
function createSimpleTextPdf($html, $filename = 'document', $output_mode = 'D') {
    // Extract text content
    $dom = new DOMDocument();
    libxml_use_internal_errors(true);
    $dom->loadHTML($html);
    libxml_clear_errors();
    
    $text = '';
    $body = $dom->getElementsByTagName('body');
    if ($body->length > 0) {
        $text = $body->item(0)->textContent;
    } else {
        $text = $dom->textContent;
    }
    
    // Create a very simple PDF structure
    $pdf = "%PDF-1.4\n";
    $pdf .= "1 0 obj\n<<\n/Type /Catalog\n/Pages 2 0 R\n>>\nendobj\n";
    $pdf .= "2 0 obj\n<<\n/Type /Pages\n/Kids [3 0 R]\n/Count 1\n>>\nendobj\n";
    $pdf .= "3 0 obj\n<<\n/Type /Page\n/Parent 2 0 R\n/MediaBox [0 0 595 842]\n/Contents 4 0 R\n/Resources <<\n/Font <<\n/F1 5 0 R\n>>\n>>\n>>\nendobj\n";
    $pdf .= "5 0 obj\n<<\n/Type /Font\n/Subtype /Type1\n/BaseFont /Helvetica\n>>\nendobj\n";
    
    // Simple content
    $content = "BT\n/F1 12 Tf\n72 720 Td\n(" . addslashes("Document generated on: " . date('Y-m-d H:i:s')) . ") Tj\n" .
               "\n72 700 Td\n(" . addslashes("Note: This is a simplified PDF due to system limitations.") . ") Tj\n" .
               "\n72 680 Td\n(" . addslashes(substr($text, 0, 1000) . (strlen($text) > 1000 ? '...' : '')) . ") Tj\nET\n";
    
    $pdf .= "4 0 obj\n<<\n/Length " . strlen($content) . "\n>>\nstream\n" . $content . "endstream\nendobj\n";
    $pdf .= "xref\n0 6\n0000000000 65535 f \n0000000010 00000 n \n0000000053 00000 n \n0000000114 00000 n \n0000000275 00000 n \n0000000221 00000 n \n";
    $pdf .= "trailer\n<<\n/Size 6\n/Root 1 0 R\n>>\nstartxref\n" . strlen($pdf) . "\n%%EOF";
    
    switch ($output_mode) {
        case 'D': // Download
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
            echo $pdf;
            exit;
            
        case 'I': // Inline
            header('Content-Type: application/pdf');
            header('Content-Disposition: inline; filename="' . $filename . '.pdf"');
            echo $pdf;
            exit;
            
        case 'F': // File
            $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
            $dir = dirname($file_path);
            if (!file_exists($dir)) {
                mkdir($dir, 0777, true);
            }
            file_put_contents($file_path, $pdf);
            return $file_path;
            
        default:
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
            echo $pdf;
            exit;
    }
}

/**
 * Generate a simple PDF report from data array
 * 
 * @param array $data Associative array of data to include in report
 * @param string $title Report title
 * @param string $filename Name of the PDF file
 * @return void
 */
function generateSimpleReport($data, $title = 'Report', $filename = 'report') {
    // Generate HTML for the report
    $html = generateReportHtml($data, $title);
    
    // Generate PDF
    return generatePdfFromHtml($html, $filename);
}

/**
 * Generate HTML for a simple report
 * 
 * @param array $data Data to include in report
 * @param string $title Report title
 * @return string HTML content
 */
function generateReportHtml($data, $title = 'Report') {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: DejaVu Sans, sans-serif; font-size: 11pt; }
            .header { text-align: center; margin-bottom: 20px; }
            .header h1 { color: #14375d; margin: 0; }
            table { width: 100%; border-collapse: collapse; margin: 20px 0; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .footer { margin-top: 30px; font-size: 9pt; color: #666; text-align: center; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>' . htmlspecialchars($title) . '</h1>
        </div>
        <table>
            <thead>
                <tr>';
    
    // Generate table headers from first row keys if data is not empty
    if (!empty($data)) {
        $first_row = reset($data);
        foreach (array_keys($first_row) as $header) {
            $html .= '<th>' . htmlspecialchars(ucfirst(str_replace('_', ' ', $header))) . '</th>';
        }
        $html .= '</tr>
            </thead>
            <tbody>';
        
        // Generate table rows
        foreach ($data as $row) {
            $html .= '<tr>';
            foreach ($row as $cell) {
                $html .= '<td>' . htmlspecialchars($cell) . '</td>';
            }
            $html .= '</tr>';
        }
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            Generated on ' . date('F j, Y') . ' by MFT GROUP System
        </div>
    </body>
    </html>';
    
    return $html;
}

/**
 * Generate a PDF table from data
 * 
 * @param array $headers Table headers
 * @param array $data Table data (array of arrays)
 * @param string $title Table title
 * @param string $filename PDF filename
 * @return void
 */
function generatePdfTable($headers, $data, $title = '', $filename = 'table') {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: DejaVu Sans, sans-serif; font-size: 10pt; }
            .title { text-align: center; font-size: 14pt; margin-bottom: 20px; }
            table { width: 100%; border-collapse: collapse; }
            th, td { border: 1px solid #000; padding: 6px; text-align: left; }
            th { background-color: #e0e0e0; font-weight: bold; }
        </style>
    </head>
    <body>';
    
    if (!empty($title)) {
        $html .= '<div class="title">' . htmlspecialchars($title) . '</div>';
    }
    
    $html .= '<table>
        <thead>
            <tr>';
    
    // Add headers
    foreach ($headers as $header) {
        $html .= '<th>' . htmlspecialchars($header) . '</th>';
    }
    
    $html .= '</tr>
        </thead>
        <tbody>';
    
    // Add data rows
    foreach ($data as $row) {
        $html .= '<tr>';
        foreach ($row as $cell) {
            $html .= '<td>' . htmlspecialchars($cell) . '</td>';
        }
        $html .= '</tr>';
    }
    
    $html .= '
        </tbody>
    </table>
    </body>
    </html>';
    
    return generatePdfFromHtml($html, $filename);
}

/**
 * Enhanced version that supports multiple PDF engines including Dompdf
 * 
 * @param string $html HTML content to convert to PDF
 * @param string $filename Name of the PDF file (without extension)
 * @param array $options Configuration options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @param string $engine PDF engine to use: 'mpdf', 'dompdf', 'tcpdf'
 * @return mixed PDF output or file path if saved to file
 */
function generatePdfFromHtmlMultiEngine($html, $filename = 'document', $options = [], $output_mode = 'D', $engine = 'mpdf') {
    try {
        switch ($engine) {
            case 'dompdf':
                // Try to use Dompdf
                $dompdfUtilsPath = __DIR__ . '/dompdf-utils.php';
                if (file_exists($dompdfUtilsPath)) {
                    require_once $dompdfUtilsPath;
                    if (function_exists('generatePdfWithDompdf')) {
                        return generatePdfWithDompdf($html, $filename, $options, $output_mode);
                    }
                }
                // If Dompdf is not available, fall through to mPDF
                error_log('Dompdf not available, falling back to mPDF');
                
            case 'mpdf':
            default:
                // Use existing mPDF implementation
                return generatePdfFromHtml($html, $filename, $options, $output_mode);
        }
    } catch (Exception $e) {
        error_log("PDF Generation Error with $engine: " . $e->getMessage());
        
        // Fallback to another engine
        switch ($engine) {
            case 'dompdf':
                // Fallback to mPDF
                try {
                    return generatePdfFromHtml($html, $filename, $options, $output_mode);
                } catch (Exception $e2) {
                    error_log("mPDF fallback also failed: " . $e2->getMessage());
                    throw new Exception("All PDF generation methods failed: " . $e->getMessage() . "; " . $e2->getMessage());
                }
                
            default:
                throw new Exception("PDF generation failed: " . $e->getMessage());
        }
    }
}

?>