<?php
// MFT GROUP Payments API
// This API provides payment data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific payment
        getPayment($_GET['id']);
    } else if (isset($_GET['member_id'])) {
        // Get payments for a specific member
        getMemberPayments($_GET['member_id']);
    } else if (isset($_GET['account_id'])) {
        // Get payments for a specific account
        getAccountPayments($_GET['account_id']);
    } else {
        // Get all payments with optional filters
        getPayments();
    }
}

function getPayment($id) {
    $sql = "SELECT p.id, p.reference, p.member_id, p.account_id, p.amount, p.payment_method,
                   p.transaction_id, p.description, p.status, p.processed_by, p.processed_at,
                   p.created_at, p.updated_at, m.first_name, m.last_name, m.member_id as member_ref,
                   a.name as account_name
            FROM payments p
            LEFT JOIN members m ON p.member_id = m.id
            LEFT JOIN accounts a ON p.account_id = a.id
            WHERE p.id = ?";
    
    $payment = fetchSingleRow($sql, [$id]);
    
    if ($payment) {
        http_response_code(200);
        echo json_encode($payment);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Payment not found']);
    }
}

function getMemberPayments($member_id) {
    $sql = "SELECT p.id, p.reference, p.member_id, p.account_id, p.amount, p.payment_method,
                   p.transaction_id, p.description, p.status, p.processed_by, p.processed_at,
                   p.created_at, p.updated_at, m.first_name, m.last_name, m.member_id as member_ref,
                   a.name as account_name
            FROM payments p
            LEFT JOIN members m ON p.member_id = m.id
            LEFT JOIN accounts a ON p.account_id = a.id
            WHERE p.member_id = ?
            ORDER BY p.created_at DESC";
    
    $payments = executeQuery($sql, [$member_id]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($payments),
        'payments' => $payments
    ]);
}

function getAccountPayments($account_id) {
    $sql = "SELECT p.id, p.reference, p.member_id, p.account_id, p.amount, p.payment_method,
                   p.transaction_id, p.description, p.status, p.processed_by, p.processed_at,
                   p.created_at, p.updated_at, m.first_name, m.last_name, m.member_id as member_ref,
                   a.name as account_name
            FROM payments p
            LEFT JOIN members m ON p.member_id = m.id
            LEFT JOIN accounts a ON p.account_id = a.id
            WHERE p.account_id = ?
            ORDER BY p.created_at DESC";
    
    $payments = executeQuery($sql, [$account_id]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($payments),
        'payments' => $payments
    ]);
}

function getPayments() {
    $status_filter = isset($_GET['status']) ? $_GET['status'] : null;
    $method_filter = isset($_GET['method']) ? $_GET['method'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    $sql = "SELECT p.id, p.reference, p.member_id, p.account_id, p.amount, p.payment_method,
                   p.transaction_id, p.description, p.status, p.processed_by, p.processed_at,
                   p.created_at, p.updated_at, m.first_name, m.last_name, m.member_id as member_ref,
                   a.name as account_name
            FROM payments p
            LEFT JOIN members m ON p.member_id = m.id
            LEFT JOIN accounts a ON p.account_id = a.id";
    
    $params = [];
    $conditions = [];
    
    if ($status_filter) {
        $conditions[] = "p.status = ?";
        $params[] = $status_filter;
    }
    
    if ($method_filter) {
        $conditions[] = "p.payment_method = ?";
        $params[] = $method_filter;
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY p.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $payments = executeQuery($sql, $params);
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM payments";
    $count_params = [];
    
    if (!empty($conditions)) {
        $count_sql .= " WHERE " . implode(" AND ", $conditions);
        $count_params = $params;
        // Remove the limit and offset parameters
        array_pop($count_params);
        array_pop($count_params);
    }
    
    $count_result = fetchSingleRow($count_sql, $count_params);
    $total = $count_result['total'];
    
    http_response_code(200);
    echo json_encode([
        'count' => count($payments),
        'total' => $total,
        'offset' => $offset,
        'limit' => $limit,
        'payments' => $payments
    ]);
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['account_id', 'amount', 'payment_method'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Set default values
    if (!isset($input['status'])) {
        $input['status'] = 'Pending';
    }
    
    // Generate reference if not provided
    if (!isset($input['reference']) || empty($input['reference'])) {
        $input['reference'] = generatePaymentReference();
    }
    
    // Insert payment
    $sql = "INSERT INTO payments (reference, member_id, account_id, amount, payment_method, 
            transaction_id, description, status, processed_by, processed_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $params = [
        $input['reference'],
        $input['member_id'] ?? null,
        $input['account_id'],
        $input['amount'],
        $input['payment_method'],
        $input['transaction_id'] ?? null,
        $input['description'] ?? null,
        $input['status'],
        $input['processed_by'] ?? null,
        $input['processed_at'] ?? null
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $payment_id = getDatabaseConnection()->lastInsertId();
        
        // Update account balance if payment is completed
        if ($input['status'] === 'Completed') {
            $account_sql = "UPDATE accounts SET balance = balance + ? WHERE id = ?";
            executeNonQuery($account_sql, [$input['amount'], $input['account_id']]);
        }
        
        commitTransaction();
        
        // Create audit log for payment creation
        createAuditLog(null, 'create', 'payments', $payment_id, null, [
            'reference' => $input['reference'],
            'account_id' => $input['account_id'],
            'amount' => $input['amount'],
            'payment_method' => $input['payment_method'],
            'status' => $input['status']
        ]);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Payment recorded successfully',
            'payment_id' => $payment_id,
            'reference' => $input['reference']
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Payment ID is required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Get current payment status for balance adjustment
    $current_payment = fetchSingleRow("SELECT status, amount, account_id FROM payments WHERE id = ?", [$_GET['id']]);
    
    if (!$current_payment) {
        http_response_code(404);
        echo json_encode(['error' => 'Payment not found']);
        return;
    }
    
    // Update payment
    $sql = "UPDATE payments SET 
            reference = ?, 
            member_id = ?, 
            account_id = ?, 
            amount = ?, 
            payment_method = ?, 
            transaction_id = ?, 
            description = ?, 
            status = ?, 
            processed_by = ?, 
            processed_at = ?
            WHERE id = ?";
    
    $params = [
        $input['reference'] ?? $current_payment['reference'],
        $input['member_id'] ?? $current_payment['member_id'],
        $input['account_id'] ?? $current_payment['account_id'],
        $input['amount'] ?? $current_payment['amount'],
        $input['payment_method'] ?? $current_payment['payment_method'],
        $input['transaction_id'] ?? null,
        $input['description'] ?? null,
        $input['status'] ?? $current_payment['status'],
        $input['processed_by'] ?? null,
        $input['processed_at'] ?? null,
        $_GET['id']
    ];
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $params);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Payment not found']);
            return;
        }
        
        // Update account balance if status changed from/to Completed
        $old_status = $current_payment['status'];
        $new_status = $input['status'] ?? $old_status;
        $amount = $input['amount'] ?? $current_payment['amount'];
        $account_id = $input['account_id'] ?? $current_payment['account_id'];
        
        if ($old_status !== $new_status) {
            if ($old_status === 'Completed') {
                // Remove from balance
                $account_sql = "UPDATE accounts SET balance = balance - ? WHERE id = ?";
                executeNonQuery($account_sql, [$amount, $account_id]);
            }
            if ($new_status === 'Completed') {
                // Add to balance
                $account_sql = "UPDATE accounts SET balance = balance + ? WHERE id = ?";
                executeNonQuery($account_sql, [$amount, $account_id]);
            }
        }
        
        commitTransaction();
        
        // Create audit log for payment update
        createAuditLog(null, 'update', 'payments', $_GET['id'], $current_payment, $input);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Payment updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Payment ID is required']);
        return;
    }
    
    // Get current payment for balance adjustment
    $current_payment = fetchSingleRow("SELECT status, amount, account_id FROM payments WHERE id = ?", [$_GET['id']]);
    
    if (!$current_payment) {
        http_response_code(404);
        echo json_encode(['error' => 'Payment not found']);
        return;
    }
    
    $sql = "DELETE FROM payments WHERE id = ?";
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, [$_GET['id']]);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Payment not found']);
            return;
        }
        
        // Adjust account balance if payment was completed
        if ($current_payment['status'] === 'Completed') {
            $account_sql = "UPDATE accounts SET balance = balance - ? WHERE id = ?";
            executeNonQuery($account_sql, [$current_payment['amount'], $current_payment['account_id']]);
        }
        
        commitTransaction();
        
        // Create audit log for payment deletion
        createAuditLog(null, 'delete', 'payments', $_GET['id'], $current_payment, null);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Payment deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

// Helper function to generate payment reference
function generatePaymentReference() {
    return 'PAY-' . date('Y') . '-' . str_pad(rand(1, 99999), 5, '0', STR_PAD_LEFT);
}
?>