<?php
// MFT GROUP M-PESA STK Push API
// This API handles M-PESA STK push requests for contributions

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Initialize database connection
global $pdo;
$pdo = getDatabaseConnection();

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handlePostRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handlePostRequest() {
    // Initiate M-PESA STK push
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Required fields
    $required_fields = ['phone_number', 'amount', 'member_id', 'payment_type'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Validate phone number format (accept both 07XXXXXXXX and 2547XXXXXXXX)
    $phoneRegex = '/^(?:2547\d{8}|07\d{8})$/';
    if (!preg_match($phoneRegex, $input['phone_number'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid phone number format. Must be in format 07XXXXXXXX or 2547XXXXXXXX']);
        return;
    }
    
    // Convert 07 format to 2547 format if needed
    $formattedPhoneNumber = $input['phone_number'];
    if (strpos($formattedPhoneNumber, '07') === 0) {
        $formattedPhoneNumber = '254' . substr($formattedPhoneNumber, 1);
    }
    
    // Validate amount
    if (!is_numeric($input['amount']) || $input['amount'] <= 0) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid amount. Must be a positive number']);
        return;
    }
    
    // In a real implementation, this would integrate with the M-PESA API
    // For now, we'll simulate a successful STK push
    
    // Generate a unique request ID
    $requestId = 'STK' . date('YmdHis') . rand(1000, 9999);
    
    // Log the STK request (in a real implementation, you would store this in a database)
    error_log("M-PESA STK Request: " . json_encode([
        'request_id' => $requestId,
        'phone_number' => $formattedPhoneNumber,
        'amount' => $input['amount'],
        'member_id' => $input['member_id'],
        'payment_type' => $input['payment_type'],
        'description' => $input['payment_type'] . ' payment'
    ]));
    
    // Return success response
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'M-PESA STK Push sent successfully. Please check your phone to complete payment.',
        'request_id' => $requestId,
        'phone_number' => $formattedPhoneNumber,
        'amount' => $input['amount'],
        'payment_type' => $input['payment_type']
    ]);
}
?>