<?php
// MFT GROUP Members API
// This API provides member data for the frontend dashboards

// Start output buffering to prevent any accidental output
ob_start();

// Disable error reporting to prevent interference with JSON responses
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Handle preflight requests (OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // Specify the allowed methods
    header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
    // Specify the allowed headers
    header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
    // Specify the allowed origin
    header("Access-Control-Allow-Origin: *");
    // Allow credentials if needed
    header("Access-Control-Allow-Credentials: true");
    // Set max age for preflight request caching
    header("Access-Control-Max-Age: 86400");
    // Exit early for OPTIONS requests
    exit(0);
}

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");
header("Access-Control-Allow-Credentials: true");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Function to check if a column exists in a table
if (!function_exists('columnExists')) {
    function columnExists($table, $column) {
        try {
            $pdo = getDatabaseConnection();
            $stmt = $pdo->query("SHOW COLUMNS FROM `$table` LIKE '$column'");
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            error_log("Error checking column existence: " . $e->getMessage());
            return false;
        }
    }
}

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    // Test database connection first
    $pdo = getDatabaseConnection();
    
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        case 'OPTIONS':
            // Already handled above, but just in case
            exit(0);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
} finally {
    // Flush any output buffer to send JSON response
    if (ob_get_level()) {
        ob_end_flush();
    }
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Check if the ID is numeric (primary key) or a member_id string
        if (is_numeric($_GET['id'])) {
            // Get specific member by primary key ID
            getMember($_GET['id']);
        } else {
            // Get specific member by member_id
            getMemberByMemberId($_GET['id']);
        }
    } else if (isset($_GET['member_id'])) {
        // Get specific member by member_id (new parameter for clarity)
        getMemberByMemberId($_GET['member_id']);
    } else if (isset($_GET['pending'])) {
        // Get pending membership applications
        getPendingApplications();
    } else if (isset($_GET['project_id'])) {
        // Get members assigned to a specific project
        getProjectMembers($_GET['project_id']);
    } else {
        // Get all members with optional filters
        getMembers();
    }
}

function handlePostRequest() {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Check if this is for approving a pending application
    if (isset($_GET['id']) && isset($input['action']) && $input['action'] === 'approve_pending') {
        approvePendingApplication($_GET['id'], $input);
        return;
    }
    
    // Check if this is for updating an existing member
    if (isset($_GET['id'])) {
        updateMember($_GET['id'], $input);
        return;
    }
    
    // Otherwise, this is for creating a new member
    addMember($input);
}

function addMember($data) {
    try {
        // Validate required fields
        if (empty($data['first_name']) || empty($data['last_name']) || empty($data['id_number'])) {
            http_response_code(400);
            echo json_encode(['error' => 'First name, last name, and ID number are required']);
            return;
        }
        
        // Generate member ID in the format MFT + ID Number
        $member_id = 'MFT' . $data['id_number'];
        
        // Set default values
        $registration_date = $data['registration_date'] ?? date('Y-m-d');
        $status = $data['status'] ?? 'Active'; // Set to Active by default
        $membership_type = $data['membership_type'] ?? 'Regular';
        
        // Prepare data for insertion
        $insert_data = array_merge([
            'member_id' => $member_id,
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'id_number' => $data['id_number'],
            'registration_date' => $registration_date,
            'status' => $status,
            'membership_type' => $membership_type
        ], $data);
        
        // Build INSERT query with only existing columns
        $columns = [
            'member_id', 'first_name', 'last_name', 'id_number', 
            'registration_date', 'status', 'membership_type'
        ];
        
        $values = [
            $insert_data['member_id'],
            $insert_data['first_name'],
            $insert_data['last_name'],
            $insert_data['id_number'],
            $insert_data['registration_date'],
            $insert_data['status'],
            $insert_data['membership_type']
        ];
        
        // Check for additional columns that exist in the table
        $optional_columns = [
            'middle_name' => 'middle_name',
            'email' => 'email',
            'phone' => 'phone',
            'occupation' => 'occupation',
            'date_of_birth' => 'date_of_birth',
            'gender' => 'gender',
            'address' => 'address',
            'home_address' => 'home_address',
            'current_address' => 'current_address',
            'next_of_kin_name' => 'next_of_kin_name',
            'next_of_kin_phone' => 'next_of_kin_phone',
            'next_of_kin_id' => 'next_of_kin_id'
        ];
        
        foreach ($optional_columns as $key => $column) {
            if (isset($insert_data[$key]) && columnExists('members', $column)) {
                $columns[] = $column;
                $values[] = $insert_data[$key];
            }
        }
        
        // Create placeholders for the prepared statement
        $placeholders = str_repeat('?,', count($columns) - 1) . '?';
        $columns_str = implode(', ', $columns);
        
        $sql = "INSERT INTO members ($columns_str) VALUES ($placeholders)";
        
        beginTransaction();
        
        $result = executeNonQuery($sql, $values);
        $new_member_id = getDatabaseConnection()->lastInsertId();
        
        // Assign role to member
        $role = $insert_data['role'] ?? 'Member';
        
        // Get role ID from roles table
        $role_sql = "SELECT id FROM roles WHERE name = ?";
        $role_stmt = executeQuery($role_sql, [$role]);
        
        if (!empty($role_stmt)) {
            $role_id = $role_stmt[0]['id'];
            
            // Insert into member_roles table
            $member_role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
            executeNonQuery($member_role_sql, [$new_member_id, $role_id, $new_member_id]);
        } else {
            // If role not found, assign default Member role
            $default_role_sql = "SELECT id FROM roles WHERE name = 'Member'";
            $default_role_stmt = executeQuery($default_role_sql, []);
            
            if (!empty($default_role_stmt)) {
                $default_role_id = $default_role_stmt[0]['id'];
                $member_role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
                executeNonQuery($member_role_sql, [$new_member_id, $default_role_id, $new_member_id]);
            }
        }
        
        commitTransaction();
        
        // Return the newly created member
        $member = fetchSingleRow("SELECT * FROM members WHERE id = ?", [$new_member_id]);
        
        // Create audit log for member creation
        createAuditLog(null, 'create', 'members', $new_member_id, null, $member);
        
        http_response_code(201);
        echo json_encode([
            'success' => true,
            'message' => 'Member added successfully',
            'member' => $member
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        $errorMessage = $e->getMessage();
        if (strpos($errorMessage, 'Duplicate entry') !== false && strpos($errorMessage, 'member_id') !== false) {
            http_response_code(409);
            echo json_encode([
                'success' => false,
                'error' => 'A member with this ID number is already registered in the system. Please check the ID number and try again.'
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to add member: ' . $errorMessage
            ]);
        }
    }
}

function updateMember($id, $data) {
    try {
        // Check if member exists
        $existing_member = fetchSingleRow("SELECT * FROM members WHERE id = ?", [$id]);
        if (!$existing_member) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Build UPDATE query with only existing columns
        $updates = [];
        $params = [];
        
        // Check for columns that exist in the table
        $updatable_columns = [
            'first_name' => 'first_name',
            'middle_name' => 'middle_name',
            'last_name' => 'last_name',
            'email' => 'email',
            'phone' => 'phone',
            'id_number' => 'id_number',
            'occupation' => 'occupation',
            'date_of_birth' => 'date_of_birth',
            'gender' => 'gender',
            'address' => 'address',
            'home_address' => 'home_address',
            'current_address' => 'current_address',
            'status' => 'status',
            'membership_type' => 'membership_type',
            'next_of_kin_name' => 'next_of_kin_name',
            'next_of_kin_phone' => 'next_of_kin_phone',
            'next_of_kin_id' => 'next_of_kin_id'
        ];
        
        foreach ($updatable_columns as $key => $column) {
            if (isset($data[$key]) && columnExists('members', $column)) {
                $updates[] = "$column = ?";
                $params[] = $data[$key];
            }
        }
        
        // If no updatable fields, return early
        if (empty($updates)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        // Add ID to parameters
        $params[] = $id;
        
        $updates_str = implode(', ', $updates);
        $sql = "UPDATE members SET $updates_str WHERE id = ?";
        
        $result = executeNonQuery($sql, $params);
        
        // Handle role update if provided
        if (isset($data['role'])) {
            // Get role ID from roles table
            $role_sql = "SELECT id FROM roles WHERE name = ?";
            $role_stmt = executeQuery($role_sql, [$data['role']]);
            
            if (!empty($role_stmt)) {
                $role_id = $role_stmt[0]['id'];
                
                // Check if member already has a role
                $existing_role = fetchSingleRow("SELECT * FROM member_roles WHERE member_id = ?", [$id]);
                
                if ($existing_role) {
                    // Update existing role
                    $update_role_sql = "UPDATE member_roles SET role_id = ?, assigned_by = ? WHERE member_id = ?";
                    executeNonQuery($update_role_sql, [$role_id, $id, $id]);
                } else {
                    // Insert new role
                    $insert_role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
                    executeNonQuery($insert_role_sql, [$id, $role_id, $id]);
                }
            }
        }
        
        // If status changed to Active and member doesn't have a role, assign default member role
        if (isset($data['status']) && $data['status'] === 'Active') {
            // Check if member already has a role
            $role_check = fetchSingleRow("SELECT * FROM member_roles WHERE member_id = ?", [$id]);
            if (!$role_check) {
                $role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) 
                             SELECT ?, r.id, ? 
                             FROM roles r 
                             WHERE r.name = 'Member'";
                executeNonQuery($role_sql, [$id, $id]);
            }
        }
        
        // Return the updated member
        $member = fetchSingleRow("SELECT * FROM members WHERE id = ?", [$id]);
        
        // Get member's role
        $role_sql = "SELECT r.name as role FROM member_roles mr 
                     JOIN roles r ON mr.role_id = r.id 
                     WHERE mr.member_id = ? 
                     LIMIT 1";
        $role_result = fetchSingleRow($role_sql, [$id]);
        
        if ($role_result) {
            $member['role'] = $role_result['role'];
        } else {
            $member['role'] = 'Member'; // Default role
        }
        
        // Create audit log for member update
        createAuditLog(null, 'update', 'members', $id, $existing_member, $member);
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Member updated successfully',
            'member' => $member
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to update member: ' . $e->getMessage()
        ]);
    }
}

function approvePendingApplication($id, $data) {
    try {
        // Check if member exists and is pending
        $existing_member = fetchSingleRow("SELECT * FROM members WHERE id = ? AND status = 'Pending'", [$id]);
        if (!$existing_member) {
            http_response_code(404);
            echo json_encode(['error' => 'Pending member application not found']);
            return;
        }
        
        // Update member status to Active
        $status = $data['status'] ?? 'Active';
        $sql = "UPDATE members SET status = ? WHERE id = ?";
        executeNonQuery($sql, [$status, $id]);
        
        // Assign role based on input data or default to 'Member'
        $role_name = $data['role'] ?? 'Member';
        
        // Get role ID from roles table
        $role_sql = "SELECT id FROM roles WHERE name = ?";
        $role_stmt = executeQuery($role_sql, [$role_name]);
        
        if (!empty($role_stmt)) {
            $role_id = $role_stmt[0]['id'];
            
            // Insert into member_roles table
            $member_role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
            executeNonQuery($member_role_sql, [$id, $role_id, $id]);
        } else {
            // If role not found, assign default Member role
            $default_role_sql = "SELECT id FROM roles WHERE name = 'Member'";
            $default_role_stmt = executeQuery($default_role_sql, []);
            
            if (!empty($default_role_stmt)) {
                $default_role_id = $default_role_stmt[0]['id'];
                $member_role_sql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
                executeNonQuery($member_role_sql, [$id, $default_role_id, $id]);
            }
        }
        
        // Return the updated member
        $member = fetchSingleRow("SELECT * FROM members WHERE id = ?", [$id]);
        
        // Create audit log for member approval
        createAuditLog(null, 'approve', 'members', $id, $existing_member, $member);
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Application approved successfully',
            'member' => $member
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to approve application: ' . $e->getMessage()
        ]);
    }
}

function handlePutRequest() {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Check if ID is provided
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    updateMember($_GET['id'], $input);
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    deleteMember($_GET['id']);
}

function deleteMember($id) {
    try {
        // Check if member exists
        $existing_member = fetchSingleRow("SELECT * FROM members WHERE id = ?", [$id]);
        if (!$existing_member) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        beginTransaction();
        
        // Delete related records in other tables first (foreign key constraints)
        // Delete member projects (must be done before deleting projects or members)
        executeNonQuery("DELETE FROM member_projects WHERE member_id = ?", [$id]);
        
        // Delete member roles
        executeNonQuery("DELETE FROM member_roles WHERE member_id = ?", [$id]);
        
        // Handle contributions:
        // 1. Set confirmed_by to NULL for contributions confirmed by this member
        executeNonQuery("UPDATE contributions SET confirmed_by = NULL WHERE confirmed_by = ?", [$id]);
        
        // 2. Delete contributions and their payments
        // First, get all payment IDs associated with this member's contributions
        $contributionPayments = executeQuery("SELECT payment_id FROM contributions WHERE member_id = ?", [$id]);
        foreach ($contributionPayments as $payment) {
            if ($payment['payment_id']) {
                executeNonQuery("DELETE FROM payments WHERE id = ?", [$payment['payment_id']]);
            }
        }
        // Then delete the contributions themselves
        executeNonQuery("DELETE FROM contributions WHERE member_id = ?", [$id]);
        
        // Delete loans and their repayments
        // First, delete the loan repayments (no payment_id in actual table structure)
        executeNonQuery("DELETE FROM loan_repayments WHERE loan_id IN (SELECT id FROM loans WHERE member_id = ?)", [$id]);
        // Then delete the loans themselves
        executeNonQuery("DELETE FROM loans WHERE member_id = ?", [$id]);
        
        // Delete dividends
        executeNonQuery("DELETE FROM dividends WHERE member_id = ?", [$id]);
        
        // Delete fines (need to handle both member_id and created_by references)
        executeNonQuery("DELETE FROM fines WHERE member_id = ? OR created_by = ?", [$id, $id]);
        
        // Delete documents (need to handle both related_member_id and uploaded_by references)
        executeNonQuery("DELETE FROM documents WHERE related_member_id = ? OR uploaded_by = ?", [$id, $id]);
        
        // Delete meetings (need to handle created_by reference)
        executeNonQuery("DELETE FROM meetings WHERE created_by = ?", [$id]);
        
        // Delete member_projects records where this member is the project manager
        // This handles the case where projects reference members as managers
        executeNonQuery("DELETE FROM member_projects WHERE project_id IN (SELECT id FROM projects WHERE manager_id = ?)", [$id]);
        
        // Delete projects (need to handle manager_id and created_by references)
        executeNonQuery("DELETE FROM projects WHERE manager_id = ? OR created_by = ?", [$id, $id]);
        
        // Delete audit logs
        executeNonQuery("DELETE FROM audit_logs WHERE user_id = ?", [$id]);
        
        // Finally, delete the member
        $sql = "DELETE FROM members WHERE id = ?";
        $result = executeNonQuery($sql, [$id]);
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Member deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to delete member: ' . $e->getMessage()
        ]);
    }
}

function getMember($id) {
    try {
        // Build SELECT clause with only existing columns
        $base_columns = "id, member_id, first_name, last_name, email, phone, id_number, 
                        date_of_birth, gender, address, registration_date, status, membership_type";
        
        // Check for additional columns
        $additional_columns = [];
        
        if (columnExists('members', 'middle_name')) {
            $additional_columns[] = 'middle_name';
        }
        
        if (columnExists('members', 'occupation')) {
            $additional_columns[] = 'occupation';
        }
        
        if (columnExists('members', 'home_address')) {
            $additional_columns[] = 'home_address';
        }
        
        if (columnExists('members', 'current_address')) {
            $additional_columns[] = 'current_address';
        }
        
        if (columnExists('members', 'next_of_kin_name')) {
            $additional_columns[] = 'next_of_kin_name';
        }
        
        if (columnExists('members', 'next_of_kin_phone')) {
            $additional_columns[] = 'next_of_kin_phone';
        }
        
        if (columnExists('members', 'next_of_kin_id')) {
            $additional_columns[] = 'next_of_kin_id';
        }
        
        // Check for document columns
        if (columnExists('members', 'front_id_document')) {
            $additional_columns[] = 'front_id_document';
        }
        
        if (columnExists('members', 'back_id_document')) {
            $additional_columns[] = 'back_id_document';
        }
        
        if (columnExists('members', 'passport_image')) {
            $additional_columns[] = 'passport_image';
        }
        
        // Check for profile picture column
        if (columnExists('members', 'profile_picture')) {
            $additional_columns[] = 'profile_picture';
        }
        
        // Combine columns
        $all_columns = $base_columns;
        if (!empty($additional_columns)) {
            $all_columns .= ', ' . implode(', ', $additional_columns);
        }
        
        $sql = "SELECT $all_columns,
                       last_login, last_login_ip, created_at, updated_at
                FROM members 
                WHERE id = ?";
        
        $member = fetchSingleRow($sql, [$id]);
        
        if ($member) {
            // Get member's role
            $role_sql = "SELECT r.name as role FROM member_roles mr 
                         JOIN roles r ON mr.role_id = r.id 
                         WHERE mr.member_id = ? 
                         LIMIT 1";
            $role_result = fetchSingleRow($role_sql, [$id]);
            
            if ($role_result) {
                $member['role'] = $role_result['role'];
            } else {
                $member['role'] = 'Member'; // Default role
            }
            
            http_response_code(200);
            echo json_encode($member);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
        }
    } catch (Exception $e) {
        error_log("Error in getMember: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve member: ' . $e->getMessage()]);
    }
}

function getMemberByMemberId($member_id) {
    // Build SELECT clause with only existing columns
    $base_columns = "id, member_id, first_name, last_name, email, phone, id_number, 
                    date_of_birth, gender, address, registration_date, status, membership_type";
    
    // Check for additional columns
    $additional_columns = [];
    
    if (columnExists('members', 'middle_name')) {
        $additional_columns[] = 'middle_name';
    }
    
    if (columnExists('members', 'occupation')) {
        $additional_columns[] = 'occupation';
    }
    
    if (columnExists('members', 'home_address')) {
        $additional_columns[] = 'home_address';
    }
    
    if (columnExists('members', 'current_address')) {
        $additional_columns[] = 'current_address';
    }
    
    if (columnExists('members', 'next_of_kin_name')) {
        $additional_columns[] = 'next_of_kin_name';
    }
    
    if (columnExists('members', 'next_of_kin_phone')) {
        $additional_columns[] = 'next_of_kin_phone';
    }
    
    if (columnExists('members', 'next_of_kin_id')) {
        $additional_columns[] = 'next_of_kin_id';
    }
    
    // Check for document columns
    if (columnExists('members', 'front_id_document')) {
        $additional_columns[] = 'front_id_document';
    }
    
    if (columnExists('members', 'back_id_document')) {
        $additional_columns[] = 'back_id_document';
    }
    
    if (columnExists('members', 'passport_image')) {
        $additional_columns[] = 'passport_image';
    }
    
    // Check for profile picture column
    if (columnExists('members', 'profile_picture')) {
        $additional_columns[] = 'profile_picture';
    }
    
    // Combine columns
    $all_columns = $base_columns;
    if (!empty($additional_columns)) {
        $all_columns .= ', ' . implode(', ', $additional_columns);
    }
    
    $sql = "SELECT $all_columns,
                   last_login, last_login_ip, created_at, updated_at
            FROM members 
            WHERE member_id = ?";
    
    $member = fetchSingleRow($sql, [$member_id]);
    
    if ($member) {
        // Get member's role
        $role_sql = "SELECT r.name as role FROM member_roles mr 
                     JOIN roles r ON mr.role_id = r.id 
                     WHERE mr.member_id = ? 
                     LIMIT 1";
        $role_result = fetchSingleRow($role_sql, [$member['id']]);
        
        if ($role_result) {
            $member['role'] = $role_result['role'];
        } else {
            $member['role'] = 'Member'; // Default role
        }
        
        // Add contribution balance to the member data
        $contribution_balance = getMemberContributionBalance($member['id']);
        $member['contribution_balance'] = $contribution_balance;
        
        // Add active loans count
        $active_loans = getMemberActiveLoansCount($member['id']);
        $member['active_loans'] = $active_loans;
        
        http_response_code(200);
        echo json_encode($member);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Member not found']);
    }
}

// Function to get member's contribution balance
function getMemberContributionBalance($member_id) {
    try {
        $sql = "SELECT COALESCE(SUM(amount), 0) as total_balance
                FROM contributions 
                WHERE member_id = ? AND status = 'Confirmed'";
        $result = fetchSingleRow($sql, [$member_id]);
        return $result ? (float)$result['total_balance'] : 0;
    } catch (Exception $e) {
        error_log("Error getting member contribution balance: " . $e->getMessage());
        return 0;
    }
}

// Function to get member's active loans count
function getMemberActiveLoansCount($member_id) {
    try {
        $sql = "SELECT COUNT(*) as active_loans
                FROM loans 
                WHERE member_id = ? AND status = 'Active'";
        $result = fetchSingleRow($sql, [$member_id]);
        return $result ? (int)$result['active_loans'] : 0;
    } catch (Exception $e) {
        error_log("Error getting member active loans count: " . $e->getMessage());
        return 0;
    }
}

function getProjectMembers($project_id) {
    try {
        // Build SELECT clause with only existing columns
        $base_columns = "m.id, m.member_id, m.first_name, m.last_name, m.email, m.phone, m.id_number, 
                        m.date_of_birth, m.gender, m.address, m.registration_date, m.status, m.membership_type";
        
        // Check for additional columns
        $additional_columns = [];
        
        if (columnExists('members', 'middle_name')) {
            $additional_columns[] = 'middle_name';
        }
        
        if (columnExists('members', 'occupation')) {
            $additional_columns[] = 'occupation';
        }
        
        if (columnExists('members', 'home_address')) {
            $additional_columns[] = 'home_address';
        }
        
        if (columnExists('members', 'current_address')) {
            $additional_columns[] = 'current_address';
        }
        
        if (columnExists('members', 'next_of_kin_name')) {
            $additional_columns[] = 'next_of_kin_name';
        }
        
        if (columnExists('members', 'next_of_kin_phone')) {
            $additional_columns[] = 'next_of_kin_phone';
        }
        
        if (columnExists('members', 'next_of_kin_id')) {
            $additional_columns[] = 'next_of_kin_id';
        }
        
        // Check for document columns
        if (columnExists('members', 'front_id_document')) {
            $additional_columns[] = 'front_id_document';
        }
        
        if (columnExists('members', 'back_id_document')) {
            $additional_columns[] = 'back_id_document';
        }
        
        if (columnExists('members', 'passport_image')) {
            $additional_columns[] = 'passport_image';
        }
        
        // Check for profile picture column
        if (columnExists('members', 'profile_picture')) {
            $additional_columns[] = 'profile_picture';
        }
        
        // Check for profile picture column
        if (columnExists('members', 'profile_picture')) {
            $additional_columns[] = 'profile_picture';
        }
        
        // Combine columns
        $all_columns = $base_columns;
        if (!empty($additional_columns)) {
            $all_columns .= ', ' . implode(', ', $additional_columns);
        }
        
        // Add member_projects columns (using correct column names from actual database)
        $all_columns .= ", mp.role, mp.contribution_amount, mp.joined_date, mp.status as member_project_status";
        
        $sql = "SELECT $all_columns,
                       m.created_at, m.updated_at
                FROM members m
                JOIN member_projects mp ON m.id = mp.member_id
                WHERE mp.project_id = ?
                ORDER BY mp.joined_date DESC";
        
        $members = executeQuery($sql, [$project_id]);
        
        http_response_code(200);
        echo json_encode([
            'count' => count($members),
            'members' => $members
        ]);
    } catch (Exception $e) {
        error_log("Error in getProjectMembers: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve project members: ' . $e->getMessage()]);
    }
}

function getMembers() {
    try {
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        
        // Get filter parameters
        $status = isset($_GET['status']) ? $_GET['status'] : null;
        $search = isset($_GET['search']) ? $_GET['search'] : null;
        
        // Build WHERE clause
        $where_conditions = ["status != 'Pending'"];
        $params = [];
        
        if ($status && $status !== 'all') {
            $where_conditions[] = "status = ?";
            $params[] = $status;
        }
        
        if ($search && $search !== 'all') {
            $where_conditions[] = "(first_name LIKE ? OR last_name LIKE ? OR member_id LIKE ?)";
            $search_param = '%' . $search . '%';
            $params[] = $search_param;
            $params[] = $search_param;
            $params[] = $search_param;
        }
        
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        
        // Build SELECT clause with only existing columns
        $base_columns = "id, member_id, first_name, last_name, email, phone, id_number, 
                        date_of_birth, gender, address, registration_date, status, membership_type";
        
        // Check for additional columns
        $additional_columns = [];
        
        if (columnExists('members', 'middle_name')) {
            $additional_columns[] = 'middle_name';
        }
        
        if (columnExists('members', 'occupation')) {
            $additional_columns[] = 'occupation';
        }
        
        if (columnExists('members', 'home_address')) {
            $additional_columns[] = 'home_address';
        }
        
        if (columnExists('members', 'current_address')) {
            $additional_columns[] = 'current_address';
        }
        
        if (columnExists('members', 'next_of_kin_name')) {
            $additional_columns[] = 'next_of_kin_name';
        }
        
        if (columnExists('members', 'next_of_kin_phone')) {
            $additional_columns[] = 'next_of_kin_phone';
        }
        
        if (columnExists('members', 'next_of_kin_id')) {
            $additional_columns[] = 'next_of_kin_id';
        }
        
        // Check for document columns
        if (columnExists('members', 'front_id_document')) {
            $additional_columns[] = 'front_id_document';
        }
        
        if (columnExists('members', 'back_id_document')) {
            $additional_columns[] = 'back_id_document';
        }
        
        if (columnExists('members', 'passport_image')) {
            $additional_columns[] = 'passport_image';
        }
        
        // Check for profile picture column
        if (columnExists('members', 'profile_picture')) {
            $additional_columns[] = 'profile_picture';
        }
        
        // Combine columns
        $all_columns = $base_columns;
        if (!empty($additional_columns)) {
            $all_columns .= ', ' . implode(', ', $additional_columns);
        }
        
        // Get all members except pending applications with pagination and filters
        $sql = "SELECT $all_columns,
                       created_at, updated_at
                FROM members 
                $where_clause
                ORDER BY registration_date DESC
                LIMIT ? OFFSET ?";
        
        // Add limit and offset to parameters
        $params[] = $limit;
        $params[] = $offset;
        
        error_log("Executing members query: " . $sql);
        error_log("With parameters: " . json_encode($params));
        
        $members = executeQuery($sql, $params);
        
        // Get total count
        $count_sql = "SELECT COUNT(*) as total FROM members $where_clause";
        // For count query, we only need the filter parameters, not limit/offset
        $count_params = array_slice($params, 0, count($params) - 2);
        error_log("Executing count query: " . $count_sql);
        error_log("With parameters: " . json_encode($count_params));
        $count_result = fetchSingleRow($count_sql, $count_params);
        $total = $count_result ? $count_result['total'] : 0;
        
        http_response_code(200);
        echo json_encode([
            'count' => count($members),
            'total' => $total,
            'offset' => $offset,
            'limit' => $limit,
            'members' => $members
        ]);
    } catch (Exception $e) {
        error_log("Error in getMembers: " . $e->getMessage());
        error_log("Error trace: " . $e->getTraceAsString());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve members: ' . $e->getMessage()]);
    }
}

function getPendingApplications() {
    try {
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        
        // Build SELECT clause with only existing columns
        $base_columns = "id, member_id, first_name, last_name, email, phone, id_number, 
                        date_of_birth, gender, address, registration_date, status, membership_type";
        
        // Check for additional columns
        $additional_columns = [];
        
        if (columnExists('members', 'middle_name')) {
            $additional_columns[] = 'middle_name';
        }
        
        if (columnExists('members', 'occupation')) {
            $additional_columns[] = 'occupation';
        }
        
        if (columnExists('members', 'home_address')) {
            $additional_columns[] = 'home_address';
        }
        
        if (columnExists('members', 'current_address')) {
            $additional_columns[] = 'current_address';
        }
        
        if (columnExists('members', 'next_of_kin_name')) {
            $additional_columns[] = 'next_of_kin_name';
        }
        
        if (columnExists('members', 'next_of_kin_phone')) {
            $additional_columns[] = 'next_of_kin_phone';
        }
        
        if (columnExists('members', 'next_of_kin_id')) {
            $additional_columns[] = 'next_of_kin_id';
        }
        
        // Check for document columns
        if (columnExists('members', 'front_id_document')) {
            $additional_columns[] = 'front_id_document';
        }
        
        if (columnExists('members', 'back_id_document')) {
            $additional_columns[] = 'back_id_document';
        }
        
        if (columnExists('members', 'passport_image')) {
            $additional_columns[] = 'passport_image';
        }
        
        // Check for profile picture column
        if (columnExists('members', 'profile_picture')) {
            $additional_columns[] = 'profile_picture';
        }
        
        // Combine columns
        $all_columns = $base_columns;
        if (!empty($additional_columns)) {
            $all_columns .= ', ' . implode(', ', $additional_columns);
        }
        
        // Get only pending membership applications with pagination
        $sql = "SELECT $all_columns,
                       created_at, updated_at
                FROM members 
                WHERE status = 'Pending'
                ORDER BY created_at DESC
                LIMIT ? OFFSET ?";
        
        $members = executeQuery($sql, [$limit, $offset]);
        
        // Get total count
        $count_sql = "SELECT COUNT(*) as total FROM members WHERE status = 'Pending'";
        $count_result = fetchSingleRow($count_sql);
        $total = $count_result ? $count_result['total'] : 0;
        
        http_response_code(200);
        echo json_encode([
            'count' => count($members),
            'total' => $total,
            'offset' => $offset,
            'limit' => $limit,
            'applications' => $members
        ]);
    } catch (Exception $e) {
        error_log("Error in getPendingApplications: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve pending applications: ' . $e->getMessage()]);
    }
}