<?php
// MFT GROUP Member Login API
// This API handles member login and updates last login time and IP address

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

if ($method !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get input data
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['member_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid input data']);
    exit;
}

$member_id = $input['member_id'];

try {
    // Verify member exists and is active
    $sql = "SELECT id, member_id, first_name, last_name, status 
            FROM members 
            WHERE member_id = ? AND status = 'Active'";
    
    $member = fetchSingleRow($sql, [$member_id]);
    
    if (!$member) {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid membership number or account not active']);
        exit;
    }
    
    // Get client IP address
    $client_ip = getClientIPAddress();
    
    // Update last login time and IP address
    $update_sql = "UPDATE members 
                   SET last_login = NOW(), last_login_ip = ? 
                   WHERE id = ?";
    
    executeNonQuery($update_sql, [$client_ip, $member['id']]);
    
    // Return success response with member data
    http_response_code(200);
    echo json_encode([
        'message' => 'Login successful',
        'member' => $member,
        'last_login' => date('Y-m-d H:i:s'),
        'last_login_ip' => $client_ip
    ]);
    
} catch (Exception $e) {
    error_log("Member login error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

// Function to get client IP address
function getClientIPAddress() {
    // Check for various headers that might contain the IP
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        // Check if multiple IPs exist (comma separated)
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        $ip = trim($ips[0]);
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED'];
    } elseif (!empty($_SERVER['HTTP_X_CLUSTER_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_X_CLUSTER_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_FORWARDED_FOR'];
    } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
        $ip = $_SERVER['HTTP_FORWARDED'];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    
    // Validate IP address
    if (filter_var($ip, FILTER_VALIDATE_IP)) {
        return $ip;
    }
    
    return 'Unknown';
}
?>