<?php
// MFT GROUP Member Profile API
// This API provides member profile data for the frontend

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'PUT':
        case 'POST':
            handlePutRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (!isset($_GET['member_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    $member_id = $_GET['member_id'];
    
    // Get member profile data
    getMemberProfile($member_id);
}

function handlePutRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    if (!isset($input['member_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Member ID is required']);
        return;
    }
    
    // Update member profile
    updateMemberProfile($input);
}

function getMemberProfile($member_id) {
    try {
        $pdo = getDatabaseConnection();
        
        // Get member data
        $sql = "SELECT id, member_id, first_name, middle_name, last_name, email, phone, id_number, 
                       occupation, date_of_birth, gender, address, home_address, current_address,
                       registration_date, status, membership_type, profile_picture,
                       next_of_kin_name, next_of_kin_phone, next_of_kin_id, role,
                       signature_image, created_at, updated_at
                FROM members 
                WHERE member_id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$member_id]);
        $member = $stmt->fetch();
        
        if (!$member) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Get next of kin data
        $next_of_kin = [
            'name' => $member['next_of_kin_name'] ?? '',
            'phone' => $member['next_of_kin_phone'] ?? '',
            'id' => $member['next_of_kin_id'] ?? ''
        ];
        
        // Get member roles
        $roles_sql = "SELECT r.name, r.description FROM member_roles mr 
                      JOIN roles r ON mr.role_id = r.id 
                      WHERE mr.member_id = ?";
        $roles_stmt = $pdo->prepare($roles_sql);
        $roles_stmt->execute([$member['id']]);
        $roles = $roles_stmt->fetchAll();
        
        http_response_code(200);
        echo json_encode([
            'member' => $member,
            'next_of_kin' => $next_of_kin,
            'roles' => $roles
        ]);
    } catch (Exception $e) {
        throw $e;
    }
}

function updateMemberProfile($input) {
    try {
        $pdo = getDatabaseConnection();
        
        // Begin transaction
        $pdo->beginTransaction();
        
        // Update member data
        $sql = "UPDATE members SET 
                first_name = ?, 
                middle_name = ?, 
                last_name = ?, 
                email = ?, 
                phone = ?, 
                address = ?,
                home_address = ?,
                current_address = ?,
                updated_at = CURRENT_TIMESTAMP";
        
        $params = [
            $input['first_name'] ?? '',
            $input['middle_name'] ?? '',
            $input['last_name'] ?? '',
            $input['email'] ?? '',
            $input['phone'] ?? '',
            $input['address'] ?? '',
            $input['home_address'] ?? '',
            $input['current_address'] ?? ''
        ];
        
        // Add role update if provided
        if (isset($input['role'])) {
            $sql .= ", role = ?";
            $params[] = $input['role'];
        }
        
        // Add signature update if provided
        if (isset($input['signature_image'])) {
            $sql .= ", signature_image = ?";
            $params[] = $input['signature_image'];
        }
        
        $sql .= " WHERE member_id = ?";
        $params[] = $input['member_id'];
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        // Update next of kin data
        $nextOfKinSql = "UPDATE members SET 
                        next_of_kin_name = ?, 
                        next_of_kin_phone = ?, 
                        next_of_kin_id = ?
                        WHERE member_id = ?";
        
        $nextOfKinStmt = $pdo->prepare($nextOfKinSql);
        $nextOfKinStmt->execute([
            $input['next_of_kin_name'] ?? '',
            $input['next_of_kin_phone'] ?? '',
            $input['next_of_kin_id'] ?? '',
            $input['member_id']
        ]);
        
        // Commit transaction
        $pdo->commit();
        
        http_response_code(200);
        echo json_encode(['message' => 'Profile updated successfully']);
    } catch (Exception $e) {
        $pdo->rollback();
        throw $e;
    }
}
?>