<?php
// MFT GROUP Loans API
// This API handles loan applications and related operations

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Initialize database connection
global $pdo;
$pdo = getDatabaseConnection();

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    // Get query parameters
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $loan_type = isset($_GET['loan_type']) ? $_GET['loan_type'] : null;
    
    if ($id) {
        // Get specific loan by ID
        getLoanById($id);
    } else if ($member_id) {
        // Get loans for a specific member
        getLoansByMember($member_id, $limit, $offset, $status);
    } else {
        // Get all loans with filters
        getAllLoans($limit, $offset, $status, $loan_type);
    }
}

function handlePostRequest() {
    // Add a new loan
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    addLoan($input);
}

function handlePutRequest() {
    // Update a loan
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing ID parameter']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    updateLoan($id, $input);
}

function handleDeleteRequest() {
    // Delete a loan
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing ID parameter']);
        return;
    }
    
    deleteLoan($id);
}

function getLoanById($id) {
    global $pdo;
    
    try {
        $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                FROM loans l
                JOIN members m ON l.member_id = m.id
                WHERE l.id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$id]);
        $loan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($loan) {
            echo json_encode($loan);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Loan not found']);
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function getLoansByMember($member_id, $limit, $offset, $status = null) {
    global $pdo;
    
    try {
        // First, get the member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$member_id]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Build the query with filters
        $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                FROM loans l
                JOIN members m ON l.member_id = m.id
                WHERE l.member_id = ?";
        
        $params = [$member_numeric_id];
        
        // Add status filter
        if ($status) {
            $sql .= " AND l.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY l.application_date DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total FROM loans WHERE member_id = ?";
        $count_params = [$member_numeric_id];
        
        if ($status) {
            $count_sql .= " AND status = ?";
            $count_params[] = $status;
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'loans' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function getAllLoans($limit, $offset, $status = null, $loan_type = null) {
    global $pdo;
    
    try {
        // Build the query with filters
        $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                FROM loans l
                JOIN members m ON l.member_id = m.id";
        
        $where_conditions = [];
        $params = [];
        
        // Add status filter
        if ($status) {
            $where_conditions[] = "l.status = ?";
            $params[] = $status;
        }
        
        // Add loan type filter
        if ($loan_type) {
            $where_conditions[] = "l.loan_type = ?";
            $params[] = $loan_type;
        }
        
        // Add WHERE clause if there are conditions
        if (!empty($where_conditions)) {
            $sql .= " WHERE " . implode(" AND ", $where_conditions);
        }
        
        $sql .= " ORDER BY l.application_date DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination
        $count_sql = "SELECT COUNT(*) as total FROM loans l";
        
        $count_params = [];
        $count_where_conditions = [];
        
        // Add status filter to count query
        if ($status) {
            $count_where_conditions[] = "l.status = ?";
            $count_params[] = $status;
        }
        
        // Add loan type filter to count query
        if ($loan_type) {
            $count_where_conditions[] = "l.loan_type = ?";
            $count_params[] = $loan_type;
        }
        
        // Add WHERE clause to count query if there are conditions
        if (!empty($count_where_conditions)) {
            $count_sql .= " WHERE " . implode(" AND ", $count_where_conditions);
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'loans' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function addLoan($data) {
    global $pdo;
    
    try {
        // Required fields
        $required_fields = ['member_id', 'loan_type', 'principal_amount', 'interest_rate', 'repayment_period', 'purpose', 'application_date'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$data['member_id']]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Calculate derived fields
        $principal_amount = $data['principal_amount'];
        $interest_rate = $data['interest_rate'];
        $repayment_period = $data['repayment_period'];
        
        // Calculate total amount with simple interest
        $total_amount = $principal_amount * (1 + ($interest_rate / 100));
        $monthly_payment = $total_amount / $repayment_period;
        
        // Generate reference
        $reference = "LOAN-" . date('Ymd') . "-" . strtoupper(substr(md5(uniqid()), 0, 6));
        
        // Set default status
        $status = isset($data['status']) ? $data['status'] : 'Pending';
        
        // Insert loan
        $sql = "INSERT INTO loans (reference, member_id, loan_type, principal_amount, interest_rate, total_amount, repayment_period, monthly_payment, purpose, application_date, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $reference,
            $member_numeric_id,
            $data['loan_type'],
            $principal_amount,
            $interest_rate,
            $total_amount,
            $repayment_period,
            $monthly_payment,
            $data['purpose'],
            $data['application_date'],
            $status
        ];
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        if ($result) {
            // Get the inserted record
            $select_sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                           FROM loans l
                           JOIN members m ON l.member_id = m.id
                           WHERE l.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$pdo->lastInsertId()]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(201);
            echo json_encode([
                'message' => 'Loan application submitted successfully',
                'loan' => $record
            ]);
        } else {
            throw new Exception('Failed to add loan');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function updateLoan($id, $data) {
    global $pdo;
    
    try {
        // Check if loan exists
        $check_sql = "SELECT * FROM loans WHERE id = ?";
        $check_stmt = $pdo->prepare($check_sql);
        $check_stmt->execute([$id]);
        $loan = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$loan) {
            http_response_code(404);
            echo json_encode(['error' => 'Loan not found']);
            return;
        }
        
        // Build update query
        $fields = [];
        $params = [];
        
        // Handle specific fields that can be updated
        $updatable_fields = ['status', 'approval_date', 'disbursement_date', 'first_payment_date', 'reviewer_notes'];
        
        foreach ($updatable_fields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        
        // Handle approved_by field separately - convert member reference ID to numeric ID
        if (isset($data['approved_by'])) {
            // Check if approved_by is numeric (direct ID) or string (member reference)
            if (is_numeric($data['approved_by'])) {
                // It's already a numeric ID
                $fields[] = "approved_by = ?";
                $params[] = $data['approved_by'];
            } else {
                // It's a member reference ID (like ADM-2025-001), convert to numeric ID
                $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
                $member_stmt = $pdo->prepare($member_check_sql);
                $member_stmt->execute([$data['approved_by']]);
                $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($member_result) {
                    $fields[] = "approved_by = ?";
                    $params[] = $member_result['id'];
                } else {
                    // If member not found, return error
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid approver ID. Member not found.']);
                    return;
                }
            }
        }
        
        // If no fields to update
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No valid fields to update']);
            return;
        }
        
        // Add loan ID to parameters
        $params[] = $id;
        
        $sql = "UPDATE loans SET " . implode(', ', $fields) . " WHERE id = ?";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute($params);
        
        if ($result) {
            // Check if the status was updated to 'Approved' and generate loan agreement
            if (isset($data['status']) && $data['status'] === 'Approved') {
                // Generate loan agreement when loan is approved
                generateLoanAgreementForLoan($id);
            }
            
            // Get the updated record
            $select_sql = "SELECT l.*, m.first_name, m.last_name, m.member_id as member_ref 
                           FROM loans l
                           JOIN members m ON l.member_id = m.id
                           WHERE l.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$id]);
            $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'message' => 'Loan updated successfully',
                'loan' => $record
            ]);
        } else {
            throw new Exception('Failed to update loan');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteLoan($id) {
    global $pdo;
    
    try {
        // Check if loan exists
        $check_sql = "SELECT * FROM loans WHERE id = ?";
        $check_stmt = $pdo->prepare($check_sql);
        $check_stmt->execute([$id]);
        $loan = $check_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$loan) {
            http_response_code(404);
            echo json_encode(['error' => 'Loan not found']);
            return;
        }
        
        // Check if loan has any repayments
        $repayment_sql = "SELECT COUNT(*) as count FROM loan_repayments WHERE loan_id = ?";
        $repayment_stmt = $pdo->prepare($repayment_sql);
        $repayment_stmt->execute([$id]);
        $repayment_result = $repayment_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($repayment_result['count'] > 0) {
            http_response_code(400);
            echo json_encode(['error' => 'Cannot delete loan with existing repayments']);
            return;
        }
        
        // Delete the loan
        $sql = "DELETE FROM loans WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([$id]);
        
        if ($result) {
            echo json_encode(['message' => 'Loan deleted successfully']);
        } else {
            throw new Exception('Failed to delete loan');
        }
    } catch (Exception $e) {
        throw $e;
    }
}

/**
 * Generate loan agreement for a specific loan
 * 
 * @param int $loanId Loan ID
 * @return void
 */
function generateLoanAgreementForLoan($loanId) {
    global $pdo;
    
    try {
        // Get loan details with member information
        $sql = "SELECT l.*, m.first_name, m.last_name, m.member_id, m.id_number, m.phone, m.address, m.signature
                FROM loans l
                JOIN members m ON l.member_id = m.id
                WHERE l.id = ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$loanId]);
        $loan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$loan) {
            error_log("Loan not found for agreement generation: " . $loanId);
            return;
        }
        
        // Get system settings for lender information
        $settings_sql = "SELECT setting_key, setting_value FROM system_settings";
        $settings_stmt = $pdo->prepare($settings_sql);
        $settings_stmt->execute();
        $settings_rows = $settings_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $settings = [];
        foreach ($settings_rows as $row) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        // Prepare loan data for agreement generation
        $loanData = [
            'loanRef' => $loan['reference'],
            'agreementDate' => date('F j, Y'),
            'lenderName' => $settings['organizationName'] ?? 'MFT SELF HELP GROUP',
            'lenderRegNo' => $settings['registrationNumber'] ?? 'REG-2025-001',
            'lenderAddress' => $settings['organizationAddress'] ?? 'Nairobi, Kenya',
            'borrowerName' => $loan['first_name'] . ' ' . $loan['last_name'],
            'borrowerMemberNo' => $loan['member_id'],
            'borrowerId' => $loan['id_number'],
            'borrowerAddress' => $loan['address'],
            'borrowerPhone' => $loan['phone'],
            'loanType' => $loan['loan_type'],
            'loanPurpose' => $loan['purpose'],
            'principalAmount' => $loan['principal_amount'],
            'interestRate' => $loan['interest_rate'],
            'totalAmount' => $loan['total_amount'],
            'repaymentPeriod' => $loan['repayment_period'],
            'monthlyPayment' => $loan['monthly_payment'],
            'applicationDate' => $loan['application_date'],
            'approvalDate' => $loan['approval_date'] ?? date('F j, Y'),
            'disbursementDate' => $loan['disbursement_date'] ?? date('F j, Y'),
            'firstPaymentDate' => $loan['first_payment_date'] ?? date('F j, Y', strtotime('+1 month')),
            'lenderRep' => $settings['treasurerFullName'] ?? 'Treasurer',
            'verificationUrl' => 'https://mft-group.org/verify/' . $loan['reference'],
            'borrowerSignature' => $loan['signature'] ?? '',
            'schedule' => [] // In a real implementation, this would be calculated
        ];
        
        // Include the loan agreement generator
        require_once __DIR__ . '/loan-agreement-generator.php';
        
        // Generate the loan agreement
        $result = generateLoanAgreement($loanData, 'F');
        
        if ($result) {
            error_log("Loan agreement generated successfully for loan ID: " . $loanId);
        } else {
            error_log("Failed to generate loan agreement for loan ID: " . $loanId);
        }
    } catch (Exception $e) {
        error_log("Error generating loan agreement: " . $e->getMessage());
    }
}
?>