<?php
/**
 * Loan Agreement Generator
 * This file handles the generation of loan agreements using Dompdf as the primary PDF engine
 */

// Include necessary utilities
require_once __DIR__ . '/dompdf-utils.php';
require_once __DIR__ . '/pdf-utils.php';

use Dompdf\Dompdf;
use Dompdf\Options;

/**
 * Generate a loan agreement PDF and save it to the documents folder
 * 
 * @param array $loanData Loan data including borrower, lender, and loan details
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return string|bool File path if saved to file, true if streamed, false on error
 */
function generateLoanAgreement($loanData, $output_mode = 'F') {
    try {
        // Generate HTML from template
        $html = generateLoanAgreementHtml($loanData);
        
        // Generate filename
        $filename = 'loan_agreement_' . $loanData['loanRef'] . '_' . date('Y-m-d');
        
        // Try Dompdf first (primary engine)
        try {
            return generatePdfWithDompdf($html, $filename, [], $output_mode);
        } catch (Exception $e) {
            error_log('Dompdf failed: ' . $e->getMessage());
            
            // Fallback to mPDF
            try {
                return generatePdfFromHtml($html, $filename, [], $output_mode);
            } catch (Exception $e2) {
                error_log('mPDF also failed: ' . $e2->getMessage());
                
                // Last resort: create simple text PDF
                return createSimpleTextPdf($html, $filename, $output_mode);
            }
        }
    } catch (Exception $e) {
        error_log('Loan Agreement Generation Error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Generate HTML for loan agreement using the template
 * 
 * @param array $loanData Loan data
 * @return string HTML content
 */
function generateLoanAgreementHtml($loanData) {
    // Extract variables for template
    $loanRef = $loanData['loanRef'] ?? 'N/A';
    $agreementDate = $loanData['agreementDate'] ?? date('F j, Y');
    $lenderName = $loanData['lenderName'] ?? 'MFT SELF HELP GROUP';
    $lenderRegNo = $loanData['lenderRegNo'] ?? 'REG-2025-001';
    $lenderAddress = $loanData['lenderAddress'] ?? 'Nairobi, Kenya';
    $borrowerName = $loanData['borrowerName'] ?? 'Member';
    $borrowerMemberNo = $loanData['borrowerMemberNo'] ?? 'N/A';
    $borrowerId = $loanData['borrowerId'] ?? 'N/A';
    $borrowerAddress = $loanData['borrowerAddress'] ?? 'N/A';
    $borrowerPhone = $loanData['borrowerPhone'] ?? 'N/A';
    $loanType = $loanData['loanType'] ?? 'Standard';
    $loanPurpose = $loanData['loanPurpose'] ?? 'General';
    $principalAmount = $loanData['principalAmount'] ?? 0;
    $interestRate = $loanData['interestRate'] ?? 0;
    $totalAmount = $loanData['totalAmount'] ?? 0;
    $repaymentPeriod = $loanData['repaymentPeriod'] ?? 0;
    $monthlyPayment = $loanData['monthlyPayment'] ?? 0;
    $applicationDate = $loanData['applicationDate'] ?? 'N/A';
    $approvalDate = $loanData['approvalDate'] ?? 'N/A';
    $disbursementDate = $loanData['disbursementDate'] ?? $approvalDate; // Default to approval date
    $firstPaymentDate = $loanData['firstPaymentDate'] ?? date('F j, Y', strtotime('+1 month')); // Default to next month
    $schedule = $loanData['schedule'] ?? [];
    $borrowerSignature = $loanData['borrowerSignature'] ?? '';
    $lenderSignature = $loanData['lenderSignature'] ?? '';
    $lenderRep = $loanData['lenderRep'] ?? 'Treasurer';
    $verificationUrl = $loanData['verificationUrl'] ?? 'https://mft-group.org/verify/' . $loanRef;
    
    // Generate QR code without GD dependency if possible
    $qrCode = generateQrCodeWithoutGd($loanRef);
    
    // Include the template
    ob_start();
    include __DIR__ . '/loan-agreement-template.php';
    $html = ob_get_clean();
    
    return $html;
}

/**
 * Generate QR code without GD extension dependency
 * 
 * @param string $data Data to encode in QR code
 * @return string Base64 encoded QR code image or empty string
 */
function generateQrCodeWithoutGd($data) {
    try {
        // Try to use chillerlan QR code library (which is already installed)
        if (class_exists('\\chillerlan\\QRCode\\QRCode')) {
            $qrCodeClass = '\\chillerlan\\QRCode\\QRCode';
            $qrCode = new $qrCodeClass();
            
            // Generate QR code as data URI
            $qrDataUri = $qrCode->render($data);
            
            // Return the data URI directly - Dompdf can handle this
            return $qrDataUri;
        }
        
        // Fallback: return empty string if QR code generation fails
        return '';
    } catch (Exception $e) {
        error_log('QR Code Generation Error: ' . $e->getMessage());
        return ''; // Return empty string if QR code generation fails
    }
}

/**
 * Get borrower signature from member profile
 * 
 * @param int $memberId Member ID
 * @return string Base64 encoded signature or empty string
 */
function getBorrowerSignature($memberId) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT signature FROM members WHERE id = ?");
        $stmt->execute([$memberId]);
        $member = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($member && !empty($member['signature'])) {
            return $member['signature'];
        }
        
        return '';
    } catch (Exception $e) {
        error_log('Error getting borrower signature: ' . $e->getMessage());
        return '';
    }
}

/**
 * Save loan agreement to database
 * 
 * @param int $loanId Loan ID
 * @param string $documentPath Path to the generated document
 * @param string $loanRef Loan reference number
 * @param int $borrowerMemberId Borrower member ID
 * @return bool Success status
 */
function saveLoanAgreementToDatabase($loanId, $documentPath, $loanRef, $borrowerMemberId) {
    try {
        $pdo = getDatabaseConnection();
        
        // Insert document record
        $docStmt = $pdo->prepare("INSERT INTO documents (title, file_path, category, created_at, uploaded_by) VALUES (?, ?, ?, NOW(), ?)");
        $docStmt->execute([
            'Loan Agreement - ' . $loanRef,
            $documentPath,
            'Agreement',
            $borrowerMemberId
        ]);
        
        $documentId = $pdo->lastInsertId();
        
        // Update loan record with document ID
        // Note: agreement_document_id column doesn't exist in the database schema
        // $updateStmt = $pdo->prepare("UPDATE loans SET agreement_document_id = ? WHERE id = ?");
        // $updateStmt->execute([$documentId, $loanId]);
        
        return true;
    } catch (Exception $e) {
        error_log('Database Error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Generate a test loan agreement
 * 
 * @return void
 */
function generateTestLoanAgreement() {
    $loanData = [
        'loanRef' => 'LOAN-' . date('Ymd') . '-' . strtoupper(substr(md5(time()), 0, 6)),
        'agreementDate' => date('F j, Y'),
        'lenderName' => 'MFT SELF HELP GROUP',
        'lenderRegNo' => 'REG-2025-001',
        'lenderAddress' => 'Nairobi, Kenya',
        'borrowerName' => 'Test Borrower',
        'borrowerMemberNo' => 'MFT123456',
        'borrowerId' => '12345678',
        'borrowerAddress' => 'Test Address, Nairobi',
        'borrowerPhone' => '+254700123456',
        'loanType' => 'Emergency',
        'loanPurpose' => 'Business: Business Stock',
        'principalAmount' => 1000.00,
        'interestRate' => 10.00,
        'totalAmount' => 1100.00,
        'repaymentPeriod' => 7,
        'monthlyPayment' => 183.33,
        'applicationDate' => date('F j, Y'),
        'approvalDate' => date('F j, Y'),
        'disbursementDate' => date('F j, Y'), // Same as approval date
        'firstPaymentDate' => date('F j, Y', strtotime('+1 month')), // Next month
        'lenderRep' => 'Treasurer',
        'verificationUrl' => 'https://mft-group.org/verify/',
        'schedule' => [
            ['number' => 1, 'date' => 'Nov 2025', 'principal' => 83.33, 'interest' => 100.00, 'total' => 183.33, 'balance' => 916.67],
            ['number' => 2, 'date' => 'Dec 2025', 'principal' => 91.66, 'interest' => 91.67, 'total' => 183.33, 'balance' => 825.01],
            ['number' => 3, 'date' => 'Jan 2026', 'principal' => 100.83, 'interest' => 82.50, 'total' => 183.33, 'balance' => 724.18],
            ['number' => 4, 'date' => 'Feb 2026', 'principal' => 110.91, 'interest' => 72.42, 'total' => 183.33, 'balance' => 613.27],
            ['number' => 5, 'date' => 'Mar 2026', 'principal' => 122.00, 'interest' => 61.33, 'total' => 183.33, 'balance' => 491.26],
            ['number' => 6, 'date' => 'Apr 2026', 'principal' => 134.20, 'interest' => 49.13, 'total' => 183.33, 'balance' => 357.06],
            ['number' => 7, 'date' => 'May 2026', 'principal' => 147.62, 'interest' => 35.71, 'total' => 183.33, 'balance' => 209.43]
        ]
    ];
    
    // Set output mode to 'F' to save file
    return generateLoanAgreement($loanData, 'F');
}

// If this file is called directly, generate a test agreement
if (basename($_SERVER['SCRIPT_NAME']) === basename(__FILE__)) {
    $result = generateTestLoanAgreement();
    if ($result) {
        echo "Test loan agreement generated successfully at: " . $result . "\n";
    } else {
        echo "Failed to generate test loan agreement.\n";
    }
}

?>