<?php
// MFT GROUP Initialize Officials Settings API
// This API endpoint initializes officials system settings

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Officials settings to insert/update
    $officials_settings = [
        // Chairman
        'chairmanFullName' => '',
        'chairmanEmail' => '',
        'chairmanPhone' => '',
        'chairmanProfilePhoto' => '',
        'chairmanTermStart' => '',
        'chairmanTermEnd' => '',
        'chairmanRoleDescription' => 'Leads the group and chairs all official meetings. Provides strategic direction and ensures the group operates in accordance with its constitution.',
        
        // Assistant Chairman
        'assistantChairmanFullName' => '',
        'assistantChairmanEmail' => '',
        'assistantChairmanPhone' => '',
        'assistantChairmanProfilePhoto' => '',
        'assistantChairmanTermStart' => '',
        'assistantChairmanTermEnd' => '',
        'assistantChairmanRoleDescription' => 'Supports the Chairman and assumes leadership responsibilities in the Chairman\'s absence. Assists in decision-making processes.',
        
        // Secretary
        'secretaryFullName' => '',
        'secretaryEmail' => '',
        'secretaryPhone' => '',
        'secretaryProfilePhoto' => '',
        'secretaryTermStart' => '',
        'secretaryTermEnd' => '',
        'secretaryRoleDescription' => 'Maintains official records, documents meeting minutes, and manages all correspondence. Ensures proper documentation of group activities.',
        
        // Organising Secretary
        'organisingSecretaryFullName' => '',
        'organisingSecretaryEmail' => '',
        'organisingSecretaryPhone' => '',
        'organisingSecretaryProfilePhoto' => '',
        'organisingSecretaryTermStart' => '',
        'organisingSecretaryTermEnd' => '',
        'organisingSecretaryRoleDescription' => 'Plans and coordinates group events, meetings, and activities. Manages logistics for all group functions and ensures smooth operations.',
        
        // Treasurer
        'treasurerFullName' => '',
        'treasurerEmail' => '',
        'treasurerPhone' => '',
        'treasurerProfilePhoto' => '',
        'treasurerTermStart' => '',
        'treasurerTermEnd' => '',
        'treasurerRoleDescription' => 'Manages all financial matters including contributions, expenses, and investments. Maintains accurate financial records and prepares reports.',
        
        // Loans Officer
        'loansOfficerFullName' => '',
        'loansOfficerEmail' => '',
        'loansOfficerPhone' => '',
        'loansOfficerProfilePhoto' => '',
        'loansOfficerTermStart' => '',
        'loansOfficerTermEnd' => '',
        'loansOfficerRoleDescription' => 'Oversees the loan application process, evaluates requests, and manages repayment schedules. Ensures responsible lending practices.'
    ];

    // Initialize counters
    $inserted_count = 0;
    $updated_count = 0;
    $failed_count = 0;
    
    // Begin transaction
    beginTransaction();
    
    // Insert or update each officials setting
    foreach ($officials_settings as $key => $value) {
        try {
            // Check if setting already exists
            $check_sql = "SELECT id FROM system_settings WHERE setting_key = ?";
            $existing = fetchSingleRow($check_sql, [$key]);
            
            if (!$existing) {
                // Insert setting
                $sql = "INSERT INTO system_settings (setting_key, setting_value, description) VALUES (?, ?, ?)";
                $description = ucfirst(str_replace(['FullName', 'Email', 'Phone', 'ProfilePhoto', 'TermStart', 'TermEnd', 'RoleDescription'], 
                    [' Full Name', ' Email', ' Phone', ' Profile Photo', ' Term Start', ' Term End', ' Role Description'], $key));
                executeNonQuery($sql, [$key, $value, $description]);
                $inserted_count++;
            } else {
                // Update setting if needed (only update description if empty)
                $check_desc_sql = "SELECT description FROM system_settings WHERE setting_key = ?";
                $setting = fetchSingleRow($check_desc_sql, [$key]);
                
                if (empty($setting['description'])) {
                    $description = ucfirst(str_replace(['FullName', 'Email', 'Phone', 'ProfilePhoto', 'TermStart', 'TermEnd', 'RoleDescription'], 
                        [' Full Name', ' Email', ' Phone', ' Profile Photo', ' Term Start', ' Term End', ' Role Description'], $key));
                    $sql = "UPDATE system_settings SET description = ? WHERE setting_key = ?";
                    executeNonQuery($sql, [$description, $key]);
                    $updated_count++;
                }
            }
        } catch (Exception $e) {
            error_log("Failed to process setting $key: " . $e->getMessage());
            $failed_count++;
        }
    }
    
    // Commit transaction
    commitTransaction();
    
    http_response_code(200);
    echo json_encode([
        'status' => 'success',
        'message' => "Successfully processed officials settings: $inserted_count inserted, $updated_count updated, $failed_count failed.",
        'inserted' => $inserted_count,
        'updated' => $updated_count,
        'failed' => $failed_count
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    try {
        rollbackTransaction();
    } catch (Exception $rollbackException) {
        // Ignore rollback errors
    }
    
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'error' => 'Failed to initialize officials settings: ' . $e->getMessage()
    ]);
}
?>