<?php
/**
 * Generate Project Members PDF
 * Simple endpoint to generate and download project members PDF
 */

// Include required files
require_once __DIR__ . '/../database/config.php';
require_once __DIR__ . '/dompdf-utils.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON data');
    }
    
    // Extract data
    $project_name = isset($input['project_name']) ? $input['project_name'] : 'Project';
    $members = isset($input['members']) ? $input['members'] : [];
    $filename = isset($input['filename']) ? $input['filename'] : 'project_members_' . date('Y-m-d');
    
    if (empty($members)) {
        throw new Exception('No members data provided');
    }
    
    // Generate HTML content
    $html = generateProjectMembersHtml($members, $project_name);
    
    // Set headers for PDF download
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
    header('Cache-Control: private, max-age=0, must-revalidate');
    header('Pragma: public');
    
    // Generate and output PDF
    generatePdfWithDompdf($html, $filename, [], 'I');
    exit;
    
} catch (Exception $e) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Error generating PDF: ' . $e->getMessage()]);
    exit;
}

/**
 * Generate HTML for project members PDF
 */
function generateProjectMembersHtml($members, $project_name) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Project Members - ' . htmlspecialchars($project_name) . '</title>
        <style>
            body { 
                font-family: Arial, sans-serif; 
                font-size: 12px; 
                margin: 20px;
                color: #333;
            }
            .header { 
                text-align: center; 
                margin-bottom: 20px; 
                padding-bottom: 10px; 
                border-bottom: 2px solid #FF5C1B;
            }
            .header h1 { 
                color: #FF5C1B; 
                margin: 0;
                font-size: 20px;
            }
            .project-info { 
                margin-bottom: 20px; 
                font-size: 14px;
                text-align: center;
            }
            table { 
                width: 100%; 
                border-collapse: collapse; 
                margin-top: 20px;
            }
            th, td { 
                border: 1px solid #999; 
                padding: 8px; 
                text-align: left;
            }
            th { 
                background-color: #FF5C1B; 
                color: white;
                font-weight: bold;
            }
            tr:nth-child(even) { 
                background-color: #f2f2f2;
            }
            .footer { 
                margin-top: 30px; 
                text-align: center; 
                font-size: 10px; 
                color: #666;
            }
            .report-info {
                text-align: center;
                margin-bottom: 20px;
                font-size: 12px;
                color: #666;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>Project Members Report</h1>
            <p>Project: ' . htmlspecialchars($project_name) . '</p>
        </div>
        
        <div class="report-info">
            <p>Report generated on: ' . date('F j, Y') . ' at ' . date('g:i A') . '</p>
            <p>Total Members: ' . count($members) . '</p>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Member ID</th>
                    <th>Name</th>
                    <th>Role</th>
                    <th>Contribution (KSH)</th>
                    <th>Join Date</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>';
    
    // Generate table rows
    foreach ($members as $member) {
        $html .= '<tr>';
        $html .= '<td>' . htmlspecialchars($member['member_id'] ?? $member['member_ref'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars(($member['first_name'] ?? '') . ' ' . ($member['last_name'] ?? '')) . '</td>';
        $html .= '<td>' . htmlspecialchars($member['role'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars(number_format($member['contribution_amount'] ?? 0, 2)) . '</td>';
        $html .= '<td>' . htmlspecialchars($member['joined_date'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars($member['member_project_status'] ?? 'Active') . '</td>';
        $html .= '</tr>';
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            <p>Generated by MFT GROUP Administration System</p>
        </div>
    </body>
    </html>';
    
    return $html;
}
?>