<?php
// MFT GROUP Fines API
// This API provides fine data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';
require_once __DIR__ . '/notification-service-class.php';
require_once 'send_sms.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];
error_log("Fines API called with method: " . $method);

// Add debugging to see if this code is executed multiple times
static $execution_count = 0;
$execution_count++;
error_log("Fines API execution count: " . $execution_count);

// Function to send notification about fine updates
function notifyFineUpdate($memberId, $fineId, $updateType, $details = []) {
    try {
        // Send database notification
        $pdo = getDatabaseConnection();
        
        $notificationData = [
            'member_id' => $memberId,
            'fine_id' => $fineId,
            'update_type' => $updateType,
            'details' => $details
        ];
        
        $sql = "INSERT INTO notifications (type, data, target, member_id) VALUES (?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'fine_update',
            json_encode($notificationData),
            'member_dashboard',
            $memberId
        ]);
        
        // Send real-time notification
        $notificationService = new MFTNotificationService();
        $notificationService->sendNotification('fine_update', $notificationData, 'member_dashboard', $memberId);
        
        // Send SMS notification
        if ($updateType === 'created' && isset($details['amount']) && isset($details['reason']) && isset($details['outstanding_fines'])) {
            sendFineNotificationSMS($memberId, $details['amount'], $details['reason'], $details['outstanding_fines']);
        }
        
        error_log("Fine notification sent for member $memberId, fine $fineId");
    } catch (Exception $e) {
        error_log("Failed to send fine notification: " . $e->getMessage());
    }
}

// Function to get total outstanding fines for a member
function getMemberOutstandingFines($memberId) {
    try {
        $sql = "SELECT SUM(amount) as total_outstanding FROM fines 
                WHERE member_id = ? AND status != 'Paid'";
        $result = fetchSingleRow($sql, [$memberId]);
        return $result['total_outstanding'] ?? 0;
    } catch (Exception $e) {
        error_log("Error getting outstanding fines: " . $e->getMessage());
        return 0;
    }
}

error_log("Method detected: " . $method);

try {
    switch ($method) {
        case 'GET':
            error_log("Handling GET request");
            handleGetRequest();
            exit(); // Important: exit after handling request
            break;
        case 'POST':
            error_log("Handling POST request");
            handlePostRequest();
            exit(); // Important: exit after handling request
            break;
        case 'PUT':
            error_log("Handling PUT request");
            handlePutRequest();
            exit(); // Important: exit after handling request
            break;
        case 'DELETE':
            error_log("Handling DELETE request");
            handleDeleteRequest();
            exit(); // Important: exit after handling request
            break;
        default:
            error_log("Handling default case for method: " . $method);
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit(); // Important: exit after sending error response
            break;
    }
} catch (Exception $e) {
    error_log("Exception caught: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
    exit(); // Important: exit after sending error response
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific fine
        getFine($_GET['id']);
    } else if (isset($_GET['member_id'])) {
        // Get fines for a specific member
        getMemberFines($_GET['member_id']);
    } else {
        // Get all fines with optional filters
        getFines();
    }
}

function getFine($id) {
    $sql = "SELECT f.id, f.member_id, f.amount, f.reason, f.description, f.status, 
                   f.payment_reference, f.fine_date, f.paid_date, f.created_by,
                   f.created_at, f.updated_at, m.first_name, m.last_name, m.member_id as member_ref,
                   m.phone as member_phone, m.email as member_email
            FROM fines f
            JOIN members m ON f.member_id = m.id
            WHERE f.id = ?";
    
    $fine = fetchSingleRow($sql, [$id]);
    
    if ($fine) {
        http_response_code(200);
        echo json_encode($fine);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Fine not found']);
    }
}

function getMemberFines($member_id) {
    $sql = "SELECT f.id, f.member_id, f.amount, f.reason, f.description, f.status, 
                   f.payment_reference, f.fine_date, f.paid_date, f.created_by,
                   f.created_at, f.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM fines f
            JOIN members m ON f.member_id = m.id
            WHERE f.member_id = ?
            ORDER BY f.fine_date DESC";
    
    $fines = executeQuery($sql, [$member_id]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($fines),
        'fines' => $fines
    ]);
}

function getFines() {
    $status_filter = isset($_GET['status']) ? $_GET['status'] : null;
    $member_filter = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $date_from_filter = isset($_GET['date_from']) ? $_GET['date_from'] : null;
    $date_to_filter = isset($_GET['date_to']) ? $_GET['date_to'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    $sql = "SELECT f.id, f.member_id, f.amount, f.reason, f.description, f.status, 
                   f.payment_reference, f.fine_date, f.paid_date, f.created_by,
                   f.created_at, f.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM fines f
            JOIN members m ON f.member_id = m.id";
    
    $params = [];
    $where_conditions = [];
    
    if ($status_filter) {
        $where_conditions[] = "f.status = ?";
        $params[] = $status_filter;
    }
    
    if ($member_filter) {
        $where_conditions[] = "f.member_id = ?";
        $params[] = $member_filter;
    }
    
    if ($date_from_filter) {
        $where_conditions[] = "f.fine_date >= ?";
        $params[] = $date_from_filter;
    }
    
    if ($date_to_filter) {
        $where_conditions[] = "f.fine_date <= ?";
        $params[] = $date_to_filter;
    }
    
    if (!empty($where_conditions)) {
        $sql .= " WHERE " . implode(' AND ', $where_conditions);
    }
    
    $sql .= " ORDER BY f.fine_date DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $fines = executeQuery($sql, $params);
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM fines f";
    $count_params = [];
    
    if (!empty($where_conditions)) {
        $count_sql .= " WHERE " . implode(' AND ', $where_conditions);
        $count_params = $params;
        // Remove limit and offset parameters
        array_pop($count_params);
        array_pop($count_params);
    }
    
    $count_result = fetchSingleRow($count_sql, $count_params);
    $total = $count_result['total'];
    
    http_response_code(200);
    echo json_encode([
        'count' => count($fines),
        'total' => $total,
        'offset' => $offset,
        'limit' => $limit,
        'fines' => $fines
    ]);
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['member_id', 'amount', 'reason', 'fine_date'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Set default values
    if (!isset($input['status'])) {
        $input['status'] = 'Pending';
    }
    
    // Insert fine
    $sql = "INSERT INTO fines (member_id, amount, reason, description, status, fine_date, created_by) 
            VALUES (?, ?, ?, ?, ?, ?, ?)";
    
    $params = [
        $input['member_id'],
        $input['amount'],
        $input['reason'],
        $input['description'] ?? null,
        $input['status'],
        $input['fine_date'],
        $input['created_by'] ?? null
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $fine_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        // Create audit log for fine creation
        createAuditLog(null, 'create', 'fines', $fine_id, null, [
            'member_id' => $input['member_id'],
            'amount' => $input['amount'],
            'reason' => $input['reason'],
            'status' => $input['status']
        ]);
        
        // Get member details for notification
        $member_sql = "SELECT first_name, last_name, phone, email FROM members WHERE id = ?";
        $member = fetchSingleRow($member_sql, [$input['member_id']]);
        
        // Get total outstanding fines
        $outstanding_fines = getMemberOutstandingFines($input['member_id']);
        
        // Send notification about the new fine
        notifyFineUpdate($input['member_id'], $fine_id, 'created', [
            'amount' => $input['amount'],
            'reason' => $input['reason'],
            'status' => $input['status'],
            'outstanding_fines' => $outstanding_fines
        ]);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Fine recorded successfully',
            'fine_id' => $fine_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("Error creating fine: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to create fine: ' . $e->getMessage()]);
        exit(); // Important: exit after sending error response
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Fine ID is required']);
        return;
    }
    
    // Get the existing fine to get member_id
    $existing_sql = "SELECT member_id, amount, reason, status FROM fines WHERE id = ?";
    $existing = fetchSingleRow($existing_sql, [$_GET['id']]);
    
    if (!$existing) {
        http_response_code(404);
        echo json_encode(['error' => 'Fine not found']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Update fine
    $fields = [];
    $params = [];
    
    foreach ($input as $key => $value) {
        // Skip fields that shouldn't be updated directly
        if (in_array($key, ['id', 'created_at'])) {
            continue;
        }
        
        $fields[] = "$key = ?";
        $params[] = $value;
    }
    
    // Add fine ID to parameters
    $params[] = $_GET['id'];
    
    if (empty($fields)) {
        http_response_code(400);
        echo json_encode(['error' => 'No valid fields to update']);
        return;
    }
    
    $sql = "UPDATE fines SET " . implode(', ', $fields) . " WHERE id = ?";
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $params);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Fine not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for fine update
        createAuditLog(null, 'update', 'fines', $_GET['id'], $existing, $input);
        
        // Get total outstanding fines
        $outstanding_fines = getMemberOutstandingFines($existing['member_id']);
        
        // Send notification about the updated fine
        notifyFineUpdate($existing['member_id'], $_GET['id'], 'updated', array_merge($input, [
            'outstanding_fines' => $outstanding_fines
        ]));
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Fine updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("Error updating fine: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to update fine: ' . $e->getMessage()]);
        exit(); // Important: exit after sending error response
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Fine ID is required']);
        return;
    }
    
    // Get fine details before deleting
    $fine_sql = "SELECT member_id, amount FROM fines WHERE id = ?";
    $fine = fetchSingleRow($fine_sql, [$_GET['id']]);
    
    if (!$fine) {
        http_response_code(404);
        echo json_encode(['error' => 'Fine not found']);
        return;
    }
    
    $sql = "DELETE FROM fines WHERE id = ?";
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, [$_GET['id']]);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Fine not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for fine deletion
        createAuditLog(null, 'delete', 'fines', $_GET['id'], $fine, null);
        
        // Get total outstanding fines
        $outstanding_fines = getMemberOutstandingFines($fine['member_id']);
        
        // Send notification about the deleted fine
        notifyFineUpdate($fine['member_id'], $_GET['id'], 'deleted', [
            'amount' => $fine['amount'],
            'outstanding_fines' => $outstanding_fines
        ]);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Fine deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        error_log("Error deleting fine: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Failed to delete fine: ' . $e->getMessage()]);
        exit(); // Important: exit after sending error response
    }
}
?>