<?php
/**
 * Export Project Members to PDF API
 * Handles PDF export requests for project members data
 */

// Include required files
require_once __DIR__ . '/../database/config.php';
require_once __DIR__ . '/dompdf-utils.php';

try {
    // Get the raw POST data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid data');
    }
    
    $project_id = isset($input['project_id']) ? $input['project_id'] : null;
    $project_name = isset($input['project_name']) ? $input['project_name'] : 'Project';
    $filename = isset($input['filename']) ? $input['filename'] : 'project_members_' . date('Y-m-d_H-i-s');
    $members = isset($input['members']) ? $input['members'] : [];
    
    if (!$project_id || empty($members)) {
        throw new Exception('Project ID and members data are required');
    }
    
    // Generate HTML for the PDF
    $html = generateProjectMembersHtml($members, $project_name);
    
    // Set headers for PDF download
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
    header('Cache-Control: private, max-age=0, must-revalidate');
    header('Pragma: public');
    
    // Generate PDF using Dompdf and output directly to browser
    generatePdfWithDompdf($html, $filename, [], 'I');
    
} catch (Exception $e) {
    // Set error headers
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['error' => 'Export failed: ' . $e->getMessage()]);
    exit;
}

/**
 * Generate HTML for project members export
 */
function generateProjectMembersHtml($members, $project_name) {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Project Members - ' . htmlspecialchars($project_name) . '</title>
        <style>
            body { 
                font-family: Arial, sans-serif; 
                font-size: 10pt; 
                margin: 20px;
                color: #333;
            }
            .header { 
                text-align: center; 
                margin-bottom: 20px; 
                padding-bottom: 10px; 
                border-bottom: 2px solid #FF5C1B;
            }
            .header h1 { 
                color: #FF5C1B; 
                margin: 0;
                font-size: 18pt;
            }
            .project-info { 
                margin-bottom: 20px; 
                font-size: 12pt;
                text-align: center;
            }
            table { 
                width: 100%; 
                border-collapse: collapse; 
                margin-top: 20px;
            }
            th, td { 
                border: 1px solid #ddd; 
                padding: 8px; 
                text-align: left;
            }
            th { 
                background-color: #FF5C1B; 
                color: white;
                font-weight: bold;
            }
            tr:nth-child(even) { 
                background-color: #f2f2f2;
            }
            .footer { 
                margin-top: 30px; 
                text-align: center; 
                font-size: 9pt; 
                color: #666;
            }
            .report-info {
                text-align: center;
                margin-bottom: 20px;
                font-size: 10pt;
                color: #666;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>Project Members Report</h1>
            <p>Project: ' . htmlspecialchars($project_name) . '</p>
        </div>
        
        <div class="report-info">
            <p>Report generated on: ' . date('F j, Y') . ' at ' . date('g:i A') . '</p>
            <p>Total Members: ' . count($members) . '</p>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Member ID</th>
                    <th>Name</th>
                    <th>Role</th>
                    <th>Contribution (KSH)</th>
                    <th>Join Date</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>';
    
    // Generate table rows
    foreach ($members as $member) {
        $html .= '<tr>';
        $html .= '<td>' . htmlspecialchars($member['member_id'] ?? $member['member_ref'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars(($member['first_name'] ?? '') . ' ' . ($member['last_name'] ?? '')) . '</td>';
        $html .= '<td>' . htmlspecialchars($member['role'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars(number_format($member['contribution_amount'] ?? 0, 2)) . '</td>';
        $html .= '<td>' . htmlspecialchars($member['joined_date'] ?? 'N/A') . '</td>';
        $html .= '<td>' . htmlspecialchars($member['member_project_status'] ?? 'Active') . '</td>';
        $html .= '</tr>';
    }
    
    $html .= '
            </tbody>
        </table>
        <div class="footer">
            <p>Generated by MFT GROUP Administration System</p>
        </div>
    </body>
    </html>';
    
    return $html;
}
?>