<?php
// MFT GROUP Expenses API
// This API provides expense tracking functionality for the organization

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    error_log("Error in expenses API: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    $action = isset($_GET['action']) ? $_GET['action'] : '';
    
    switch ($action) {
        case 'getExpensesSummary':
            getExpensesSummary();
            break;
        case 'getAllExpenses':
            getAllExpenses();
            break;
        case 'getExpenseById':
            $expense_id = isset($_GET['id']) ? $_GET['id'] : null;
            if ($expense_id) {
                getExpenseById($expense_id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Missing expense ID']);
            }
            break;
        default:
            getAllExpenses();
            break;
    }
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    createExpense($input);
}

function handlePutRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    $expense_id = isset($_GET['id']) ? $_GET['id'] : null;
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    if (!$expense_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing expense ID']);
        return;
    }
    
    updateExpense($expense_id, $input);
}

function handleDeleteRequest() {
    $expense_id = isset($_GET['id']) ? $_GET['id'] : null;
    
    if (!$expense_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing expense ID']);
        return;
    }
    
    deleteExpense($expense_id);
}

function getExpensesSummary() {
    try {
        // Get total expenses from Expense account type
        $expenses_sql = "SELECT COALESCE(SUM(balance), 0) as total_expenses FROM accounts WHERE account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')";
        $expenses_data = fetchSingleRow($expenses_sql);
        
        // Get expense categories
        $categories_sql = "SELECT at.name as category, COALESCE(SUM(a.balance), 0) as amount 
                          FROM account_types at 
                          LEFT JOIN accounts a ON at.id = a.account_type_id 
                          WHERE at.name = 'Expense' 
                          GROUP BY at.name";
        $categories_data = executeQuery($categories_sql);
        
        $data = [
            'total_expenses' => $expenses_data['total_expenses'] ?? 0,
            'categories' => $categories_data
        ];
        
        http_response_code(200);
        echo json_encode($data);
    } catch (Exception $e) {
        error_log("Error getting expenses summary: " . $e->getMessage());
        throw $e;
    }
}

function getAllExpenses() {
    try {
        // Get all expense accounts
        $expenses_sql = "SELECT a.id, a.name, a.balance, a.description, a.created_at, a.updated_at,
                                at.name as category
                         FROM accounts a
                         JOIN account_types at ON a.account_type_id = at.id
                         WHERE at.name = 'Expense'
                         ORDER BY a.created_at DESC";
        $expenses_data = executeQuery($expenses_sql);
        
        http_response_code(200);
        echo json_encode($expenses_data);
    } catch (Exception $e) {
        error_log("Error getting all expenses: " . $e->getMessage());
        throw $e;
    }
}

function getExpenseById($expense_id) {
    try {
        $expense_sql = "SELECT a.id, a.name, a.balance, a.description, a.created_at, a.updated_at,
                               at.name as category
                        FROM accounts a
                        JOIN account_types at ON a.account_type_id = at.id
                        WHERE at.name = 'Expense' AND a.id = ?";
        $expense_data = fetchSingleRow($expense_sql, [$expense_id]);
        
        if (!$expense_data) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        http_response_code(200);
        echo json_encode($expense_data);
    } catch (Exception $e) {
        error_log("Error getting expense by ID: " . $e->getMessage());
        throw $e;
    }
}

function createExpense($data) {
    try {
        // Validate required fields
        $required_fields = ['name', 'amount', 'description'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Get Expense account type ID
        $account_type_sql = "SELECT id FROM account_types WHERE name = 'Expense'";
        $account_type = fetchSingleRow($account_type_sql);
        
        if (!$account_type) {
            http_response_code(500);
            echo json_encode(['error' => 'Expense account type not found']);
            return;
        }
        
        // Insert new expense account
        $insert_sql = "INSERT INTO accounts (name, account_type_id, balance, description) VALUES (?, ?, ?, ?)";
        $params = [
            $data['name'],
            $account_type['id'],
            $data['amount'],
            $data['description']
        ];
        
        $account_id = executeInsert($insert_sql, $params);
        
        // Create audit log for expense creation
        createAuditLog(null, 'create', 'accounts', $account_id, null, [
            'name' => $data['name'],
            'balance' => $data['amount'],
            'description' => $data['description']
        ]);
        
        // Return the created expense
        getExpenseById($account_id);
    } catch (Exception $e) {
        error_log("Error creating expense: " . $e->getMessage());
        throw $e;
    }
}

function updateExpense($expense_id, $data) {
    try {
        // Check if expense exists
        $check_sql = "SELECT id FROM accounts WHERE id = ? AND account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')";
        $existing = fetchSingleRow($check_sql, [$expense_id]);
        
        if (!$existing) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        // Update expense account
        $update_fields = [];
        $params = [];
        
        if (isset($data['name'])) {
            $update_fields[] = "name = ?";
            $params[] = $data['name'];
        }
        
        if (isset($data['amount'])) {
            $update_fields[] = "balance = ?";
            $params[] = $data['amount'];
        }
        
        if (isset($data['description'])) {
            $update_fields[] = "description = ?";
            $params[] = $data['description'];
        }
        
        if (empty($update_fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'No fields to update']);
            return;
        }
        
        $update_fields[] = "updated_at = CURRENT_TIMESTAMP";
        $params[] = $expense_id;
        
        $update_sql = "UPDATE accounts SET " . implode(', ', $update_fields) . " WHERE id = ?";
        executeUpdate($update_sql, $params);
        
        // Create audit log for expense update
        createAuditLog(null, 'update', 'accounts', $expense_id, null, $data);
        
        // Return the updated expense
        getExpenseById($expense_id);
    } catch (Exception $e) {
        error_log("Error updating expense: " . $e->getMessage());
        throw $e;
    }
}

function deleteExpense($expense_id) {
    try {
        // Check if expense exists
        $check_sql = "SELECT id FROM accounts WHERE id = ? AND account_type_id = (SELECT id FROM account_types WHERE name = 'Expense')";
        $existing = fetchSingleRow($check_sql, [$expense_id]);
        
        if (!$existing) {
            http_response_code(404);
            echo json_encode(['error' => 'Expense not found']);
            return;
        }
        
        // Delete expense account
        $delete_sql = "DELETE FROM accounts WHERE id = ?";
        executeDelete($delete_sql, [$expense_id]);
        
        // Create audit log for expense deletion
        createAuditLog(null, 'delete', 'accounts', $expense_id, null, null);
        
        http_response_code(200);
        echo json_encode(['message' => 'Expense deleted successfully']);
    } catch (Exception $e) {
        error_log("Error deleting expense: " . $e->getMessage());
        throw $e;
    }
}
?>