<?php
/**
 * Dompdf Utility Functions
 * This file provides Dompdf integration for the MFT application as an alternative to mPDF/TCPDF
 */

// Include Dompdf autoloader with error handling
$dompdfLoaded = false;
$dompdfLoadError = '';
try {
    require_once __DIR__ . '/../vendor/autoload.php';
    $dompdfLoaded = true;
} catch (RuntimeException $e) {
    // Handle PHP version incompatibility
    $dompdfLoadError = $e->getMessage();
    error_log('Dompdf loading failed: ' . $e->getMessage());
}

/**
 * Generate a PDF from HTML content using Dompdf
 * 
 * @param string $html HTML content to convert to PDF
 * @param string $filename Name of the PDF file (without extension)
 * @param array $options Dompdf configuration options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return mixed PDF output or file path if saved to file
 */
function generatePdfWithDompdf($html, $filename = 'document', $options = [], $output_mode = 'D') {
    try {
        // Check if Dompdf is available
        if (!class_exists('Dompdf\Dompdf')) {
            throw new Exception('Dompdf library not available. Please check PHP version compatibility.');
        }
        
        // Default configuration
        $default_options = [
            'defaultFont' => 'Arial',
            'isRemoteEnabled' => true,
            'isPhpEnabled' => false, // Security consideration
            'isJavascriptEnabled' => false, // Security consideration
            'debugPng' => false,
            'debugKeepTemp' => false,
            'debugCss' => false,
            'debugLayout' => false,
            'debugLayoutLines' => false,
            'debugLayoutBlocks' => false,
            'debugLayoutInline' => false,
            'debugLayoutPaddingBox' => false,
        ];
        
        // Merge user options with defaults
        $config = array_merge($default_options, $options);
        
        // Create new Dompdf options instance
        $dompdfOptions = new Dompdf\Options();
        
        // Set options
        foreach ($config as $key => $value) {
            // Convert snake_case to CamelCase for setter methods
            $parts = explode('_', $key);
            $camelCase = '';
            foreach ($parts as $part) {
                $camelCase .= ucfirst($part);
            }
            $setterMethod = 'set' . $camelCase;
            
            if (method_exists($dompdfOptions, $setterMethod)) {
                $dompdfOptions->$setterMethod($value);
            }
        }
        
        // Create new Dompdf instance
        $dompdf = new Dompdf\Dompdf($dompdfOptions);
        
        // Set document properties
        $dompdf->setPaper('A4', 'portrait');
        
        // Preprocess HTML to fix common issues
        $html = preprocessHtmlForDompdf($html);
        
        // Load HTML content
        $dompdf->loadHtml($html);
        
        // Render the PDF
        $dompdf->render();
        
        // Handle different output modes
        switch ($output_mode) {
            case 'D': // Download
                $dompdf->stream($filename . '.pdf', ['Attachment' => true]);
                exit;
                
            case 'I': // Inline
                $dompdf->stream($filename . '.pdf', ['Attachment' => false]);
                exit;
                
            case 'F': // File
                $output = $dompdf->output();
                
                // Check if this is a loan agreement and save in the correct directory
                if (strpos($filename, 'loan_agreement') !== false) {
                    $file_path = __DIR__ . '/../documents/loan_agreements/' . $filename . '.pdf';
                } else {
                    $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
                }
                
                // Create directory if it doesn't exist
                $dir = dirname($file_path);
                if (!file_exists($dir)) {
                    mkdir($dir, 0777, true);
                }
                
                file_put_contents($file_path, $output);
                return $file_path;
                
            default:
                $dompdf->stream($filename . '.pdf', ['Attachment' => true]);
                exit;
        }
    } catch (Exception $e) {
        error_log('Dompdf Generation Error: ' . $e->getMessage());
        throw new Exception('Failed to generate PDF with Dompdf: ' . $e->getMessage());
    }
}

/**
 * Preprocess HTML to fix common issues that cause problems with Dompdf
 * @param string $html HTML content
 * @return string Processed HTML content
 */
function preprocessHtmlForDompdf($html) {
    // Replace footer tags with div tags since footer is not well supported
    $html = str_replace('<footer>', '<div class="footer">', $html);
    $html = str_replace('</footer>', '</div>', $html);
    
    // Remove or replace SVG elements which may cause issues
    $html = preg_replace('/<svg[^>]*>.*?<\/svg>/is', '<div>[SVG Content]</div>', $html);
    
    // Remove or replace other problematic elements
    $html = preg_replace('/<script[^>]*>.*?<\/script>/is', '', $html);
    
    return $html;
}

/**
 * Generate a PDF table from data using Dompdf
 * 
 * @param array $headers Table headers
 * @param array $data Table data (array of arrays)
 * @param string $title Table title
 * @param string $filename PDF filename
 * @param array $options Dompdf options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @return void
 */
function generatePdfTableWithDompdf($headers, $data, $title = '', $filename = 'table', $options = [], $output_mode = 'F') {
    $html = '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>' . htmlspecialchars($title) . '</title>
        <style>
            body { font-family: Arial, sans-serif; font-size: 10pt; }
            .title { text-align: center; font-size: 14pt; margin-bottom: 20px; }
            table { width: 100%; border-collapse: collapse; }
            th, td { border: 1px solid #000; padding: 6px; text-align: left; }
            th { background-color: #e0e0e0; font-weight: bold; }
        </style>
    </head>
    <body>';
    
    if (!empty($title)) {
        $html .= '<div class="title">' . htmlspecialchars($title) . '</div>';
    }
    
    $html .= '<table>
        <thead>
            <tr>';
    
    // Add headers
    foreach ($headers as $header) {
        $html .= '<th>' . htmlspecialchars($header) . '</th>';
    }
    
    $html .= '</tr>
        </thead>
        <tbody>';
    
    // Add data rows
    foreach ($data as $row) {
        $html .= '<tr>';
        foreach ($row as $cell) {
            $html .= '<td>' . htmlspecialchars($cell) . '</td>';
        }
        $html .= '</tr>';
    }
    
    $html .= '
        </tbody>
    </table>
    </body>
    </html>';
    
    return generatePdfWithDompdf($html, $filename, $options, $output_mode);
}

/**
 * Enhanced version of the existing generatePdfFromHtml function that supports multiple engines
 * This is a Dompdf-specific version to avoid naming conflicts
 * 
 * @param string $html HTML content to convert to PDF
 * @param string $filename Name of the PDF file (without extension)
 * @param array $options Configuration options
 * @param string $output_mode Output mode: 'D' = download, 'I' = inline, 'F' = file
 * @param string $engine PDF engine to use: 'dompdf', 'mpdf', 'tcpdf'
 * @return mixed PDF output or file path if saved to file
 */
function generatePdfFromHtmlMultiEngineDompdf($html, $filename = 'document', $options = [], $output_mode = 'D', $engine = 'dompdf') {
    try {
        switch ($engine) {
            case 'dompdf':
                return generatePdfWithDompdf($html, $filename, $options, $output_mode);
                
            case 'mpdf':
                // Use existing mPDF implementation
                require_once __DIR__ . '/pdf-utils.php';
                return generatePdfFromHtml($html, $filename, $options, $output_mode);
                
            case 'tcpdf':
                // Use TCPDF implementation
                require_once __DIR__ . '/../lib/tcpdf/SimpleTCPDF.php';
                $tcpdfClass = 'SimpleTCPDF';
                $pdf = new $tcpdfClass();
                $pdf->AddPage();
                $pdf->WriteHTML($html);
                $pdf_content = $pdf->Output('', 'S');
                
                switch ($output_mode) {
                    case 'D': // Download
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                        
                    case 'I': // Inline
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: inline; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                        
                    case 'F': // File
                        $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
                        $dir = dirname($file_path);
                        if (!file_exists($dir)) {
                            mkdir($dir, 0777, true);
                        }
                        file_put_contents($file_path, $pdf_content);
                        return $file_path;
                        
                    default:
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: attachment; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                }
                break;
                
            default:
                // Default to Dompdf
                return generatePdfWithDompdf($html, $filename, $options, $output_mode);
        }
    } catch (Exception $e) {
        error_log("PDF Generation Error with $engine: " . $e->getMessage());
        
        // Fallback to another engine
        switch ($engine) {
            case 'dompdf':
                // Fallback to mPDF
                try {
                    require_once __DIR__ . '/pdf-utils.php';
                    return generatePdfFromHtml($html, $filename, $options, $output_mode);
                } catch (Exception $e2) {
                    error_log("mPDF fallback also failed: " . $e2->getMessage());
                    throw new Exception("All PDF generation methods failed: " . $e->getMessage() . "; " . $e2->getMessage());
                }
                
            case 'mpdf':
                // Fallback to TCPDF
                try {
                    require_once __DIR__ . '/../lib/tcpdf/SimpleTCPDF.php';
                    $tcpdfClass = 'SimpleTCPDF';
                    $pdf = new $tcpdfClass();
                    $pdf->AddPage();
                    $pdf->WriteHTML($html);
                    $pdf_content = $pdf->Output('', 'S');
                    
                    if ($output_mode === 'F') {
                        $file_path = __DIR__ . '/../documents/' . $filename . '.pdf';
                        $dir = dirname($file_path);
                        if (!file_exists($dir)) {
                            mkdir($dir, 0777, true);
                        }
                        file_put_contents($file_path, $pdf_content);
                        return $file_path;
                    } else {
                        header('Content-Type: application/pdf');
                        header('Content-Disposition: ' . ($output_mode === 'I' ? 'inline' : 'attachment') . '; filename="' . $filename . '.pdf"');
                        echo $pdf_content;
                        exit;
                    }
                } catch (Exception $e2) {
                    error_log("TCPDF fallback also failed: " . $e2->getMessage());
                    throw new Exception("All PDF generation methods failed: " . $e->getMessage() . "; " . $e2->getMessage());
                }
                
            default:
                throw new Exception("PDF generation failed: " . $e->getMessage());
        }
    }
}

// Only load Dompdf classes if the library was successfully loaded
if ($dompdfLoaded && class_exists('Dompdf\Dompdf')) {
    // Dompdf is available and can be used
} else {
    // Define fallback functions if Dompdf is not available
    if (!function_exists('generatePdfWithDompdf')) {
        function generatePdfWithDompdf($html, $filename = 'document', $options = [], $output_mode = 'D') {
            throw new Exception('Dompdf library not available. Please check PHP version compatibility. Error: ' . ($GLOBALS['dompdfLoadError'] ?? 'Unknown error'));
        }
    }
}

?>