<?php
// MFT GROUP Dividends API
// This API provides dividend data for the frontend dashboards

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific dividend
        getDividend($_GET['id']);
    } else if (isset($_GET['member_id'])) {
        // Get dividends for a specific member
        getMemberDividends($_GET['member_id']);
    } else if (isset($_GET['year'])) {
        // Get dividends for a specific year
        getYearDividends($_GET['year']);
    } else {
        // Get all dividends with optional filters
        getDividends();
    }
}

function getDividend($id) {
    $sql = "SELECT d.id, d.member_id, d.amount, d.financial_year, d.payment_date, d.status, 
                   d.payment_reference, d.created_at, d.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM dividends d
            JOIN members m ON d.member_id = m.id
            WHERE d.id = ?";
    
    $dividend = fetchSingleRow($sql, [$id]);
    
    if ($dividend) {
        http_response_code(200);
        echo json_encode($dividend);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Dividend not found']);
    }
}

function getMemberDividends($member_id) {
    $sql = "SELECT d.id, d.member_id, d.amount, d.financial_year, d.payment_date, d.status, 
                   d.payment_reference, d.created_at, d.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM dividends d
            JOIN members m ON d.member_id = m.id
            WHERE d.member_id = ?
            ORDER BY d.financial_year DESC";
    
    $dividends = executeQuery($sql, [$member_id]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($dividends),
        'dividends' => $dividends
    ]);
}

function getYearDividends($year) {
    $sql = "SELECT d.id, d.member_id, d.amount, d.financial_year, d.payment_date, d.status, 
                   d.payment_reference, d.created_at, d.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM dividends d
            JOIN members m ON d.member_id = m.id
            WHERE d.financial_year = ?
            ORDER BY d.payment_date DESC";
    
    $dividends = executeQuery($sql, [$year]);
    
    http_response_code(200);
    echo json_encode([
        'count' => count($dividends),
        'dividends' => $dividends
    ]);
}

function getDividends() {
    $status_filter = isset($_GET['status']) ? $_GET['status'] : null;
    $year_filter = isset($_GET['year']) ? $_GET['year'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    
    $sql = "SELECT d.id, d.member_id, d.amount, d.financial_year, d.payment_date, d.status, 
                   d.payment_reference, d.created_at, d.updated_at, m.first_name, m.last_name, m.member_id as member_ref
            FROM dividends d
            JOIN members m ON d.member_id = m.id";
    
    $params = [];
    $conditions = [];
    
    if ($status_filter) {
        $conditions[] = "d.status = ?";
        $params[] = $status_filter;
    }
    
    if ($year_filter) {
        $conditions[] = "d.financial_year = ?";
        $params[] = $year_filter;
    }
    
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    $sql .= " ORDER BY d.financial_year DESC, d.payment_date DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $dividends = executeQuery($sql, $params);
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM dividends";
    $count_params = [];
    
    if (!empty($conditions)) {
        $count_sql .= " WHERE " . implode(" AND ", $conditions);
        $count_params = $params;
        // Remove the limit and offset parameters
        array_pop($count_params);
        array_pop($count_params);
    }
    
    $count_result = fetchSingleRow($count_sql, $count_params);
    $total = $count_result['total'];
    
    http_response_code(200);
    echo json_encode([
        'count' => count($dividends),
        'total' => $total,
        'offset' => $offset,
        'limit' => $limit,
        'dividends' => $dividends
    ]);
}

function handlePostRequest() {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Validate required fields
    $required_fields = ['member_id', 'amount', 'financial_year'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Set default values
    if (!isset($input['status'])) {
        $input['status'] = 'Pending';
    }
    
    // Insert dividend
    $sql = "INSERT INTO dividends (member_id, amount, financial_year, payment_date, status) 
            VALUES (?, ?, ?, ?, ?)";
    
    $params = [
        $input['member_id'],
        $input['amount'],
        $input['financial_year'],
        $input['payment_date'] ?? null,
        $input['status']
    ];
    
    try {
        beginTransaction();
        
        executeNonQuery($sql, $params);
        $dividend_id = getDatabaseConnection()->lastInsertId();
        
        commitTransaction();
        
        // Create audit log for dividend creation
        createAuditLog(null, 'create', 'dividends', $dividend_id, null, [
            'member_id' => $input['member_id'],
            'amount' => $input['amount'],
            'financial_year' => $input['financial_year'],
            'status' => $input['status']
        ]);
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Dividend recorded successfully',
            'dividend_id' => $dividend_id
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handlePutRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dividend ID is required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Update dividend
    $sql = "UPDATE dividends SET 
            amount = ?, 
            financial_year = ?, 
            payment_date = ?, 
            status = ?, 
            payment_reference = ?
            WHERE id = ?";
    
    $params = [
        $input['amount'] ?? null,
        $input['financial_year'] ?? null,
        $input['payment_date'] ?? null,
        $input['status'] ?? null,
        $input['payment_reference'] ?? null,
        $_GET['id']
    ];
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, $params);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Dividend not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for dividend update
        createAuditLog(null, 'update', 'dividends', $_GET['id'], null, $input);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Dividend updated successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dividend ID is required']);
        return;
    }
    
    $sql = "DELETE FROM dividends WHERE id = ?";
    
    try {
        beginTransaction();
        
        $rows_affected = executeNonQuery($sql, [$_GET['id']]);
        
        if ($rows_affected === 0) {
            http_response_code(404);
            echo json_encode(['error' => 'Dividend not found']);
            return;
        }
        
        commitTransaction();
        
        // Create audit log for dividend deletion
        createAuditLog(null, 'delete', 'dividends', $_GET['id'], null, null);
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Dividend deleted successfully'
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        throw $e;
    }
}
?>