<?php
// MFT GROUP Contributions API
// This API handles member contributions and related operations

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Credentials: true");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Origin, X-Requested-With, Content-Type, Accept, Authorization");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include database configuration
require_once dirname(__DIR__) . '/database/config.php';

// Initialize database connection
global $pdo;
$pdo = getDatabaseConnection();

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    // Get query parameters
    $member_id = isset($_GET['member_id']) ? $_GET['member_id'] : null;
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
    $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
    $status = isset($_GET['status']) ? $_GET['status'] : null;
    $date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
    $date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;
    $month = isset($_GET['month']) ? (int)$_GET['month'] : null;
    $year = isset($_GET['year']) ? (int)$_GET['year'] : null;
    
    if ($member_id) {
        // Get contributions for a specific member
        getMemberContributions($member_id, $limit, $offset, $status, $date_from, $date_to, $month, $year);
    } else {
        // Get all contributions (admin only) with filters
        getAllContributions($limit, $offset, $status, $date_from, $date_to, $month, $year);
    }
}

function handlePostRequest() {
    // Add a new contribution
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    addContribution($input);
}

function handlePutRequest() {
    // Update a contribution
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    updateContribution($input);
}

function handleDeleteRequest() {
    // Delete a contribution
    $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$id) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing ID parameter']);
        return;
    }
    
    deleteContribution($id);
}

function getMemberContributions($member_id, $limit, $offset, $status = null, $date_from = null, $date_to = null, $month = null, $year = null) {
    global $pdo;
    
    try {
        // First, get the member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$member_id]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Build the query with all filters
        $sql = "SELECT c.*, m.first_name, m.last_name, m.member_id, p.reference as payment_reference 
                FROM contributions c
                JOIN members m ON c.member_id = m.id
                LEFT JOIN payments p ON c.payment_id = p.id
                WHERE c.member_id = ?";
        
        $params = [$member_numeric_id];
        
        // Add status filter
        if ($status && $status !== 'all') {
            $sql .= " AND c.status = ?";
            $params[] = $status;
        }
        
        // Add date range filters
        if ($date_from) {
            $sql .= " AND c.contribution_date >= ?";
            $params[] = $date_from;
        }
        
        if ($date_to) {
            $sql .= " AND c.contribution_date <= ?";
            $params[] = $date_to;
        }
        
        // Add month filter
        if ($month) {
            $sql .= " AND MONTH(c.contribution_date) = ?";
            $params[] = $month;
        }
        
        // Add year filter
        if ($year) {
            $sql .= " AND YEAR(c.contribution_date) = ?";
            $params[] = $year;
        }
        
        $sql .= " ORDER BY c.contribution_date DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination with same filters
        $count_sql = "SELECT COUNT(*) as total FROM contributions WHERE member_id = ?";
        $count_params = [$member_numeric_id];
        
        // Add status filter to count query
        if ($status && $status !== 'all') {
            $count_sql .= " AND status = ?";
            $count_params[] = $status;
        }
        
        // Add date range filters to count query
        if ($date_from) {
            $count_sql .= " AND contribution_date >= ?";
            $count_params[] = $date_from;
        }
        
        if ($date_to) {
            $count_sql .= " AND contribution_date <= ?";
            $count_params[] = $date_to;
        }
        
        // Add month filter to count query
        if ($month) {
            $count_sql .= " AND MONTH(contribution_date) = ?";
            $count_params[] = $month;
        }
        
        // Add year filter to count query
        if ($year) {
            $count_sql .= " AND YEAR(contribution_date) = ?";
            $count_params[] = $year;
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get member's total contribution balance with same filters
        $balance_sql = "SELECT COALESCE(SUM(amount), 0) as total_balance 
                        FROM contributions 
                        WHERE member_id = ? AND status = 'Confirmed'";
        
        // Add date range filters to balance query
        if ($date_from) {
            $balance_sql .= " AND contribution_date >= ?";
            $balance_params = [$member_numeric_id, $date_from];
        } else {
            $balance_params = [$member_numeric_id];
        }
        
        if ($date_to) {
            $balance_sql .= " AND contribution_date <= ?";
            $balance_params[] = $date_to;
        }
        
        // Add month filter to balance query
        if ($month) {
            $balance_sql .= " AND MONTH(contribution_date) = ?";
            $balance_params[] = $month;
        }
        
        // Add year filter to balance query
        if ($year) {
            $balance_sql .= " AND YEAR(contribution_date) = ?";
            $balance_params[] = $year;
        }
        
        $balance_stmt = $pdo->prepare($balance_sql);
        $balance_stmt->execute($balance_params);
        $balance_result = $balance_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'contributions' => $results,
            'total' => (int)$count_result['total'],
            'total_balance' => (float)$balance_result['total_balance'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function getAllContributions($limit, $offset, $status = null, $date_from = null, $date_to = null, $month = null, $year = null) {
    global $pdo;
    
    try {
        // Build the query with all filters
        $sql = "SELECT c.*, m.first_name, m.last_name, m.member_id, p.reference as payment_reference 
                FROM contributions c
                JOIN members m ON c.member_id = m.id
                LEFT JOIN payments p ON c.payment_id = p.id";
        
        $where_conditions = [];
        $params = [];
        
        // Add status filter
        if ($status && $status !== 'all') {
            $where_conditions[] = "c.status = ?";
            $params[] = $status;
        }
        
        // Add date range filters
        if ($date_from) {
            $where_conditions[] = "c.contribution_date >= ?";
            $params[] = $date_from;
        }
        
        if ($date_to) {
            $where_conditions[] = "c.contribution_date <= ?";
            $params[] = $date_to;
        }
        
        // Add month filter
        if ($month) {
            $where_conditions[] = "MONTH(c.contribution_date) = ?";
            $params[] = $month;
        }
        
        // Add year filter
        if ($year) {
            $where_conditions[] = "YEAR(c.contribution_date) = ?";
            $params[] = $year;
        }
        
        // Add WHERE clause if there are conditions
        if (!empty($where_conditions)) {
            $sql .= " WHERE " . implode(" AND ", $where_conditions);
        }
        
        $sql .= " ORDER BY c.contribution_date DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get total count for pagination with same filters
        $count_sql = "SELECT COUNT(*) as total FROM contributions c";
        
        // Add JOIN for member filtering if needed
        if (!empty($where_conditions)) {
            $count_sql .= " JOIN members m ON c.member_id = m.id";
        }
        
        $count_params = [];
        $count_where_conditions = [];
        
        // Add status filter to count query
        if ($status && $status !== 'all') {
            $count_where_conditions[] = "c.status = ?";
            $count_params[] = $status;
        }
        
        // Add date range filters to count query
        if ($date_from) {
            $count_where_conditions[] = "c.contribution_date >= ?";
            $count_params[] = $date_from;
        }
        
        if ($date_to) {
            $count_where_conditions[] = "c.contribution_date <= ?";
            $count_params[] = $date_to;
        }
        
        // Add month filter to count query
        if ($month) {
            $count_where_conditions[] = "MONTH(c.contribution_date) = ?";
            $count_params[] = $month;
        }
        
        // Add year filter to count query
        if ($year) {
            $count_where_conditions[] = "YEAR(c.contribution_date) = ?";
            $count_params[] = $year;
        }
        
        // Add WHERE clause to count query if there are conditions
        if (!empty($count_where_conditions)) {
            $count_sql .= " WHERE " . implode(" AND ", $count_where_conditions);
        }
        
        $count_stmt = $pdo->prepare($count_sql);
        $count_stmt->execute($count_params);
        $count_result = $count_stmt->fetch(PDO::FETCH_ASSOC);
        
        $response = [
            'contributions' => $results,
            'total' => (int)$count_result['total'],
            'limit' => $limit,
            'offset' => $offset
        ];
        
        echo json_encode($response);
    } catch (Exception $e) {
        throw $e;
    }
}

function addContribution($data) {
    global $pdo;
    
    try {
        // Required fields
        $required_fields = ['member_id', 'amount', 'contribution_date'];
        foreach ($required_fields as $field) {
            if (!isset($data[$field])) {
                http_response_code(400);
                echo json_encode(['error' => "Missing required field: $field"]);
                return;
            }
        }
        
        // Get member's numeric ID
        $member_check_sql = "SELECT id FROM members WHERE member_id = ?";
        $member_stmt = $pdo->prepare($member_check_sql);
        $member_stmt->execute([$data['member_id']]);
        $member_result = $member_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$member_result) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        $member_numeric_id = $member_result['id'];
        
        // Start transaction
        $pdo->beginTransaction();
        
        try {
            // Insert payment record first
            $payment_sql = "INSERT INTO payments (reference, account_id, amount, payment_method, status, processed_at) 
                            VALUES (?, ?, ?, ?, 'Completed', ?)";
            
            // Use provided reference number or generate a unique one
            $payment_reference = isset($data['reference']) && !empty($data['reference']) ? $data['reference'] : ('PAY-' . date('Y') . '-' . strtoupper(substr(uniqid(), -6)));
            $account_id = 1; // Assuming savings account ID is 1
            $payment_method = isset($data['payment_method']) ? $data['payment_method'] : 'M-PESA'; // Default to M-PESA
            
            $payment_stmt = $pdo->prepare($payment_sql);
            $payment_stmt->execute([
                $payment_reference,
                $account_id,
                $data['amount'],
                $payment_method,
                date('Y-m-d H:i:s')
            ]);
            
            $payment_id = $pdo->lastInsertId();
            
            // Insert contribution record
            $contribution_sql = "INSERT INTO contributions 
                                (member_id, payment_id, amount, contribution_date, period_month, period_year, status, confirmed_by, confirmed_at) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $period_month = date('n', strtotime($data['contribution_date']));
            $period_year = date('Y', strtotime($data['contribution_date']));
            $status = isset($data['status']) ? $data['status'] : 'Confirmed';
            
            // Get a valid admin user ID instead of hardcoding 1
            $confirmed_by = isset($data['confirmed_by']) ? $data['confirmed_by'] : getValidAdminId();
            $confirmed_at = date('Y-m-d H:i:s');
            
            $contribution_stmt = $pdo->prepare($contribution_sql);
            $result = $contribution_stmt->execute([
                $member_numeric_id,
                $payment_id,
                $data['amount'],
                $data['contribution_date'],
                $period_month,
                $period_year,
                $status,
                $confirmed_by,
                $confirmed_at
            ]);
            
            if ($result) {
                // Check if this payment resolves any missed contributions
                checkAndResolveMissedContributions($member_numeric_id, $data['amount'], $data['contribution_date']);
                
                $pdo->commit();
                
                // Get the inserted record
                $select_sql = "SELECT c.*, m.first_name, m.last_name, m.member_id, p.reference as payment_reference 
                               FROM contributions c
                               JOIN members m ON c.member_id = m.id
                               JOIN payments p ON c.payment_id = p.id
                               WHERE c.payment_id = ?";
                $select_stmt = $pdo->prepare($select_sql);
                $select_stmt->execute([$payment_id]);
                $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
                
                http_response_code(201);
                echo json_encode([
                    'message' => 'Contribution added successfully',
                    'contribution' => $record
                ]);
            } else {
                $pdo->rollback();
                throw new Exception('Failed to add contribution');
            }
        } catch (Exception $e) {
            $pdo->rollback();
            throw $e;
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function updateContribution($data) {
    global $pdo;
    
    try {
        // Start transaction
        $pdo->beginTransaction();
        
        try {
            // Update contribution
            $sql = "UPDATE contributions c 
                    JOIN payments p ON c.payment_id = p.id
                    SET 
                    c.amount = ?, 
                    c.contribution_date = ?, 
                    c.period_month = ?, 
                    c.period_year = ?, 
                    c.status = ?, 
                    c.confirmed_by = ?, 
                    c.confirmed_at = ?,
                    c.updated_at = NOW(),
                    p.reference = COALESCE(?, p.reference),
                    p.amount = COALESCE(?, p.amount)
                    WHERE c.id = ?";
            
            $period_month = date('n', strtotime($data['contribution_date']));
            $period_year = date('Y', strtotime($data['contribution_date']));
            $confirmed_by = isset($data['confirmed_by']) ? $data['confirmed_by'] : getValidAdminId(); // Use valid admin ID
            $confirmed_at = isset($data['confirmed_at']) ? $data['confirmed_at'] : date('Y-m-d H:i:s');
            
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute([
                isset($data['amount']) ? $data['amount'] : null,
                isset($data['contribution_date']) ? $data['contribution_date'] : null,
                $period_month,
                $period_year,
                isset($data['status']) ? $data['status'] : null,
                $confirmed_by,
                $confirmed_at,
                isset($data['reference']) ? $data['reference'] : null, // Reference number for payment
                isset($data['amount']) ? $data['amount'] : null, // Amount for payment
                $data['id']
            ]);
            
            if ($result) {
                // If status was changed to Confirmed, check if this resolves any missed contributions
                if (isset($data['status']) && $data['status'] === 'Confirmed') {
                    // Get member ID and contribution date for this contribution
                    $select_sql = "SELECT member_id, amount, contribution_date FROM contributions WHERE id = ?";
                    $select_stmt = $pdo->prepare($select_sql);
                    $select_stmt->execute([$data['id']]);
                    $contribution = $select_stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($contribution) {
                        checkAndResolveMissedContributions($contribution['member_id'], $contribution['amount'], $contribution['contribution_date']);
                    }
                }
                
                $pdo->commit();
                
                // Get the updated record
                $select_sql = "SELECT c.*, m.first_name, m.last_name, m.member_id, p.reference as payment_reference 
                               FROM contributions c
                               JOIN members m ON c.member_id = m.id
                               JOIN payments p ON c.payment_id = p.id
                               WHERE c.id = ?";
                $select_stmt = $pdo->prepare($select_sql);
                $select_stmt->execute([$data['id']]);
                $record = $select_stmt->fetch(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'message' => 'Contribution updated successfully',
                    'contribution' => $record
                ]);
            } else {
                $pdo->rollback();
                throw new Exception('Failed to update contribution');
            }
        } catch (Exception $e) {
            $pdo->rollback();
            throw $e;
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function deleteContribution($id) {
    global $pdo;
    
    try {
        // Start transaction
        $pdo->beginTransaction();
        
        try {
            // Get the contribution details before deleting
            $select_sql = "SELECT c.*, p.id as payment_id FROM contributions c LEFT JOIN payments p ON c.payment_id = p.id WHERE c.id = ?";
            $select_stmt = $pdo->prepare($select_sql);
            $select_stmt->execute([$id]);
            $contribution = $select_stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$contribution) {
                http_response_code(404);
                echo json_encode(['error' => 'Contribution not found']);
                $pdo->rollback();
                return;
            }
            
            // Delete the contribution
            $sql = "DELETE FROM contributions WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute([$id]);
            
            if ($result) {
                // Also delete the associated payment
                if ($contribution['payment_id']) {
                    $payment_sql = "DELETE FROM payments WHERE id = ?";
                    $payment_stmt = $pdo->prepare($payment_sql);
                    $payment_stmt->execute([$contribution['payment_id']]);
                }
                
                $pdo->commit();
                echo json_encode(['message' => 'Contribution deleted successfully']);
            } else {
                $pdo->rollback();
                throw new Exception('Failed to delete contribution');
            }
        } catch (Exception $e) {
            $pdo->rollback();
            throw $e;
        }
    } catch (Exception $e) {
        throw $e;
    }
}

function checkAndResolveMissedContributions($member_id, $amount, $contribution_date) {
    global $pdo;
    
    try {
        // Check if there are any missed contributions for this member
        $missed_sql = "SELECT * FROM missed_contributions 
                       WHERE member_id = ? AND status = 'Unpaid' 
                       ORDER BY missed_date ASC";
        $missed_stmt = $pdo->prepare($missed_sql);
        $missed_stmt->execute([$member_id]);
        $missed_contributions = $missed_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $remaining_amount = $amount;
        
        foreach ($missed_contributions as $missed) {
            if ($remaining_amount <= 0) {
                break;
            }
            
            // If the contribution amount covers this missed contribution
            if ($remaining_amount >= $missed['amount']) {
                // Mark this missed contribution as paid
                $update_sql = "UPDATE missed_contributions 
                               SET status = 'Paid', payment_date = ? 
                               WHERE id = ?";
                $update_stmt = $pdo->prepare($update_sql);
                $update_stmt->execute([$contribution_date, $missed['id']]);
                
                $remaining_amount -= $missed['amount'];
            } else {
                // Partial payment - this would require more complex logic
                // For now, we'll just leave it as unpaid
                break;
            }
        }
        
        // If there's still remaining amount, it might be a regular contribution
        // This is normal behavior
    } catch (Exception $e) {
        // Log the error but don't fail the main operation
        error_log("Error checking missed contributions: " . $e->getMessage());
    }
}

// Function to get a valid admin ID for confirmed_by field
function getValidAdminId() {
    global $pdo;
    
    try {
        // Try to get an admin user
        $sql = "SELECT m.id FROM members m 
                JOIN member_roles mr ON m.id = mr.member_id 
                JOIN roles r ON mr.role_id = r.id 
                WHERE r.name = 'Admin' 
                LIMIT 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return $result['id'];
        }
        
        // If no admin found, try to get any active member
        $sql = "SELECT id FROM members WHERE status = 'Active' LIMIT 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return $result['id'];
        }
        
        // If no active member found, return the first member
        $sql = "SELECT id FROM members ORDER BY id ASC LIMIT 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return $result['id'];
        }
        
        // If no members at all, return 0 (this should never happen in a properly set up system)
        return 0;
    } catch (Exception $e) {
        // Log the error and return 0 as fallback
        error_log("Error getting valid admin ID: " . $e->getMessage());
        return 0;
    }
}
?>