<?php
// MFT GROUP Admin Users API
// This API provides admin user management functionality

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            handleGetRequest();
            break;
        case 'POST':
            handlePostRequest();
            break;
        case 'PUT':
            handlePutRequest();
            break;
        case 'DELETE':
            handleDeleteRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handleGetRequest() {
    if (isset($_GET['id'])) {
        // Get specific admin
        getAdmin($_GET['id']);
    } else if (isset($_GET['validate_code'])) {
        // Validate access code
        validateAccessCode($_GET['validate_code']);
    } else {
        // Get all admins
        getAdmins();
    }
}

function handlePostRequest() {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Check if this is for updating an existing admin
    if (isset($_GET['id'])) {
        updateAdmin($_GET['id'], $input);
        return;
    }
    
    // Otherwise, this is for creating a new admin
    createAdmin($input);
}

function handlePutRequest() {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Check if ID is provided
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Admin ID is required']);
        return;
    }
    
    updateAdmin($_GET['id'], $input);
}

function handleDeleteRequest() {
    if (!isset($_GET['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Admin ID is required']);
        return;
    }
    
    // Check if a specific role is to be deleted
    if (isset($_GET['role_id'])) {
        deleteSpecificRole($_GET['id'], $_GET['role_id']);
    } else {
        // Delete all admin roles (previous behavior)
        deleteAdmin($_GET['id']);
    }
}

function getAdmins() {
    try {
        // Get all members with admin roles (allowing multiple roles per member)
        $sql = "SELECT m.id, m.member_id, m.first_name, m.last_name, m.email, m.phone, m.role, 
                       mr.role_id, r.name as role_name, m.created_at, aac.access_code
                FROM members m
                JOIN member_roles mr ON m.id = mr.member_id
                JOIN roles r ON mr.role_id = r.id
                LEFT JOIN admin_access_codes aac ON m.id = aac.member_id
                WHERE r.name IN ('Admin', 'Treasurer', 'Secretary', 'Chairman', 'Assistant Chairman', 'Assistant Secretary', 'Loans Officer')
                ORDER BY m.created_at DESC";
        
        $admins = executeQuery($sql);
        
        // Format the response to group roles by member
        $formattedAdmins = [];
        $memberRoles = [];
        
        // Group roles by member
        foreach ($admins as $admin) {
            $memberId = $admin['id'];
            
            if (!isset($memberRoles[$memberId])) {
                $memberRoles[$memberId] = [
                    'id' => $admin['id'],
                    'member_id' => $admin['member_id'],
                    'name' => $admin['first_name'] . ' ' . $admin['last_name'],
                    'first_name' => $admin['first_name'],
                    'last_name' => $admin['last_name'],
                    'email' => $admin['email'],
                    'phone' => $admin['phone'],
                    'roles' => [],
                    'access_codes' => [],
                    'created_at' => $admin['created_at']
                ];
            }
            
            // Add role
            $memberRoles[$memberId]['roles'][] = [
                'role_id' => $admin['role_id'],
                'role_name' => $admin['role_name']
            ];
            
            // Add access code if exists and not already added
            if ($admin['access_code'] && !in_array($admin['access_code'], $memberRoles[$memberId]['access_codes'])) {
                $memberRoles[$memberId]['access_codes'][] = $admin['access_code'];
            }
        }
        
        // Format for output
        foreach ($memberRoles as $member) {
            $formattedAdmins[] = [
                'id' => $member['id'],
                'member_id' => $member['member_id'],
                'name' => $member['name'],
                'first_name' => $member['first_name'],
                'last_name' => $member['last_name'],
                'email' => $member['email'],
                'phone' => $member['phone'],
                'roles' => $member['roles'],
                'role_names' => array_column($member['roles'], 'role_name'),
                'access_codes' => $member['access_codes'],
                'created_at' => $member['created_at']
            ];
        }
        
        http_response_code(200);
        echo json_encode([
            'count' => count($formattedAdmins),
            'admins' => $formattedAdmins
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve admins: ' . $e->getMessage()]);
    }
}

function getAdmin($id) {
    try {
        // Get specific member with all admin roles
        $sql = "SELECT m.id, m.member_id, m.first_name, m.last_name, m.email, m.phone, m.role, 
                       mr.role_id, r.name as role_name, m.created_at, aac.access_code
                FROM members m
                JOIN member_roles mr ON m.id = mr.member_id
                JOIN roles r ON mr.role_id = r.id
                LEFT JOIN admin_access_codes aac ON m.id = aac.member_id
                WHERE m.id = ? AND r.name IN ('Admin', 'Treasurer', 'Secretary', 'Chairman', 'Assistant Chairman', 'Assistant Secretary', 'Loans Officer')";
        
        $adminRoles = executeQuery($sql, [$id]);
        
        if (!empty($adminRoles)) {
            // Group all roles for this member
            $mainAdmin = $adminRoles[0];
            $roles = [];
            $accessCodes = [];
            
            foreach ($adminRoles as $role) {
                $roles[] = [
                    'role_id' => $role['role_id'],
                    'role_name' => $role['role_name']
                ];
                
                if ($role['access_code'] && !in_array($role['access_code'], $accessCodes)) {
                    $accessCodes[] = $role['access_code'];
                }
            }
            
            $formattedAdmin = [
                'id' => $mainAdmin['id'],
                'member_id' => $mainAdmin['member_id'],
                'name' => $mainAdmin['first_name'] . ' ' . $mainAdmin['last_name'],
                'first_name' => $mainAdmin['first_name'],
                'last_name' => $mainAdmin['last_name'],
                'email' => $mainAdmin['email'],
                'phone' => $mainAdmin['phone'],
                'roles' => $roles,
                'role_names' => array_column($roles, 'role_name'),
                'access_codes' => $accessCodes,
                'created_at' => $mainAdmin['created_at']
            ];
            
            http_response_code(200);
            echo json_encode($formattedAdmin);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Admin not found']);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to retrieve admin: ' . $e->getMessage()]);
    }
}

function validateAccessCode($accessCode) {
    try {
        // Validate access code
        $sql = "SELECT m.id, m.member_id, m.first_name, m.last_name, m.email, m.role, 
                       mr.role_id, r.name as role_name, aac.access_code
                FROM admin_access_codes aac
                JOIN members m ON aac.member_id = m.id
                JOIN member_roles mr ON m.id = mr.member_id
                JOIN roles r ON mr.role_id = r.id
                WHERE aac.access_code = ?";
        
        $admin = fetchSingleRow($sql, [$accessCode]);
        
        if ($admin) {
            $formattedAdmin = [
                'id' => $admin['id'],
                'member_id' => $admin['member_id'],
                'name' => $admin['first_name'] . ' ' . $admin['last_name'],
                'first_name' => $admin['first_name'],
                'last_name' => $admin['last_name'],
                'email' => $admin['email'],
                'role' => $admin['role_name'],
                'role_id' => $admin['role_id'],
                'access_code' => $admin['access_code']
            ];
            
            http_response_code(200);
            echo json_encode([
                'valid' => true,
                'admin' => $formattedAdmin
            ]);
        } else {
            http_response_code(404);
            echo json_encode([
                'valid' => false,
                'error' => 'Invalid access code'
            ]);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to validate access code: ' . $e->getMessage()]);
    }
}

function createAdmin($data) {
    try {
        // Validate required fields
        if (empty($data['member_id']) || empty($data['role_id']) || empty($data['access_code'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Member ID, role ID, and access code are required']);
            return;
        }
        
        // Check if member exists
        $memberSql = "SELECT id, first_name, last_name FROM members WHERE member_id = ?";
        $member = fetchSingleRow($memberSql, [$data['member_id']]);
        
        if (!$member) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Check if this specific role is already assigned to this member
        $checkSql = "SELECT * FROM member_roles WHERE member_id = ? AND role_id = ?";
        $existingRole = fetchSingleRow($checkSql, [$member['id'], $data['role_id']]);
        
        if ($existingRole) {
            http_response_code(409);
            echo json_encode(['error' => 'This member already has this specific role assigned.']);
            return;
        }
        
        // Check if access code is unique
        $codeCheckSql = "SELECT * FROM admin_access_codes WHERE access_code = ?";
        $existingCode = fetchSingleRow($codeCheckSql, [$data['access_code']]);
        
        if ($existingCode) {
            http_response_code(409);
            echo json_encode(['error' => 'Access code already exists. Please use a unique code.']);
            return;
        }
        
        beginTransaction();
        
        // Check if member already has admin role BEFORE assigning new role
        $adminRoles = [1, 2, 3, 4, 5, 6, 7]; // Admin role IDs
        $hadAdminRole = false;
        
        $checkAdminSql = "SELECT mr.role_id, r.name as role_name 
                         FROM member_roles mr 
                         JOIN roles r ON mr.role_id = r.id 
                         WHERE mr.member_id = ? AND mr.role_id IN (" . implode(',', $adminRoles) . ")";
        $adminRolesResult = executeQuery($checkAdminSql, [$member['id']]);
        
        if (!empty($adminRolesResult)) {
            $hadAdminRole = true;
        }
        
        // Assign role to member (allowing multiple roles)
        $roleSql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
        executeNonQuery($roleSql, [$member['id'], $data['role_id'], $member['id']]);
        
        // Update member role field to show primary role (if not already set to admin role)
        // Only update if they didn't have an admin role before
        if (!$hadAdminRole) {
            $updateSql = "UPDATE members SET role = (SELECT name FROM roles WHERE id = ?) WHERE id = ?";
            executeNonQuery($updateSql, [$data['role_id'], $member['id']]);
        }
        
        // Store access code
        $accessCodeSql = "INSERT INTO admin_access_codes (member_id, access_code) VALUES (?, ?)";
        executeNonQuery($accessCodeSql, [$member['id'], $data['access_code']]);
        
        commitTransaction();
        
        // Return the created admin
        $newAdmin = [
            'id' => $member['id'],
            'member_id' => $data['member_id'],
            'name' => $member['first_name'] . ' ' . $member['last_name'],
            'first_name' => $member['first_name'],
            'last_name' => $member['last_name'],
            'role_id' => $data['role_id'],
            'access_code' => $data['access_code']
        ];
        
        http_response_code(201);
        echo json_encode([
            'message' => 'Admin role assigned successfully',
            'admin' => $newAdmin
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to assign admin role: ' . $e->getMessage()]);
    }
}

function updateAdmin($id, $data) {
    try {
        // Check if admin exists
        $adminSql = "SELECT m.id, m.first_name, m.last_name FROM members m 
                     WHERE m.id = ?";
        $admin = fetchSingleRow($adminSql, [$id]);
        
        if (!$admin) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        beginTransaction();
        
        // Check if member already has admin role BEFORE adding new role
        $adminRoles = [1, 2, 3, 4, 5, 6, 7]; // Admin role IDs
        $hadAdminRole = false;
        
        $checkAdminSql = "SELECT mr.role_id FROM member_roles mr WHERE mr.member_id = ? AND mr.role_id IN (" . implode(',', $adminRoles) . ")";
        $adminRolesResult = executeQuery($checkAdminSql, [$id]);
        
        if (!empty($adminRolesResult)) {
            $hadAdminRole = true;
        }
        
        // Add new role if provided
        if (!empty($data['role_id'])) {
            // Check if this specific role is already assigned to this member
            $checkSql = "SELECT * FROM member_roles WHERE member_id = ? AND role_id = ?";
            $existingRole = fetchSingleRow($checkSql, [$id, $data['role_id']]);
            
            if (!$existingRole) {
                // Assign new role to member
                $roleSql = "INSERT INTO member_roles (member_id, role_id, assigned_by) VALUES (?, ?, ?)";
                executeNonQuery($roleSql, [$id, $data['role_id'], $id]);
                
                // Update member role field if they didn't have an admin role before
                if (!$hadAdminRole) {
                    $updateSql = "UPDATE members SET role = (SELECT name FROM roles WHERE id = ?) WHERE id = ?";
                    executeNonQuery($updateSql, [$data['role_id'], $id]);
                }
            }
        }
        
        // Update access code if provided
        if (!empty($data['access_code'])) {
            // Check if access code is unique
            $codeCheckSql = "SELECT * FROM admin_access_codes WHERE access_code = ? AND member_id != ?";
            $existingCode = fetchSingleRow($codeCheckSql, [$data['access_code'], $id]);
            
            if ($existingCode) {
                http_response_code(409);
                echo json_encode(['error' => 'Access code already exists. Please use a unique code.']);
                return;
            }
            
            // Update or insert access code
            $accessCodeCheckSql = "SELECT * FROM admin_access_codes WHERE member_id = ?";
            $existingAccessCode = fetchSingleRow($accessCodeCheckSql, [$id]);
            
            if ($existingAccessCode) {
                $accessCodeSql = "UPDATE admin_access_codes SET access_code = ? WHERE member_id = ?";
                executeNonQuery($accessCodeSql, [$data['access_code'], $id]);
            } else {
                $accessCodeSql = "INSERT INTO admin_access_codes (member_id, access_code) VALUES (?, ?)";
                executeNonQuery($accessCodeSql, [$id, $data['access_code']]);
            }
        }
        
        commitTransaction();
        
        // Return the updated admin
        $updatedAdmin = [
            'id' => $admin['id'],
            'name' => $admin['first_name'] . ' ' . $admin['last_name'],
            'first_name' => $admin['first_name'],
            'last_name' => $admin['last_name']
        ];
        
        if (!empty($data['role_id'])) {
            $roleSql = "SELECT name FROM roles WHERE id = ?";
            $role = fetchSingleRow($roleSql, [$data['role_id']]);
            if ($role) {
                $updatedAdmin['role_id'] = $data['role_id'];
                $updatedAdmin['role'] = $role['name'];
            }
        }
        
        if (!empty($data['access_code'])) {
            $updatedAdmin['access_code'] = $data['access_code'];
        }
        
        http_response_code(200);
        echo json_encode([
            'message' => 'Admin role assigned successfully',
            'admin' => $updatedAdmin
        ]);
    } catch (Exception $e) {
        rollbackTransaction();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to assign admin role: ' . $e->getMessage()]);
    }
}

function deleteAdmin($id) {
    try {
        // Check if admin exists
        $adminSql = "SELECT m.id, m.first_name, m.last_name FROM members m 
                     JOIN member_roles mr ON m.id = mr.member_id 
                     WHERE m.id = ?";
        $admin = fetchSingleRow($adminSql, [$id]);
        
        if (!$admin) {
            http_response_code(404);
            echo json_encode(['error' => 'Admin not found']);
            return;
        }
        
        beginTransaction();
        
        // Remove all admin roles but keep member role
        $roleSql = "DELETE FROM member_roles WHERE member_id = ? AND role_id IN (1, 2, 3, 4, 5, 6, 7)";
        executeNonQuery($roleSql, [$id]);
        
        // Remove all access codes for this member
        $accessCodeSql = "DELETE FROM admin_access_codes WHERE member_id = ?";
        executeNonQuery($accessCodeSql, [$id]);
        
        // Reset member role field to 'member' only if they have no other roles
        $checkRolesSql = "SELECT COUNT(*) as role_count FROM member_roles WHERE member_id = ?";
        $roleCount = fetchSingleRow($checkRolesSql, [$id]);
        
        if ($roleCount && $roleCount['role_count'] == 0) {
            $updateSql = "UPDATE members SET role = 'member' WHERE id = ?";
            executeNonQuery($updateSql, [$id]);
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode(['message' => 'All admin role(s) removed successfully']);
    } catch (Exception $e) {
        rollbackTransaction();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to remove admin role(s): ' . $e->getMessage()]);
    }
}

// Delete a specific role from a member
function deleteSpecificRole($memberId, $roleId) {
    try {
        // Check if member exists
        $adminSql = "SELECT m.id, m.first_name, m.last_name FROM members m WHERE m.id = ?";
        $admin = fetchSingleRow($adminSql, [$memberId]);
        
        if (!$admin) {
            http_response_code(404);
            echo json_encode(['error' => 'Member not found']);
            return;
        }
        
        // Check if member has this specific role
        $roleCheckSql = "SELECT * FROM member_roles WHERE member_id = ? AND role_id = ?";
        $existingRole = fetchSingleRow($roleCheckSql, [$memberId, $roleId]);
        
        if (!$existingRole) {
            http_response_code(404);
            echo json_encode(['error' => 'Member does not have this role']);
            return;
        }
        
        beginTransaction();
        
        // Remove specific role
        $roleSql = "DELETE FROM member_roles WHERE member_id = ? AND role_id = ?";
        executeNonQuery($roleSql, [$memberId, $roleId]);
        
        // Remove access codes associated with this role (for now, we'll remove all access codes for this member)
        // In a more advanced implementation, we might want to track which access code belongs to which role
        $accessCodeSql = "DELETE FROM admin_access_codes WHERE member_id = ?";
        executeNonQuery($accessCodeSql, [$memberId]);
        
        // Check if member has any remaining admin roles
        $checkAdminRolesSql = "SELECT COUNT(*) as admin_role_count FROM member_roles WHERE member_id = ? AND role_id IN (1, 2, 3, 4, 5, 6, 7)";
        $adminRoleCount = fetchSingleRow($checkAdminRolesSql, [$memberId]);
        
        // If no admin roles remain, update member role field
        if ($adminRoleCount && $adminRoleCount['admin_role_count'] == 0) {
            $updateSql = "UPDATE members SET role = 'member' WHERE id = ?";
            executeNonQuery($updateSql, [$memberId]);
        }
        
        commitTransaction();
        
        http_response_code(200);
        echo json_encode(['message' => 'Role removed successfully']);
    } catch (Exception $e) {
        rollbackTransaction();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to remove role: ' . $e->getMessage()]);
    }
}
?>