<?php
// MFT GROUP Admin Authentication API
// This API provides authentication functionality for admin users using access codes

// Enable CORS for frontend access
header("Access-Control-Allow-Origin: *");
// Add credentials support for CORS
header("Access-Control-Allow-Credentials: true");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, GET");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Include database configuration
require_once __DIR__ . '/../database/config.php';

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handlePostRequest();
            break;
        case 'GET':
            handleGetRequest();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

function handlePostRequest() {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON data']);
        return;
    }
    
    // Check if access code is provided
    if (empty($input['access_code'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Access code is required']);
        return;
    }
    
    authenticateAdmin($input['access_code']);
}

function handleGetRequest() {
    // For GET requests, check if admin is logged in
    session_start();
    
    if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
        http_response_code(200);
        echo json_encode([
            'authenticated' => true,
            'admin_id' => $_SESSION['admin_id'],
            'admin_name' => $_SESSION['admin_name'],
            'admin_email' => $_SESSION['admin_email'],
            'admin_role' => $_SESSION['admin_role'],
            'admin_role_id' => $_SESSION['admin_role_id'],
            'access_code' => $_SESSION['access_code']
        ]);
    } else {
        http_response_code(401);
        echo json_encode([
            'authenticated' => false,
            'error' => 'Not authenticated'
        ]);
    }
}

function authenticateAdmin($accessCode) {
    try {
        // Validate access code
        $sql = "SELECT m.id, m.member_id, m.first_name, m.last_name, m.email, m.role, 
                       mr.role_id, r.name as role_name, aac.access_code
                FROM admin_access_codes aac
                JOIN members m ON aac.member_id = m.id
                JOIN member_roles mr ON m.id = mr.member_id
                JOIN roles r ON mr.role_id = r.id
                WHERE aac.access_code = ?";
        
        $admin = fetchSingleRow($sql, [$accessCode]);
        
        if ($admin) {
            // Create session data
            session_start();
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_name'] = $admin['first_name'] . ' ' . $admin['last_name'];
            $_SESSION['admin_email'] = $admin['email'];
            $_SESSION['admin_role'] = $admin['role_name'];
            $_SESSION['admin_role_id'] = $admin['role_id'];
            $_SESSION['access_code'] = $admin['access_code'];
            
            $formattedAdmin = [
                'id' => $admin['id'],
                'member_id' => $admin['member_id'],
                'name' => $admin['first_name'] . ' ' . $admin['last_name'],
                'first_name' => $admin['first_name'],
                'last_name' => $admin['last_name'],
                'email' => $admin['email'],
                'role' => $admin['role_name'],
                'role_id' => $admin['role_id'],
                'access_code' => $admin['access_code']
            ];
            
            http_response_code(200);
            echo json_encode([
                'valid' => true,
                'message' => 'Authentication successful',
                'admin' => $formattedAdmin
            ]);
        } else {
            http_response_code(401);
            echo json_encode([
                'valid' => false,
                'error' => 'Invalid access code'
            ]);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to authenticate: ' . $e->getMessage()]);
    }
}
?>