<?php
// Test script to verify project update functionality
require_once 'database/config.php';

echo "<h1>Project Update Test</h1>";

// Test database connection
try {
    $pdo = getDatabaseConnection();
    echo "<p>✅ Database connection successful</p>";
} catch (Exception $e) {
    echo "<p>❌ Database connection failed: " . $e->getMessage() . "</p>";
    exit;
}

// Get a test project to update
try {
    $stmt = $pdo->query("SELECT id, name, description, location, completion_date, status, total_shares, share_price FROM projects LIMIT 1");
    $project = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($project) {
        echo "<p>✅ Found test project with ID: " . $project['id'] . "</p>";
        echo "<p>Current project data:</p>";
        echo "<pre>" . htmlspecialchars(print_r($project, true)) . "</pre>";
        
        // Test updating the project
        $new_name = "Updated Project Name " . date('Y-m-d H:i:s');
        $new_description = "Updated description for testing";
        $new_location = "Updated Location";
        
        $update_data = [
            'name' => $new_name,
            'description' => $new_description,
            'location' => $new_location
        ];
        
        echo "<p>Updating project with data:</p>";
        echo "<pre>" . htmlspecialchars(print_r($update_data, true)) . "</pre>";
        
        // Test the update query directly
        $updates = [];
        $params = [];
        
        $updatable_fields = [
            'name', 'description', 'location', 
            'completion_date', 'total_shares', 'share_price', 
            'status'
        ];
        
        foreach ($updatable_fields as $field) {
            if (isset($update_data[$field])) {
                $updates[] = "$field = ?";
                $params[] = $update_data[$field];
            }
        }
        
        if (!empty($updates)) {
            $params[] = $project['id'];
            $sql = "UPDATE projects SET " . implode(', ', $updates) . ", updated_at = CURRENT_TIMESTAMP WHERE id = ?";
            
            echo "<p>Executing query: $sql</p>";
            echo "<p>With parameters: " . json_encode($params) . "</p>";
            
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute($params);
            
            if ($result) {
                $affected_rows = $stmt->rowCount();
                echo "<p>✅ Update query executed successfully</p>";
                echo "<p>Affected rows: $affected_rows</p>";
                
                // Verify the update
                $verify_stmt = $pdo->prepare("SELECT id, name, description, location FROM projects WHERE id = ?");
                $verify_stmt->execute([$project['id']]);
                $updated_project = $verify_stmt->fetch(PDO::FETCH_ASSOC);
                
                echo "<p>Updated project data:</p>";
                echo "<pre>" . htmlspecialchars(print_r($updated_project, true)) . "</pre>";
                
                if ($updated_project['name'] === $new_name) {
                    echo "<p>✅ Project name updated correctly</p>";
                } else {
                    echo "<p>❌ Project name not updated correctly</p>";
                }
            } else {
                echo "<p>❌ Update query failed</p>";
            }
        } else {
            echo "<p>❌ No valid fields to update</p>";
        }
    } else {
        echo "<p>❌ No projects found in database</p>";
    }
} catch (Exception $e) {
    echo "<p>❌ Error testing project update: " . $e->getMessage() . "</p>";
}

// Test the API endpoint
echo "<h2>Testing API endpoint</h2>";

$project_id = 1; // Test with project ID 1
$update_data = [
    'name' => 'API Test Project Update',
    'description' => 'Updated via API test',
    'location' => 'API Test Location'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, "http://localhost/MFT/api/projects.php?id=" . $project_id);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($update_data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'Content-Length: ' . strlen(json_encode($update_data))
]);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);

$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$content_type = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
$error = curl_error($ch);

curl_close($ch);

echo "<p>HTTP Code: " . $http_code . "</p>";
echo "<p>Content Type: " . $content_type . "</p>";

if ($error) {
    echo "<p>❌ cURL Error: " . $error . "</p>";
} else {
    echo "<p>✅ API Request completed</p>";
    echo "<p>Response: " . htmlspecialchars($response) . "</p>";
}
?>