// MFT GROUP Real-Time Synchronization Service
// This script handles synchronization of data between front office and back office in real-time

class MFTRTSyncService {
    constructor() {
        this.apiBaseURL = getApiBaseUrl();
        this.syncInterval = 30000; // 30 seconds
        this.isSyncing = false;
        this.syncCallbacks = [];
        this.lastSyncTime = 0;
    }
    
    /**
     * Initialize the synchronization service
     * @param {string} role - The role of the current user ('admin' or 'member')
     * @param {string} memberId - The member ID (for member role)
     */
    init(role, memberId = null) {
        this.role = role;
        this.memberId = memberId;
        
        console.log(`RTSyncService initialized for ${role} role`);
        
        // Start periodic synchronization
        this.startPeriodicSync();
        
        // Listen for manual sync requests
        document.addEventListener('syncRequested', () => {
            this.syncData();
        });
    }
    
    /**
     * Start periodic synchronization
     */
    startPeriodicSync() {
        // Clear any existing interval
        if (this.syncIntervalId) {
            clearInterval(this.syncIntervalId);
        }
        
        // Set up periodic sync
        this.syncIntervalId = setInterval(() => {
            this.syncData();
        }, this.syncInterval);
        
        console.log('Periodic synchronization started');
    }
    
    /**
     * Stop periodic synchronization
     */
    stopPeriodicSync() {
        if (this.syncIntervalId) {
            clearInterval(this.syncIntervalId);
            this.syncIntervalId = null;
            console.log('Periodic synchronization stopped');
        }
    }
    
    /**
     * Synchronize data between front office and back office
     */
    async syncData() {
        if (this.isSyncing) {
            console.log('Sync already in progress, skipping');
            return;
        }
        
        this.isSyncing = true;
        console.log('Starting data synchronization...');
        
        try {
            // Get pending notifications
            const notifications = await this.getPendingNotifications();
            
            if (notifications && notifications.length > 0) {
                console.log(`Processing ${notifications.length} notifications`);
                
                // Process each notification
                for (const notification of notifications) {
                    await this.processNotification(notification);
                }
                
                // Update last sync time
                this.lastSyncTime = Math.floor(Date.now() / 1000);
            } else {
                console.log('No pending notifications to process');
            }
            
            // Notify callbacks
            this.notifySyncComplete();
            
        } catch (error) {
            console.error('Error during synchronization:', error);
            this.notifySyncError(error);
        } finally {
            this.isSyncing = false;
        }
    }
    
    /**
     * Get pending notifications for the current user
     */
    async getPendingNotifications() {
        try {
            const params = new URLSearchParams({
                target: this.role,
                last_check: this.lastSyncTime
            });
            
            if (this.memberId && this.role === 'member') {
                params.append('member_id', this.memberId);
            }
            
            const response = await fetch(`${this.apiBaseURL}/websocket-server.php?${params}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            const data = await response.json();
            
            if (data.status === 'notifications' && data.notifications) {
                return data.notifications;
            }
            
            return [];
        } catch (error) {
            console.error('Error getting notifications:', error);
            return [];
        }
    }
    
    /**
     * Process a notification
     * @param {object} notification - The notification to process
     */
    async processNotification(notification) {
        console.log('Processing notification:', notification);
        
        // Handle different types of notifications
        switch (notification.type) {
            case 'settings_update':
                await this.handleSettingsUpdate(notification.data);
                break;
            case 'contribution_update':
                await this.handleContributionUpdate(notification.data);
                break;
            case 'loan_update':
                await this.handleLoanUpdate(notification.data);
                break;
            case 'member_update':
                await this.handleMemberUpdate(notification.data);
                break;
            case 'project_update':
                await this.handleProjectUpdate(notification.data);
                break;
            case 'dashboard_update':
                await this.handleDashboardUpdate(notification.data);
                break;
            default:
                console.log('Unknown notification type:', notification.type);
        }
    }
    
    /**
     * Handle settings update notification
     * @param {object} data - The settings update data
     */
    async handleSettingsUpdate(data) {
        console.log('Handling settings update:', data);
        
        // In a real implementation, you would update the UI based on the changed settings
        // For now, we'll just dispatch an event
        this.dispatchEvent('settingsUpdated', data);
    }
    
    /**
     * Handle contribution update notification
     * @param {object} data - The contribution update data
     */
    async handleContributionUpdate(data) {
        console.log('Handling contribution update:', data);
        
        // Update UI elements
        this.updateContributionDisplay(data);
        
        // Dispatch event
        this.dispatchEvent('contributionUpdated', data);
    }
    
    /**
     * Handle loan update notification
     * @param {object} data - The loan update data
     */
    async handleLoanUpdate(data) {
        console.log('Handling loan update:', data);
        
        // Update UI elements
        this.updateLoanDisplay(data);
        
        // Dispatch event
        this.dispatchEvent('loanUpdated', data);
    }
    
    /**
     * Handle member update notification
     * @param {object} data - The member update data
     */
    async handleMemberUpdate(data) {
        console.log('Handling member update:', data);
        
        // Update UI elements
        this.updateMemberDisplay(data);
        
        // Dispatch event
        this.dispatchEvent('memberUpdated', data);
    }
    
    /**
     * Handle project update notification
     * @param {object} data - The project update data
     */
    async handleProjectUpdate(data) {
        console.log('Handling project update:', data);
        
        // Update UI elements
        this.updateProjectDisplay(data);
        
        // Dispatch event
        this.dispatchEvent('projectUpdated', data);
    }
    
    /**
     * Handle dashboard update notification
     * @param {object} data - The dashboard update data
     */
    async handleDashboardUpdate(data) {
        console.log('Handling dashboard update:', data);
        
        // Update UI elements
        this.updateDashboardDisplay(data);
        
        // Dispatch event
        this.dispatchEvent('dashboardUpdated', data);
    }
    
    /**
     * Update contribution display
     * @param {object} data - The contribution data
     */
    updateContributionDisplay(data) {
        // This would update specific UI elements related to contributions
        console.log('Updating contribution display with:', data);
    }
    
    /**
     * Update loan display
     * @param {object} data - The loan data
     */
    updateLoanDisplay(data) {
        // This would update specific UI elements related to loans
        console.log('Updating loan display with:', data);
    }
    
    /**
     * Update member display
     * @param {object} data - The member data
     */
    updateMemberDisplay(data) {
        // This would update specific UI elements related to members
        console.log('Updating member display with:', data);
    }
    
    /**
     * Update project display
     * @param {object} data - The project data
     */
    updateProjectDisplay(data) {
        // This would update specific UI elements related to projects
        console.log('Updating project display with:', data);
    }
    
    /**
     * Update dashboard display
     * @param {object} data - The dashboard data
     */
    updateDashboardDisplay(data) {
        // This would update specific UI elements related to the dashboard
        console.log('Updating dashboard display with:', data);
    }
    
    /**
     * Dispatch a custom event
     * @param {string} eventName - The name of the event
     * @param {object} data - The data to send with the event
     */
    dispatchEvent(eventName, data) {
        const event = new CustomEvent(eventName, {
            detail: data
        });
        document.dispatchEvent(event);
    }
    
    /**
     * Register a callback for sync completion
     * @param {function} callback - The callback function
     */
    onSyncComplete(callback) {
        if (typeof callback === 'function') {
            this.syncCallbacks.push(callback);
        }
    }
    
    /**
     * Notify all callbacks that sync is complete
     */
    notifySyncComplete() {
        this.syncCallbacks.forEach(callback => {
            try {
                callback(null);
            } catch (error) {
                console.error('Error in sync callback:', error);
            }
        });
    }
    
    /**
     * Notify all callbacks that sync encountered an error
     * @param {Error} error - The error that occurred
     */
    notifySyncError(error) {
        this.syncCallbacks.forEach(callback => {
            try {
                callback(error);
            } catch (err) {
                console.error('Error in sync callback:', err);
            }
        });
    }
    
    /**
     * Request immediate synchronization
     */
    requestSync() {
        // Dispatch event to request sync
        const event = new CustomEvent('syncRequested');
        document.dispatchEvent(event);
    }
    
    /**
     * Update synchronization interval
     * @param {number} interval - The new interval in milliseconds
     */
    updateSyncInterval(interval) {
        this.syncInterval = interval;
        this.startPeriodicSync();
    }
}

// Create a global instance
const mftRTSync = new MFTRTSyncService();

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = MFTRTSyncService;
}