// Public Settings Loader for MFT GROUP
// This script loads system settings and makes them available to the public website

class MFTPublicSettings {
    constructor() {
        this.apiBaseURL = getApiBaseUrl();
        this.settings = {};
        this.isLoaded = false;
    }
    
    // Load all settings from the API
    async loadSettings() {
        try {
            console.log('Loading public settings from API...');
            const response = await fetch(`${this.apiBaseURL}/settings.php`);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            console.log('Public settings API response:', result);
            
            // More robust check for settings data
            if (result.hasOwnProperty('settings') && Array.isArray(result.settings)) {
                try {
                    // Convert array to object for easier lookup
                    this.settings = {};
                    result.settings.forEach(setting => {
                        this.settings[setting.setting_key] = setting.setting_value;
                    });
                    this.isLoaded = true;
                    console.log('Public settings loaded:', this.settings);
                    this.applySettingsToPage();
                    return true;
                } catch (processingError) {
                    console.error('Error processing public settings:', processingError);
                    console.error('Error details:', {
                        message: processingError.message,
                        stack: processingError.stack
                    });
                    console.error('Settings data:', result.settings);
                    throw new Error('Failed to process public settings: ' + processingError.message);
                }
            } else {
                console.error('Failed to load settings:', result.error || 'Invalid response format');
                console.error('Response received:', result);
                return false;
            }
        } catch (error) {
            console.error('Error loading public settings:', error);
            console.error('Error details:', {
                message: error.message,
                stack: error.stack
            });
            return false;
        }
    }
    
    // Apply settings to the page elements
    applySettingsToPage() {
        try {
            // Update organization name in header
            const orgNameElements = document.querySelectorAll('.org-name, .organization-name');
            orgNameElements.forEach(element => {
                if (this.settings.orgName) {
                    element.textContent = this.settings.orgName;
                }
            });
            
            // Update organization acronym
            const orgAcronymElements = document.querySelectorAll('.org-acronym');
            orgAcronymElements.forEach(element => {
                if (this.settings.orgAcronym) {
                    element.textContent = this.settings.orgAcronym;
                }
            });
            
            // Update organization description
            const orgDescriptionElements = document.querySelectorAll('.org-description');
            orgDescriptionElements.forEach(element => {
                if (this.settings.orgDescription) {
                    element.textContent = this.settings.orgDescription;
                }
            });
            
            // Update contact information
            const orgEmailElements = document.querySelectorAll('.org-email');
            orgEmailElements.forEach(element => {
                if (this.settings.orgEmail) {
                    element.textContent = this.settings.orgEmail;
                    element.href = `mailto:${this.settings.orgEmail}`;
                }
            });
            
            const orgPhoneElements = document.querySelectorAll('.org-phone');
            orgPhoneElements.forEach(element => {
                if (this.settings.orgPhone) {
                    element.textContent = this.settings.orgPhone;
                    element.href = `tel:${this.settings.orgPhone}`;
                }
            });
            
            // Update organization address
            const orgAddressElements = document.querySelectorAll('.org-address');
            orgAddressElements.forEach(element => {
                if (this.settings.orgAddress) {
                    element.textContent = this.settings.orgAddress;
                }
            });
            
            // Update officials information
            this.updateOfficialsInformation();
            
            // Update founded date
            const foundedDateElements = document.querySelectorAll('.founded-date');
            foundedDateElements.forEach(element => {
                if (this.settings.foundedDate) {
                    const date = new Date(this.settings.foundedDate);
                    element.textContent = date.getFullYear();
                }
            });
            
            console.log('Settings applied to page elements');
        } catch (error) {
            console.error('Error applying settings to page:', error);
            console.error('Error details:', {
                message: error.message,
                stack: error.stack
            });
            console.error('Settings available:', this.settings);
        }
    }
    
    // Update officials information on the page
    updateOfficialsInformation() {
        try {
            // Chairman
            this.updateOfficialInfo('chairman', 'Chairman');
            
            // Assistant Chairman
            this.updateOfficialInfo('assistantChairman', 'Assistant Chairman');
            
            // Secretary
            this.updateOfficialInfo('secretary', 'Secretary');
            
            // Organising Secretary
            this.updateOfficialInfo('organisingSecretary', 'Organising Secretary');
            
            // Treasurer
            this.updateOfficialInfo('treasurer', 'Treasurer');
            
            // Loans Officer
            this.updateOfficialInfo('loansOfficer', 'Loans Officer');
        } catch (error) {
            console.error('Error updating officials information:', error);
            console.error('Error details:', {
                message: error.message,
                stack: error.stack
            });
        }
    }
    
    // Update a specific official's information
    updateOfficialInfo(prefix, role) {
        try {
            // Update full name
            const fullNameElements = document.querySelectorAll(`.${prefix.toLowerCase()}-name`);
            const fullName = this.settings[`${prefix}FullName`];
            if (fullName) {
                fullNameElements.forEach(element => {
                    element.textContent = fullName;
                });
            }
            
            // Update email
            const emailElements = document.querySelectorAll(`.${prefix.toLowerCase()}-email`);
            const email = this.settings[`${prefix}Email`];
            if (email) {
                emailElements.forEach(element => {
                    element.textContent = email;
                    element.href = `mailto:${email}`;
                });
            }
            
            // Update phone
            const phoneElements = document.querySelectorAll(`.${prefix.toLowerCase()}-phone`);
            const phone = this.settings[`${prefix}Phone`];
            if (phone) {
                phoneElements.forEach(element => {
                    element.textContent = phone;
                    element.href = `tel:${phone}`;
                });
            }
            
            // Update profile photo
            const photoElements = document.querySelectorAll(`.${prefix.toLowerCase()}-photo`);
            const photo = this.settings[`${prefix}ProfilePhoto`];
            if (photo) {
                photoElements.forEach(element => {
                    if (element.tagName === 'IMG') {
                        element.src = photo;
                    } else {
                        element.style.backgroundImage = `url(${photo})`;
                    }
                });
            }
            
            // Update term dates
            const termStartElements = document.querySelectorAll(`.${prefix.toLowerCase()}-term-start`);
            const termStart = this.settings[`${prefix}TermStart`];
            if (termStart) {
                termStartElements.forEach(element => {
                    const date = new Date(termStart);
                    element.textContent = date.toLocaleDateString();
                });
            }
            
            const termEndElements = document.querySelectorAll(`.${prefix.toLowerCase()}-term-end`);
            const termEnd = this.settings[`${prefix}TermEnd`];
            if (termEnd) {
                termEndElements.forEach(element => {
                    const date = new Date(termEnd);
                    element.textContent = date.toLocaleDateString();
                });
            }
            
            // Update role description
            const roleDescriptionElements = document.querySelectorAll(`.${prefix.toLowerCase()}-role-description`);
            const roleDescription = this.settings[`${prefix}RoleDescription`];
            if (roleDescription) {
                roleDescriptionElements.forEach(element => {
                    element.textContent = roleDescription;
                });
            }
        } catch (error) {
            console.error(`Error updating official info for ${prefix}:`, error);
            console.error('Error details:', {
                message: error.message,
                stack: error.stack
            });
            console.error('Settings available:', this.settings);
        }
    }
    
    // Get a specific setting value
    getSetting(key, defaultValue = null) {
        if (this.settings[key] !== undefined) {
            return this.settings[key];
        }
        return defaultValue;
    }
    
    // Check if settings are loaded
    isSettingsLoaded() {
        return this.isLoaded;
    }
}

// Initialize and load settings when the DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    window.mftPublicSettings = new MFTPublicSettings();
    window.mftPublicSettings.loadSettings().catch(error => {
        console.error('Error initializing public settings:', error);
        console.error('Error details:', {
            message: error.message,
            stack: error.stack
        });
        // Don't show alert for public settings, just log the error
    });
});