// Export utility functions for MFT GROUP admin pages

/**
 * Export data to CSV format
 * @param {Array} data - Array of objects to export
 * @param {string} filename - Name of the file to export
 */
function exportToCSV(data, filename) {
    if (!data || data.length === 0) {
        alert('No data to export');
        return;
    }
    
    // Create CSV content
    const headers = Object.keys(data[0]);
    const csvContent = [
        headers.join(','),
        ...data.map(row => 
            headers.map(fieldName => {
                let field = row[fieldName];
                if (typeof field === 'string' && field.includes(',')) {
                    field = `"${field}"`;
                }
                return field;
            }).join(',')
        )
    ].join('\n');
    
    // Create download link
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

/**
 * Export data to Excel format using the server API
 * @param {Array} data - Array of objects to export
 * @param {string} filename - Name of the file to export
 * @param {string} title - Title for the export
 * @param {string} exportType - Type of export (members, loans, etc.)
 */
async function exportToExcel(data, filename, title, exportType = 'generic') {
    try {
        // Show loading indicator
        const originalTitle = document.title;
        document.title = 'Exporting...';
        
        const response = await fetch(getApiUrl('export.php'), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                export_type: exportType,
                format: 'excel',
                data: data,
                filename: filename,
                title: title
            })
        });
        
        // Reset title
        document.title = originalTitle;
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'Export failed');
        }
        
        // For Excel export, the server will handle the download directly
        // We don't need to do anything here as the file will be downloaded automatically
    } catch (error) {
        console.error('Error exporting to Excel:', error);
        alert('Error exporting to Excel: ' + error.message);
    }
}

/**
 * Export data to PDF format using the server API
 * @param {Array} data - Array of objects to export
 * @param {string} filename - Name of the file to export
 * @param {string} title - Title for the PDF document
 * @param {string} exportType - Type of export (members, loans, etc.)
 */
async function exportToPDF(data, filename, title, exportType = 'generic') {
    try {
        // Show loading indicator
        const originalTitle = document.title;
        document.title = 'Exporting...';
        
        const response = await fetch(getApiUrl('export.php'), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                export_type: exportType,
                format: 'pdf',
                data: data,
                filename: filename,
                title: title
            })
        });
        
        // Reset title
        document.title = originalTitle;
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'Export failed');
        }
        
        // For PDF export, the server will handle the download directly
        // We don't need to do anything here as the file will be downloaded automatically
    } catch (error) {
        console.error('Error exporting to PDF:', error);
        alert('Error exporting to PDF: ' + error.message);
    }
}

/**
 * Generic export function that handles different export types
 * @param {Array} data - Array of objects to export
 * @param {string} type - Export type (csv, excel, pdf)
 * @param {string} filename - Name of the file to export
 * @param {string} title - Title for the export (used in PDF)
 * @param {string} exportType - Type of export (members, loans, etc.)
 */
function exportData(data, type, filename, title = 'Export', exportType = 'generic') {
    try {
        switch (type.toLowerCase()) {
            case 'csv':
                exportToCSV(data, filename);
                break;
            case 'excel':
                exportToExcel(data, filename, title, exportType);
                break;
            case 'pdf':
                exportToPDF(data, filename, title, exportType);
                break;
            default:
                alert('Unsupported export format');
        }
    } catch (error) {
        console.error('Error exporting data:', error);
        alert('Error exporting data. Please try again.');
    }
}

// Export functions for use in other modules
window.exportData = exportData;
window.exportToCSV = exportToCSV;
window.exportToExcel = exportToExcel;
window.exportToPDF = exportToPDF;