// MFT GROUP Documents Real-Time Updates
// This script handles real-time updates for the documents page

// Pagination variables
let currentPage = 1;
const documentsPerPage = 6;
let allDocuments = [];

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Verify that we have access to the proper document object
    if (typeof document === 'undefined' || typeof document.getElementById === 'undefined' || typeof document.createElement === 'undefined') {
        console.error('DOM is not properly available in current context');
        return;
    }
    
    // Get member ID from localStorage or other source
    const memberId = getMemberId();
    
    if (memberId) {
        // Initialize real-time client
        mftRealTime.connect('member', memberId);
        
        // Register update callback
        mftRealTime.onUpdate(function(data, pageType) {
            if (pageType === 'documents') {
                updateDocumentsRealTime(data);
            }
        });
        
        // Load initial data
        loadDocumentsData(memberId);
        
        // Set up form submission handler
        setupFormHandlers();
        
        console.log('Real-time updates started for documents page');
    } else {
        console.warn('Member ID not found. Real-time updates not started.');
    }
});

// Function to get member ID (this would be implemented based on your auth system)
function getMemberId() {
    // For now, we'll try to get it from localStorage
    // In a real implementation, this would come from your authentication system
    const memberId = localStorage.getItem('memberId') || '1'; // Default to member ID 1 for testing
    console.log('getMemberId() returning:', memberId);
    return memberId;
}

// Function to load initial documents data
async function loadDocumentsData(memberId) {
    try {
        // Verify that we have access to the proper document object
        if (typeof document === 'undefined' || typeof document.getElementById === 'undefined') {
            throw new Error('DOM is not properly available in current context');
        }
        
        // Show loading state
        showLoadingState();
        
        // Use the proper API URL construction
        const apiUrl = getApiUrl(`documents.php?member_id=${encodeURIComponent(memberId)}`);
        console.log('Fetching documents from:', apiUrl);
        
        // Fetch documents data from API
        const response = await fetch(apiUrl);
        console.log('Documents API response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        
        const data = await response.json();
        console.log('Documents API response:', data);
        
        if (data) {
            // Store all documents for pagination
            allDocuments = data.documents || [];
            currentPage = 1;
            
            // Update the page with the fetched data
            console.log('Updating documents page with data:', data);
            updateDocumentsPage(data);
            console.log('Documents page updated successfully');
        } else {
            // Show error state
            console.log('No data received, showing error state');
            showErrorState();
        }
    } catch (error) {
        console.error('Error loading documents data:', error);
        // Show a more detailed error message
        // Verify that we have access to the proper document object
        if (typeof document !== 'undefined' && document.getElementById) {
            const documentsContainer = document.getElementById('documents-container');
            if (documentsContainer) {
                documentsContainer.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-exclamation-circle"></i>
                        <h3>Error Loading Documents</h3>
                        <p>There was an error loading your documents: ${error.message}</p>
                        <p>Please try again later or contact support.</p>
                        <button class="btn btn-primary" onclick="refreshDocuments()">Retry</button>
                    </div>`;
            }
        }
    }
}

// Function to show loading state
function showLoadingState() {
    // Update documents container with loading state
    const documentsContainer = document.getElementById('documents-container');
    if (documentsContainer) {
        documentsContainer.innerHTML = '<div class="loading-state"><i class="fas fa-spinner"></i><div>Loading documents...</div></div>';
    }
}

// Function to show error state
function showErrorState() {
    // Update documents container with error state
    const documentsContainer = document.getElementById('documents-container');
    if (documentsContainer) {
        documentsContainer.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-exclamation-circle"></i>
                <h3>Error Loading Documents</h3>
                <p>There was an error loading your documents. Please try again later.</p>
                <button class="btn btn-primary" onclick="refreshDocuments()">Retry</button>
            </div>`;
    }
}

// Function to set up form handlers
function setupFormHandlers() {
    const uploadForm = document.getElementById('document-upload-form');
    if (uploadForm) {
        uploadForm.addEventListener('submit', function(event) {
            event.preventDefault();
            handleDocumentUpload();
        });
        
        // Add event listener for adding more document fields
        const addDocumentBtn = document.getElementById('add-document');
        if (addDocumentBtn) {
            addDocumentBtn.addEventListener('click', function() {
                addDocumentField();
            });
        }
        
        // Add event listeners for file input changes
        setupFileInputListeners();
        
        // Add event listeners for remove buttons
        document.querySelectorAll('.remove-document').forEach(button => {
            button.addEventListener('click', function() {
                removeDocumentField(this.dataset.index);
            });
        });
    }
}

// Function to add a new document field
function addDocumentField() {
    // Find the first hidden document group
    for (let i = 2; i <= 3; i++) {
        const group = document.getElementById(`document-group-${i}`);
        if (group && group.style.display === 'none') {
            group.style.display = 'block';
            break;
        }
    }
    
    // Hide the add button if all fields are shown
    const allGroups = document.querySelectorAll('.document-group');
    const visibleGroups = Array.from(allGroups).filter(group => group.style.display !== 'none');
    if (visibleGroups.length >= 3) {
        document.getElementById('add-document').style.display = 'none';
    }
}

// Function to remove a document field
function removeDocumentField(index) {
    const group = document.getElementById(`document-group-${index}`);
    if (group) {
        // Clear the file input and description
        const fileInput = group.querySelector('.document-file');
        const descriptionInput = group.querySelector('.document-description');
        if (fileInput) fileInput.value = '';
        if (descriptionInput) descriptionInput.value = '';
        
        // Hide the group
        group.style.display = 'none';
        
        // Update the file input label
        document.getElementById(`file-input-label-${index}`).textContent = 'Choose File';
        
        // Show the add button
        document.getElementById('add-document').style.display = 'inline-block';
    }
}

// Function to set up file input listeners
function setupFileInputListeners() {
    document.querySelectorAll('.document-file').forEach(input => {
        input.addEventListener('change', function(e) {
            const index = this.dataset.index;
            const fileName = e.target.files[0] ? e.target.files[0].name : 'Choose File';
            document.getElementById(`file-input-label-${index}`).textContent = fileName;
        });
    });
}

// Function to handle document upload
function handleDocumentUpload() {
    console.log('=== HANDLE DOCUMENT UPLOAD ===');
    
    const documentType = document.getElementById('document-type').value;
    
    if (!documentType) {
        alert('Please select a document type');
        return;
    }
    
    // Collect all documents to upload
    const documentsToUpload = [];
    const documentGroups = document.querySelectorAll('.document-group');
    
    documentGroups.forEach(group => {
        if (group.style.display !== 'none') {
            const fileInput = group.querySelector('.document-file');
            const descriptionInput = group.querySelector('.document-description');
            
            if (fileInput && fileInput.files[0]) {
                documentsToUpload.push({
                    file: fileInput.files[0],
                    description: descriptionInput ? descriptionInput.value : ''
                });
            }
        }
    });
    
    if (documentsToUpload.length === 0) {
        alert('Please select at least one document file');
        return;
    }
    
    console.log('Documents to upload:', documentsToUpload);
    
    // Show uploading state
    const submitButton = document.querySelector('#document-upload-form .btn-primary');
    const originalText = submitButton.innerHTML;
    submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Uploading...';
    submitButton.disabled = true;
    
    // Upload documents
    uploadDocuments(documentType, documentsToUpload)
        .then(() => {
            console.log('=== UPLOAD SUCCESS ===');
            // Reset button
            submitButton.innerHTML = originalText;
            submitButton.disabled = false;
            
            // Show success message
            alert(`Successfully uploaded ${documentsToUpload.length} document(s)!`);
            
            // Reset form
            document.getElementById('document-upload-form').reset();
            
            // Reset file input labels
            document.getElementById('file-input-label-1').textContent = 'Choose File';
            if (document.getElementById('file-input-label-2')) {
                document.getElementById('file-input-label-2').textContent = 'Choose File';
            }
            if (document.getElementById('file-input-label-3')) {
                document.getElementById('file-input-label-3').textContent = 'Choose File';
            }
            
            // Hide additional document fields
            document.getElementById('document-group-2').style.display = 'none';
            document.getElementById('document-group-3').style.display = 'none';
            document.getElementById('add-document').style.display = 'inline-block';
            
            // Refresh documents list after a short delay to ensure database commit
            setTimeout(() => {
                const memberId = getMemberId();
                console.log('=== UPLOAD COMPLETION ===');
                console.log('Member ID from getMemberId():', memberId);
                console.log('Refreshing documents list for member:', memberId);
                
                // Also log what's in localStorage for debugging
                console.log('localStorage memberId:', localStorage.getItem('memberId'));
                
                if (memberId) {
                    // Force a refresh by calling the API directly
                    refreshDocumentsList(memberId);
                } else {
                    console.error('No member ID found, cannot refresh documents list');
                }
            }, 1500); // 1.5 second delay to ensure database commit
        })
        .catch(error => {
            console.log('=== UPLOAD ERROR ===');
            // Reset button
            submitButton.innerHTML = originalText;
            submitButton.disabled = false;
            
            console.error('Upload error:', error);
            alert('Error uploading documents: ' + error.message);
        });
}

// New function to refresh the documents list
async function refreshDocumentsList(memberId) {
    console.log('Refreshing documents list for member ID:', memberId);
    
    try {
        // Use the proper API URL construction
        const apiUrl = getApiUrl(`documents.php?member_id=${encodeURIComponent(memberId)}`);
        console.log('Fetching documents from:', apiUrl);
        
        // Fetch documents data from API
        const response = await fetch(apiUrl);
        console.log('Documents API response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error response:', errorText);
            throw new Error(`HTTP error! status: ${response.status}, message: ${errorText}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const errorText = await response.text();
            console.error('Non-JSON response:', errorText);
            throw new Error('Response is not JSON: ' + errorText);
        }
        
        const data = await response.json();
        console.log('Documents API response:', data);
        
        if (data) {
            // Store all documents for pagination
            allDocuments = data.documents || [];
            currentPage = 1;
            
            // Update the page with the fetched data
            console.log('Updating documents page with data:', data);
            updateDocumentsPage(data);
            console.log('Documents page updated successfully');
        } else {
            // Show error state
            console.log('No data received, showing error state');
            showErrorState();
        }
    } catch (error) {
        console.error('Error refreshing documents list:', error);
        // Show a more detailed error message
        if (typeof document !== 'undefined' && document.getElementById) {
            const documentsContainer = document.getElementById('documents-container');
            if (documentsContainer) {
                documentsContainer.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-exclamation-circle"></i>
                        <h3>Error Loading Documents</h3>
                        <p>There was an error loading your documents: ${error.message}</p>
                        <p>Please try again later or contact support.</p>
                        <button class="btn btn-primary" onclick="refreshDocuments()">Retry</button>
                    </div>`;
            }
        }
    }
}

// Function to upload documents to the server
async function uploadDocuments(documentType, documents) {
    const memberId = getMemberId();
    if (!memberId) {
        throw new Error('Member ID not found');
    }
    
    // Get member numeric ID
    const memberResponse = await fetch(getApiUrl(`members.php?id=${encodeURIComponent(memberId)}`));
    if (!memberResponse.ok) {
        throw new Error('Failed to get member information');
    }
    
    const memberData = await memberResponse.json();
    const memberNumericId = memberData.id;
    
    // Upload each document
    const uploadPromises = documents.map(doc => {
        const formData = new FormData();
        formData.append('document', doc.file);
        formData.append('title', doc.file.name);
        formData.append('description', doc.description);
        formData.append('category', documentType);
        formData.append('uploaded_by', memberNumericId);
        formData.append('related_member_id', memberNumericId);
        
        return fetch(getApiUrl('documents.php'), {
            method: 'POST',
            body: formData
        }).then(response => {
            if (!response.ok) {
                return response.text().then(text => {
                    throw new Error(`Failed to upload document: ${text}`);
                });
            }
            return response.json();
        });
    });
    
    // Wait for all uploads to complete
    await Promise.all(uploadPromises);
}

// Function to create a document record in the database
async function createDocumentRecord(documentData) {
    const response = await fetch(getApiUrl('documents.php'), {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(documentData)
    });
    
    if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Failed to create document record: ${errorText}`);
    }
    
    return response.json();
}

// Function to update the documents page with real-time data
function updateDocumentsRealTime(data) {
    console.log('Updating documents with real-time data:', data);
    
    // Update documents information
    if (data.documents) {
        // Store all documents for pagination
        allDocuments = data.documents;
        currentPage = 1;
        updateDocumentsPage(data);
    }
}

// Function to update the documents page with data
function updateDocumentsPage(data) {
    // Ensure we're in the right context and the DOM is available
    if (typeof document === 'undefined' || typeof document.getElementById === 'undefined' || typeof document.createElement === 'undefined') {
        console.error('DOM is not available in current context');
        console.error('document:', document);
        console.error('document.getElementById:', typeof document.getElementById);
        console.error('document.createElement:', typeof document.createElement);
        return;
    }
    
    const documentsContainer = document.getElementById('documents-container');
    const paginationControls = document.getElementById('pagination-controls');
    if (!documentsContainer) {
        console.error('Documents container not found');
        return;
    }
    
    console.log('=== UPDATE DOCUMENTS PAGE ===');
    console.log('Data received:', data);
    
    if (!data.documents || data.documents.length === 0) {
        console.log('No documents to display, showing empty state');
        documentsContainer.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-file"></i>
                <h3>No Documents Found</h3>
                <p>You don't have any documents uploaded yet.</p>
                <p>Upload documents to get started.</p>
            </div>`;
        
        // Hide pagination controls
        if (paginationControls) {
            paginationControls.style.display = 'none';
        }
        return;
    }
    
    // Store all documents for pagination
    allDocuments = data.documents;
    
    // Calculate pagination
    const totalPages = Math.ceil(allDocuments.length / documentsPerPage);
    const startIndex = (currentPage - 1) * documentsPerPage;
    const endIndex = Math.min(startIndex + documentsPerPage, allDocuments.length);
    const documentsToShow = allDocuments.slice(startIndex, endIndex);
    
    // Clear existing documents
    documentsContainer.innerHTML = '';
    console.log(`Displaying ${documentsToShow.length} documents (page ${currentPage} of ${totalPages})`);
    
    // Add new documents
    documentsToShow.forEach((doc, index) => {
        console.log(`Processing document ${index + 1}:`, doc);
        console.log('About to create element, document object:', document);
        
        try {
            // Create the document card element
            const documentCard = document.createElement('div');
            documentCard.className = 'document-card';
            documentCard.innerHTML = `
                <div class="document-header">
                    <div class="document-icon">${getDocumentIcon(doc.title)}</div>
                    <h3 class="document-title">${doc.title || 'Untitled Document'}</h3>
                </div>
                <p class="document-description">
                    ${doc.description || 'No description available'}
                </p>
                <div class="document-meta">
                    <span>${doc.category || 'General'} (${formatFileSize(doc.file_size || 0)})</span>
                    <span>Uploaded: ${formatDate(doc.created_at || new Date())}</span>
                </div>
                <span class="document-status ${getStatusClass(doc.status)}">
                    ${formatStatus(doc.status)}
                </span>
                <div class="document-actions" style="display: flex; gap: 10px; margin-top: 15px;">
                    <button class="btn btn-primary" style="flex: 1;" onclick="downloadDocument(${doc.id}, '${doc.file_path || ''}')">
                        <i class="fas fa-download"></i> Download
                    </button>
                    ${(doc.uploaded_by == doc.related_member_id) ? 
                    `<button class="btn btn-danger" style="flex: 1;" onclick="deleteDocument(${doc.id}, '${doc.title || 'Untitled Document'}')">
                        <i class="fas fa-trash"></i> Delete
                    </button>` : ''}
                </div>
            `;
            documentsContainer.appendChild(documentCard);
        } catch (error) {
            console.error(`Error creating document card for document ${index + 1}:`, error);
            console.error('document object at time of error:', document);
            console.error('document.createElement at time of error:', typeof document.createElement);
        }
    });
    
    // Update pagination controls
    if (paginationControls) {
        paginationControls.style.display = 'block';
        document.getElementById('page-info').textContent = `Page ${currentPage} of ${totalPages}`;
        
        // Update button states
        document.getElementById('prev-page').disabled = (currentPage === 1);
        document.getElementById('next-page').disabled = (currentPage === totalPages);
    }
    
    console.log('Documents page updated successfully');
}

// Function to change page
function changePage(direction) {
    const totalPages = Math.ceil(allDocuments.length / documentsPerPage);
    const newPage = currentPage + direction;
    
    if (newPage >= 1 && newPage <= totalPages) {
        currentPage = newPage;
        // Re-render with pagination
        updateDocumentsPage({ documents: allDocuments });
    }
}

// Helper function to get document icon based on document title
function getDocumentIcon(documentTitle) {
    if (!documentTitle) return '📁';
    
    const title = documentTitle.toLowerCase();
    if (title.includes('agreement')) {
        return '📝';
    } else if (title.includes('contribution')) {
        return '💳';
    } else if (title.includes('loan')) {
        return '💰';
    } else if (title.includes('id') || title.includes('passport')) {
        return '🆔';
    } else if (title.includes('residence') || title.includes('address')) {
        return '🏠';
    } else if (title.includes('kin')) {
        return '👨‍👩‍👧‍👦';
    } else if (title.includes('financial') || title.includes('statement')) {
        return '📊';
    } else {
        return '📁';
    }
}

// Helper function to get status class
function getStatusClass(status) {
    if (!status) return 'status-pending';
    
    const statusLower = status.toLowerCase();
    if (statusLower === 'available') {
        return 'status-available';
    } else if (statusLower === 'processing') {
        return 'status-processing';
    } else {
        return 'status-pending';
    }
}

// Helper function to format status
function formatStatus(status) {
    if (!status) return 'Pending';
    
    const statusLower = status.toLowerCase();
    if (statusLower === 'available') {
        return 'Available';
    } else if (statusLower === 'processing') {
        return 'Processing';
    } else {
        return 'Pending';
    }
}

// Helper function to format file size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Helper function to format date
function formatDate(dateString) {
    if (!dateString) return 'Unknown';
    
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Function to download a document
function downloadDocument(documentId, filePath) {
    console.log('Downloading document ID:', documentId);
    console.log('File path:', filePath);
    
    if (!filePath || filePath === 'null') {
        alert('This document is not available for download.');
        return;
    }
    
    // In a real implementation, this would download the actual document
    // For now, we'll show an alert with the file path
    alert(`Downloading document ID: ${documentId}\nFile path: ${filePath}\n\nIn a real implementation, this would download the file.`);
    
    // For a real implementation, you would do something like:
    // window.location.href = getApiUrl(`documents.php?id=${documentId}&download=true`);
}

// Function to delete a document
async function deleteDocument(documentId, documentTitle) {
    if (!confirm(`Are you sure you want to delete the document "${documentTitle}"?`)) {
        return;
    }
    
    try {
        const response = await fetch(getApiUrl(`documents.php?id=${documentId}`), {
            method: 'DELETE'
        });
        
        if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`Failed to delete document: ${errorText}`);
        }
        
        const result = await response.json();
        console.log('Document deleted successfully:', result);
        alert(`Document "${documentTitle}" deleted successfully!`);
        
        // Refresh the documents list
        const memberId = getMemberId();
        if (memberId) {
            refreshDocumentsList(memberId);
        }
    } catch (error) {
        console.error('Error deleting document:', error);
        alert(`Error deleting document: ${error.message}`);
    }
}

// Function to manually refresh the documents page
function refreshDocuments() {
    console.log('=== MANUAL REFRESH DOCUMENTS ===');
    const memberId = getMemberId();
    console.log('Refreshing documents for member ID:', memberId);
    if (memberId) {
        loadDocumentsData(memberId);
    } else {
        console.error('No member ID found for refresh');
    }
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshDocuments();
    }
});