// MFT GROUP Dashboard Data Loader
// This script fetches and displays member data on the dashboard

document.addEventListener('DOMContentLoaded', function() {
    console.log('=== DASHBOARD DATA LOADER INITIALIZED ===');
    
    // Get member ID from localStorage
    const memberId = localStorage.getItem('memberId');
    console.log('Member ID from localStorage:', memberId);
    
    // Add event listeners for action buttons
    addActionButtonListeners();
    
    // Keep trying to attach event listeners in case of timing issues
    const interval = setInterval(addActionButtonListeners, 1000);
    setTimeout(() => clearInterval(interval), 10000); // Stop after 10 seconds
    
    if (memberId) {
        console.log('Member ID found, loading member data');
        // Load member data
        loadMemberData(memberId);
        
        // Load dashboard data using optimized method
        loadDashboardDataOptimized(memberId);
    } else {
        console.error('Member ID not found in localStorage');
        // Try to get memberId from URL parameters as fallback
        const urlParams = new URLSearchParams(window.location.search);
        const urlMemberId = urlParams.get('member_id');
        if (urlMemberId) {
            console.log('Found member ID in URL parameters:', urlMemberId);
            localStorage.setItem('memberId', urlMemberId);
            loadMemberData(urlMemberId);
            loadDashboardDataOptimized(urlMemberId);
        }
    }
});

// Add a fallback to ensure data is loaded even if DOMContentLoaded fires too early
if (document.readyState === 'loading') {
    // Document is still loading, normal flow will handle it
    console.log('Document is still loading');
} else {
    // Document has already loaded, check if we need to load data
    console.log('Document already loaded, checking if data needs to be loaded');
    const memberId = localStorage.getItem('memberId');
    if (memberId && !window.memberDataLoaded) {
        console.log('Loading data on fallback');
        loadMemberData(memberId);
        loadDashboardDataOptimized(memberId);
        window.memberDataLoaded = true;
    }
}

// Function to add event listeners for action buttons
function addActionButtonListeners() {
    console.log('=== BUTTON DEBUGGING START ===');
    
    // Test if we can access the document
    console.log('Document ready state:', document.readyState);
    
    // Test if we can find the buttons
    const makeContributionBtn = document.getElementById('makeContributionBtn');
    const applyLoanBtn = document.getElementById('applyLoanBtn');
    const topUpLoanBtn = document.getElementById('topUpLoanBtn');
    const viewDocumentsBtn = document.getElementById('viewDocumentsBtn');
    
    console.log('Button elements found:');
    console.log('Make Contribution:', makeContributionBtn);
    console.log('Apply Loan:', applyLoanBtn);
    console.log('Top Up Loan:', topUpLoanBtn);
    console.log('View Documents:', viewDocumentsBtn);
    
    // Make Contribution button
    if (makeContributionBtn) {
        console.log('Attaching event listener to Make Contribution button');
        makeContributionBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Make Contribution button clicked!');
            window.location.href = 'add-contribution.html';
        };
        console.log('Event listener attached to Make Contribution button');
    } else {
        console.log('Make Contribution button NOT found');
    }
    
    // Apply for Loan button
    if (applyLoanBtn) {
        console.log('Attaching event listener to Apply for Loan button');
        applyLoanBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Apply for Loan button clicked!');
            window.location.href = 'loan-application-modern.html';
        };
        console.log('Event listener attached to Apply for Loan button');
    } else {
        console.log('Apply for Loan button NOT found');
    }
    
    // Top Up Loan button
    if (topUpLoanBtn) {
        console.log('Attaching event listener to Top Up Loan button');
        topUpLoanBtn.onclick = function(e) {
            e.preventDefault();
            console.log('Top Up Loan button clicked!');
            window.location.href = 'loan-application-modern.html?topup=true';
        };
        console.log('Event listener attached to Top Up Loan button');
    } else {
        console.log('Top Up Loan button NOT found');
    }
    
    // View Documents button
    if (viewDocumentsBtn) {
        console.log('Attaching event listener to View Documents button');
        viewDocumentsBtn.onclick = function(e) {
            e.preventDefault();
            console.log('View Documents button clicked!');
            window.location.href = 'documents.html';
        };
        console.log('Event listener attached to View Documents button');
    } else {
        console.log('View Documents button NOT found');
    }
    
    console.log('=== BUTTON DEBUGGING END ===');
}

// Handler functions for each button
function handleMakeContribution(e) {
    e.preventDefault();
    console.log('Make Contribution button clicked, redirecting to add-contribution.html');
    window.location.href = 'add-contribution.html';
}

function handleApplyLoan(e) {
    e.preventDefault();
    console.log('Apply for Loan button clicked, redirecting to loan-application-modern.html');
    window.location.href = 'loan-application-modern.html';
}

function handleTopUpLoan(e) {
    e.preventDefault();
    console.log('Top Up Loan button clicked, redirecting to loan-application-modern.html with topup parameter');
    window.location.href = 'loan-application-modern.html?topup=true';
}

function handleViewDocuments(e) {
    e.preventDefault();
    console.log('View Documents button clicked, redirecting to documents.html');
    window.location.href = 'documents.html';
}

// Function to load member data
async function loadMemberData(memberId) {
    try {
        console.log('Loading member data for memberId:', memberId);
        const memberData = await mftApi.getMemberData(memberId);
        console.log('Member data received:', memberData);
        
        if (memberData && !memberData.error) {
            // Store member data globally for later use
            window.currentMemberData = memberData;
            console.log('Stored member data in window.currentMemberData');
            
            // Update member info in header
            console.log('Calling updateMemberHeader');
            updateMemberHeader(memberData);
        } else {
            console.error('Failed to load member data:', memberData ? memberData.error : 'Unknown error');
            // Try to get member data from the dashboard API as fallback
            console.log('Trying to get member info from dashboard API as fallback');
            const dashboardData = await mftApi.getMemberDashboardData(memberId);
            if (dashboardData && dashboardData.member_info && !dashboardData.error) {
                window.currentMemberData = dashboardData.member_info;
                updateMemberHeader(dashboardData.member_info);
            }
        }
    } catch (error) {
        console.error('Error loading member data:', error);
        // Try to get member data from the dashboard API as fallback
        try {
            console.log('Trying to get member info from dashboard API as fallback due to error');
            const dashboardData = await mftApi.getMemberDashboardData(memberId);
            if (dashboardData && dashboardData.member_info && !dashboardData.error) {
                window.currentMemberData = dashboardData.member_info;
                updateMemberHeader(dashboardData.member_info);
            }
        } catch (fallbackError) {
            console.error('Error in fallback method:', fallbackError);
        }
    }
}

// Add a function to manually trigger member data loading for debugging
window.debugLoadMemberData = function() {
    const memberId = localStorage.getItem('memberId');
    if (memberId) {
        console.log('Manually loading member data for:', memberId);
        loadMemberData(memberId);
    } else {
        console.log('No memberId found in localStorage');
    }
};

// Cache for member dashboard data
const memberDashboardDataCache = {
    data: null,
    timestamp: 0,
    expirationTime: 30 * 60 * 1000 // Extended to 30 minutes cache expiration
};

// Optimized version that reduces API calls by batching requests
async function loadDashboardDataOptimized(memberId) {
    console.log('Loading optimized dashboard data for member ID:', memberId);
    
    // Check if we have valid cached data that's not expired
    const now = Date.now();
    if (memberDashboardDataCache.data && (now - memberDashboardDataCache.timestamp) < memberDashboardDataCache.expirationTime) {
        console.log('Using cached member dashboard data');
        const cachedData = memberDashboardDataCache.data;
        
        // Update all cards with cached data
        if (cachedData.contributions) {
            updateContributionsCard({ contributions: cachedData.contributions });
        }
        
        if (cachedData.loans) {
            updateLoansCard({ loans: cachedData.loans });
        }
        
        if (cachedData.fines) {
            updateFinesCard({ fines: cachedData.fines });
        }
        
        if (cachedData.projects) {
            updateProjectsCard(cachedData.projects);
        }
        
        if (cachedData.member_info) {
            updateProfileCard(cachedData.member_info);
        }
        
        // Update recent activities
        if (cachedData.recent_activities) {
            updateRecentActivities(cachedData.recent_activities);
        }
        
        // Update with cached data instead of placeholders
        updateInboxCard([]); // Placeholder - would need actual messages API
        updateAttendanceRecordsCard(cachedData.attendance || {}); // Use cached attendance data
        
        if (cachedData.missed_contributions) {
            updateMissedContributionsCard(cachedData.missed_contributions || []);
        }
        
        if (cachedData.guaranteed_loans) {
            updateGuaranteedLoansCard(cachedData.guaranteed_loans || []);
        }
        
        // Initialize real-time updates after successful data load
        if (typeof mftRealTime !== 'undefined' && mftRealTime) {
            console.log('Initializing real-time updates with cached data');
            mftRealTime.connect('member_dashboard', memberId);
        }
        
        return;
    }
    
    try {
        // Load all data in a single batch request
        const endpoint = `dashboard.php?member_id=${encodeURIComponent(memberId)}&type=member&action=getAllData`;
        // Use getApiUrl if available, otherwise fallback to direct construction
        const url = typeof getApiUrl === 'function' ? getApiUrl(endpoint) : 
                   (typeof getApiBaseUrl === 'function' ? getApiBaseUrl() + '/' + endpoint : 
                   `/api/${endpoint}`);
        console.log('Fetching all dashboard data from:', url);
        
        // Add timeout to prevent hanging requests
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 second timeout
        
        const response = await fetch(url, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            throw new Error('Response is not JSON');
        }
        
        const data = await response.json();
        console.log('All dashboard data received:', data);
        
        // Cache the data for future use
        memberDashboardDataCache.data = data;
        memberDashboardDataCache.timestamp = Date.now();
        
        // Update all cards with the received data
        if (data.contributions) {
            updateContributionsCard({ contributions: data.contributions });
        }
        
        if (data.loans) {
            updateLoansCard({ loans: data.loans });
        }
        
        if (data.fines) {
            updateFinesCard({ fines: data.fines });
        }
        
        if (data.projects) {
            updateProjectsCard(data.projects);
        }
        
        if (data.member_info) {
            updateProfileCard(data.member_info);
        }
        
        // Update recent activities
        if (data.recent_activities) {
            updateRecentActivities(data.recent_activities);
        }
        
        // Update with actual data instead of placeholders
        updateInboxCard([]); // Placeholder - would need actual messages API
        updateAttendanceRecordsCard(data.attendance || {}); // Use actual attendance data
        
        if (data.missed_contributions) {
            updateMissedContributionsCard(data.missed_contributions || []);
        }
        
        if (data.guaranteed_loans) {
            updateGuaranteedLoansCard(data.guaranteed_loans || []);
        }
        
        // Initialize real-time updates after successful data load
        if (typeof mftRealTime !== 'undefined' && mftRealTime) {
            console.log('Initializing real-time updates');
            mftRealTime.connect('member_dashboard', memberId);
        }
        
    } catch (error) {
        console.error('Error loading optimized dashboard data:', error);
        // Fallback to original method
        console.log('Falling back to original data loading method');
        await loadDashboardData(memberId);
    }
}

// Function to update member header information
function updateMemberHeader(memberData) {
    console.log('=== UPDATING MEMBER HEADER ===');
    console.log('Member data received:', memberData);
    
    // Retry mechanism to ensure elements are available
    let retries = 0;
    const maxRetries = 10;
    const retryInterval = 500; // 500ms
    
    function tryUpdateElements() {
        retries++;
        console.log(`Attempt ${retries} to update elements`);
        
        // Update greeting prefix with time-based greeting
        const greetingPrefixElement = document.getElementById('greetingPrefix');
        console.log('Greeting prefix element:', greetingPrefixElement);
        
        // Update member name
        const memberNameElement = document.getElementById('memberName');
        console.log('Member name element:', memberNameElement);
        
        // Check if both elements are available
        if (greetingPrefixElement && memberNameElement && memberData.first_name) {
            console.log('Both elements found, updating content');
            
            // Update greeting prefix with time-based greeting
            const greeting = getGreetingBasedOnTime();
            console.log('Time-based greeting:', greeting);
            greetingPrefixElement.textContent = greeting + ',';
            console.log('Updated greeting prefix to:', greetingPrefixElement.textContent);
            
            // Use full name (first name and last name)
            const fullName = memberData.first_name + (memberData.last_name ? ' ' + memberData.last_name : '');
            console.log('Full name:', fullName);
            
            // Log the current text content before updating
            console.log('Current member name text:', memberNameElement.textContent);
            
            memberNameElement.textContent = fullName;
            console.log('Updated member name to:', memberNameElement.textContent);
            
            // Verify the update was successful
            console.log('Verification - member name is now:', memberNameElement.textContent);
            
            // Update other elements
            updateOtherElements(memberData);
            
            return true; // Success
        } else {
            console.log('Elements not ready or data missing');
            if (!greetingPrefixElement) console.log('Greeting prefix element not found');
            if (!memberNameElement) console.log('Member name element not found');
            if (!memberData.first_name) console.log('First name not available in member data');
            
            if (retries < maxRetries) {
                console.log(`Retrying in ${retryInterval}ms...`);
                setTimeout(tryUpdateElements, retryInterval);
            } else {
                console.error('Failed to update elements after maximum retries');
                // Try one final time with a longer delay
                setTimeout(() => {
                    const finalGreetingPrefixElement = document.getElementById('greetingPrefix');
                    const finalMemberNameElement = document.getElementById('memberName');
                    if (finalGreetingPrefixElement && finalMemberNameElement && memberData.first_name) {
                        const greeting = getGreetingBasedOnTime();
                        finalGreetingPrefixElement.textContent = greeting + ',';
                        const fullName = memberData.first_name + (memberData.last_name ? ' ' + memberData.last_name : '');
                        finalMemberNameElement.textContent = fullName;
                        updateOtherElements(memberData);
                        console.log('Final update successful');
                    } else {
                        console.error('Final update also failed');
                    }
                }, 2000);
            }
            return false; // Not successful yet
        }
    }
    
    // Start the update process
    tryUpdateElements();
}

// Function to update other member info elements
function updateOtherElements(memberData) {
    console.log('Updating other elements with data:', memberData);
    
    // Update member ID
    const memberIdElement = document.getElementById('memberIdValue');
    if (memberIdElement && memberData.member_id) {
        memberIdElement.textContent = memberData.member_id;
        console.log('Updated memberIdValue to:', memberData.member_id);
    } else {
        console.log('memberIdValue element or data not found');
    }
    
    // Update registration date
    const registrationDateElement = document.getElementById('registrationDateValue');
    if (registrationDateElement && memberData.registration_date) {
        // Format the date nicely
        const registrationDate = new Date(memberData.registration_date);
        registrationDateElement.textContent = registrationDate.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
        console.log('Updated registrationDateValue to:', registrationDateElement.textContent);
    } else {
        console.log('registrationDateValue element or data not found');
    }
    
    // Update status
    const statusElement = document.getElementById('memberStatusValue');
    if (statusElement && memberData.status) {
        statusElement.textContent = memberData.status;
        
        // Add color based on status
        statusElement.className = 'info-card-value status-active-card';
        if (memberData.status.toLowerCase() === 'active') {
            statusElement.style.color = '#4CAF50'; // Green
        } else if (memberData.status.toLowerCase() === 'pending') {
            statusElement.style.color = '#FFC107'; // Yellow
        } else if (memberData.status.toLowerCase() === 'suspended') {
            statusElement.style.color = '#F44336'; // Red
        } else {
            statusElement.style.color = '#2196F3'; // Blue for other statuses
        }
        console.log('Updated memberStatusValue to:', memberData.status);
    } else {
        console.log('memberStatusValue element or data not found');
    }
    
    // Update last login
    const lastLoginElement = document.getElementById('lastLoginValue');
    if (lastLoginElement && memberData.last_login) {
        // Format the date nicely
        const lastLogin = new Date(memberData.last_login);
        lastLoginElement.textContent = lastLogin.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        console.log('Updated lastLoginValue to:', lastLoginElement.textContent);
    } else if (lastLoginElement) {
        lastLoginElement.textContent = 'Never';
        console.log('Set lastLoginValue to: Never');
    } else {
        console.log('lastLoginValue element not found');
    }
    
    // Note: IP address is no longer displayed on the dashboard per requirements
    // but is still logged in the database for security purposes
    
    // Load and display member photo
    if (memberData.member_id) {
        loadMemberPhoto(memberData.member_id);
    }
}

// Add a function to manually trigger header update for debugging
window.debugUpdateMemberHeader = function() {
    if (window.currentMemberData) {
        console.log('Manually updating member header with:', window.currentMemberData);
        updateMemberHeader(window.currentMemberData);
    } else {
        console.log('No member data available in window.currentMemberData');
    }
};

// Function to load and display member photo
async function loadMemberPhoto(memberId) {
    try {
        // Fetch member photo information
        const response = await fetch(`api/member-photo.php?member_id=${memberId}`);
        const data = await response.json();
        
        const memberPhoto = document.getElementById('memberPhoto');
        const noPhotoMessage = document.getElementById('noPhotoMessage');
        
        if (data.success && data.photo_url) {
            // Photo exists, display it
            memberPhoto.src = data.photo_url;
            memberPhoto.style.display = 'block';
            noPhotoMessage.style.display = 'none';
        } else {
            // No photo available
            memberPhoto.style.display = 'none';
            noPhotoMessage.style.display = 'block';
        }
    } catch (error) {
        console.error('Error loading member photo:', error);
        // Show no photo message on error
        const memberPhoto = document.getElementById('memberPhoto');
        const noPhotoMessage = document.getElementById('noPhotoMessage');
        memberPhoto.style.display = 'none';
        noPhotoMessage.style.display = 'block';
    }
}

// Function to get appropriate greeting based on time of day
function getGreetingBasedOnTime() {
    const hours = new Date().getHours();
    
    if (hours >= 5 && hours < 12) {
        return 'Good morning';
    } else if (hours >= 12 && hours < 18) {
        return 'Good afternoon';
    } else {
        return 'Good evening';
    }
}

// Function to update contributions card
function updateContributionsCard(data) {
    console.log('Updating contributions card with data:', data);
    
    // Get total balance from the correct field
    let totalBalance = 0;
    if (data.contributions && typeof data.contributions === 'object') {
        // Check if it's the dashboard API response format
        if (data.contributions.total_balance !== undefined) {
            totalBalance = parseFloat(data.contributions.total_balance) || 0;
        } 
        // Check if it's the contributions API response format
        else if (data.total_balance !== undefined) {
            totalBalance = parseFloat(data.total_balance) || 0;
        }
        // Calculate from contributions array if needed
        else if (Array.isArray(data.contributions)) {
            totalBalance = data.contributions.reduce((sum, contribution) => {
                return sum + (contribution.status === 'Confirmed' ? parseFloat(contribution.amount) || 0 : 0);
            }, 0);
        }
    } else if (data.total_balance !== undefined) {
        totalBalance = parseFloat(data.total_balance) || 0;
    }
    
    console.log('Calculated total balance:', totalBalance);
    
    // Update contributions card
    const contributionCards = document.querySelectorAll('.dashboard-card.contributions');
    console.log('Found contribution cards:', contributionCards.length);
    contributionCards.forEach((card, index) => {
        console.log('Updating contribution card', index);
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            console.log('Setting contribution value to:', `KSH ${Number(totalBalance).toLocaleString()}`);
            valueElement.textContent = `KSH ${Number(totalBalance).toLocaleString()}`;
        } else {
            console.log('No .card-value element found in contribution card', index);
        }
    });
    
    // Also update the specific element by ID if it exists
    const totalContributionsElement = document.getElementById('totalContributions');
    if (totalContributionsElement) {
        console.log('Updating totalContributions element with:', `KSH ${Number(totalBalance).toLocaleString()}`);
        totalContributionsElement.textContent = `KSH ${Number(totalBalance).toLocaleString()}`;
    } else {
        console.log('No #totalContributions element found');
    }
}

// Function to update loans card
function updateLoansCard(data) {
    console.log('Updating loans card with data:', data);
    
    // Get outstanding balance from the correct field
    let outstandingBalance = 0;
    if (data.loans && typeof data.loans === 'object') {
        // Check if it's the dashboard API response format
        if (data.loans.outstanding_balance !== undefined) {
            outstandingBalance = parseFloat(data.loans.outstanding_balance) || 0;
        } 
        // Check if it's the loans API response format
        else if (data.outstanding_balance !== undefined) {
            outstandingBalance = parseFloat(data.outstanding_balance) || 0;
        }
        // Calculate from loans array if needed
        else if (Array.isArray(data.loans)) {
            outstandingBalance = data.loans.reduce((sum, loan) => {
                return sum + (loan.status === 'Active' ? parseFloat(loan.principal_amount) || 0 : 0);
            }, 0);
        }
    } else if (data.outstanding_balance !== undefined) {
        outstandingBalance = parseFloat(data.outstanding_balance) || 0;
    }
    
    console.log('Calculated outstanding balance:', outstandingBalance);
    
    // Update loans card
    const loanCards = document.querySelectorAll('.dashboard-card.loans');
    console.log('Found loan cards:', loanCards.length);
    loanCards.forEach((card, index) => {
        console.log('Updating loan card', index);
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            console.log('Setting loan value to:', `KSH ${Number(outstandingBalance).toLocaleString()}`);
            valueElement.textContent = `KSH ${Number(outstandingBalance).toLocaleString()}`;
        } else {
            console.log('No .card-value element found in loan card', index);
        }
    });
    
    // Also update the specific element by ID if it exists
    const activeLoansElement = document.getElementById('activeLoans');
    if (activeLoansElement) {
        console.log('Updating activeLoans element with:', `KSH ${Number(outstandingBalance).toLocaleString()}`);
        activeLoansElement.textContent = `KSH ${Number(outstandingBalance).toLocaleString()}`;
    } else {
        console.log('No #activeLoans element found');
    }
}

// Function to update fines card
function updateFinesCard(data) {
    console.log('Updating fines card with data:', data);
    
    // Calculate total fines
    let totalFines = 0;
    
    // Check if we have fines data as an object with total_fines field (from dashboard API)
    if (data.fines && typeof data.fines === 'object' && data.fines.total_fines !== undefined) {
        totalFines = parseFloat(data.fines.total_fines) || 0;
    } 
    // Check if we have fines data as an object with total_fines field (from fines API response wrapped in object)
    else if (data.total_fines !== undefined) {
        totalFines = parseFloat(data.total_fines) || 0;
    }
    // Check if we have fines data as an array (from fines API - {fines: {count: 0, fines: []}})
    else if (data.fines && typeof data.fines === 'object' && Array.isArray(data.fines.fines)) {
        totalFines = data.fines.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    }
    // Check if we have fines data as an array directly (from fines API)
    else if (data.fines && Array.isArray(data.fines)) {
        totalFines = data.fines.reduce((sum, fine) => {
            return sum + ((fine.status === 'Unpaid' || fine.status === 'Pending') ? parseFloat(fine.amount) || 0 : 0);
        }, 0);
    } 
    // Check if we have total_fines directly (fallback)
    else if (data.total_fines !== undefined) {
        totalFines = parseFloat(data.total_fines) || 0;
    }
    
    console.log('Fines cards found:', document.querySelectorAll('.dashboard-card.fines').length, 'Total fines:', totalFines);
    
    // Update fines card
    const finesCards = document.querySelectorAll('.dashboard-card.fines');
    console.log('Found fines cards:', finesCards.length);
    finesCards.forEach((card, index) => {
        console.log('Updating fines card', index);
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            console.log('Setting fines value to:', `KSH ${Number(totalFines).toLocaleString()}`);
            valueElement.textContent = `KSH ${Number(totalFines).toLocaleString()}`;
        } else {
            console.log('No .card-value element found in fines card', index);
        }
    });
    
    // Also update the specific element by ID if it exists
    const pendingFinesElement = document.getElementById('pendingFines');
    if (pendingFinesElement) {
        console.log('Updating pendingFines element with:', `KSH ${Number(totalFines).toLocaleString()}`);
        pendingFinesElement.textContent = `KSH ${Number(totalFines).toLocaleString()}`;
    } else {
        console.log('No #pendingFines element found');
    }
}

// Function to update projects card
function updateProjectsCard(data) {
    console.log('Updating projects card with data:', data);
    
    // Get projects data from the correct location
    let projects = [];
    if (data.projects && Array.isArray(data.projects)) {
        // Direct array from individual API
        projects = data.projects;
    } else if (data.projects && typeof data.projects === 'object' && Array.isArray(data.projects.projects)) {
        // From dashboard API
        projects = data.projects.projects;
    } else if (data && Array.isArray(data)) {
        // If data is directly an array of projects (from individual API)
        projects = data;
    } else if (data.projects && typeof data.projects === 'object' && !Array.isArray(data.projects)) {
        // If data.projects is an object with project data (new dashboard API format)
        projects = Array.isArray(data.projects) ? data.projects : [];
    }
    
    const projectCount = projects.length;
    console.log('Project count:', projectCount);
    
    // Update projects card
    const projectsCards = document.querySelectorAll('.dashboard-card.projects');
    console.log('Found projects cards:', projectsCards.length);
    projectsCards.forEach((card, index) => {
        console.log('Updating projects card', index);
        // Update card value (count of projects)
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            console.log('Setting projects count to:', projectCount);
            valueElement.textContent = projectCount;
        } else {
            console.log('No .card-value element found in projects card', index);
        }
        
        // Update card detail
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            const text = projectCount === 1 ? 'Project Assigned' : 'Projects Assigned';
            console.log('Setting projects detail to:', text);
            detailElement.textContent = text;
        } else {
            console.log('No .card-detail element found in projects card', index);
        }
    });
    
    // Also update the specific "My Projects" card
    const myProjectsCountElement = document.getElementById('myProjectsCount');
    if (myProjectsCountElement) {
        console.log('Updating myProjectsCount element with:', projectCount);
        myProjectsCountElement.textContent = projectCount;
    } else {
        console.log('No #myProjectsCount element found');
    }
}

// Function to update profile card
function updateProfileCard(profileData) {
    console.log('Updating profile card with data:', profileData);
    
    // Update profile card
    const profileCards = document.querySelectorAll('.dashboard-card.profile');
    console.log('Found profile cards:', profileCards.length);
    profileCards.forEach((card, index) => {
        console.log('Updating profile card', index);
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            const text = profileData.verified ? 'Account verified' : 'Account not verified';
            console.log('Setting profile detail to:', text);
            detailElement.textContent = text;
        } else {
            console.log('No .card-detail element found in profile card', index);
        }
        
        // Update profile status value
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            const status = profileData.status || 'Active';
            console.log('Setting profile status to:', status);
            valueElement.textContent = status;
        } else {
            console.log('No .card-value element found in profile card', index);
        }
        
        // Update the specific profile status element
        const profileStatusElement = document.getElementById('profileStatus');
        if (profileStatusElement) {
            const status = profileData.status || 'Active';
            console.log('Updating profileStatus element with:', status);
            profileStatusElement.textContent = status;
        } else {
            console.log('No #profileStatus element found');
        }
    });
}

// Function to update inbox card
function updateInboxCard(messages) {
    const unreadCount = messages && Array.isArray(messages) ? 
        messages.filter(msg => !msg.read).length : 0;
    
    // Update inbox card
    const inboxCards = document.querySelectorAll('.dashboard-card.inbox');
    inboxCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = unreadCount;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${unreadCount} new`;
            if (unreadCount > 0) {
                statusElement.className = 'card-status status-warning';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific unread messages element
    const unreadMessagesElement = document.getElementById('unreadMessages');
    if (unreadMessagesElement) {
        unreadMessagesElement.textContent = unreadCount;
    }
}

// Function to update attendance records card (replaces dividends)
function updateAttendanceRecordsCard(attendanceData) {
    console.log('Updating attendance records card with data:', attendanceData);
    
    // Get attendance rate from the data
    let attendanceRate = '0%';
    if (attendanceData && typeof attendanceData === 'object') {
        if (attendanceData.attendance_rate !== undefined) {
            attendanceRate = attendanceData.attendance_rate + '%';
        } else if (attendanceData.attended_meetings !== undefined && attendanceData.total_meetings !== undefined) {
            const rate = attendanceData.total_meetings > 0 ? 
                Math.round((attendanceData.attended_meetings / attendanceData.total_meetings) * 100) : 0;
            attendanceRate = rate + '%';
        }
    }
    
    // Update attendance records card
    const attendanceCards = document.querySelectorAll('.dashboard-card.dividends');
    attendanceCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = attendanceRate;
        }
        
        const detailElement = card.querySelector('.card-detail');
        if (detailElement) {
            detailElement.textContent = 'Meeting Attendance';
        }
    });
    
    // Also update the specific attendance rate element
    const attendanceRateElement = document.getElementById('attendanceRate');
    if (attendanceRateElement) {
        attendanceRateElement.textContent = attendanceRate;
    }
}

// Function to update missed contributions card
function updateMissedContributionsCard(contributions) {
    // Calculate missed contributions (simplified logic)
    let missedAmount = 0;
    let monthsMissed = 0;
    
    if (contributions && Array.isArray(contributions)) {
        // Filter for missed/unpaid contributions
        const missedContributions = contributions.filter(contribution => 
            contribution.status === 'Missed' || contribution.status === 'Unpaid');
        
        // Calculate total missed amount
        missedAmount = missedContributions.reduce((sum, contribution) => 
            sum + (parseFloat(contribution.amount) || 0), 0);
        
        // For months missed, we would need more detailed data
        monthsMissed = missedContributions.length;
    }
    
    // Update missed contributions card
    const missedCards = document.querySelectorAll('.dashboard-card.missed');
    missedCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(missedAmount || 0).toLocaleString()}`;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${monthsMissed || 0} months`;
            if (monthsMissed > 0) {
                statusElement.className = 'card-status status-danger';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific missed contributions element
    const missedContributionsElement = document.getElementById('missedContributions');
    if (missedContributionsElement) {
        missedContributionsElement.textContent = `KSH ${Number(missedAmount || 0).toLocaleString()}`;
    }
}

// Function to update guaranteed loans card
function updateGuaranteedLoansCard(loans) {
    // Calculate guaranteed loans amount (simplified logic)
    let guaranteedAmount = 0;
    let activeGuarantees = 0;
    
    if (loans && Array.isArray(loans)) {
        // Filter for loans where the member is a guarantor
        const guaranteedLoans = loans.filter(loan => 
            loan.guarantor_id && loan.status === 'Active');
        
        // Calculate total guaranteed amount
        guaranteedAmount = guaranteedLoans.reduce((sum, loan) => 
            sum + (parseFloat(loan.principal_amount) || 0), 0);
        
        // Count active guarantees
        activeGuarantees = guaranteedLoans.length;
    }
    
    // Update guaranteed loans card
    const guaranteedCards = document.querySelectorAll('.dashboard-card.guaranteed');
    guaranteedCards.forEach(card => {
        const valueElement = card.querySelector('.card-value');
        if (valueElement) {
            valueElement.textContent = `KSH ${Number(guaranteedAmount || 0).toLocaleString()}`;
        }
        
        const statusElement = card.querySelector('.card-status');
        if (statusElement) {
            statusElement.textContent = `${activeGuarantees} active`;
            if (activeGuarantees > 0) {
                statusElement.className = 'card-status status-warning';
            } else {
                statusElement.className = 'card-status status-active';
            }
        }
    });
    
    // Also update the specific guaranteed loans element
    const guaranteedLoansElement = document.getElementById('guaranteedLoans');
    if (guaranteedLoansElement) {
        guaranteedLoansElement.textContent = `KSH ${Number(guaranteedAmount || 0).toLocaleString()}`;
    }
}

// Function to manually clear the dashboard data cache
function clearDashboardCache() {
    memberDashboardDataCache.data = null;
    memberDashboardDataCache.timestamp = 0;
    console.log('Dashboard cache cleared');
}

// Expose the function globally for debugging
window.clearDashboardCache = clearDashboardCache;

// Function to update recent activities section
function updateRecentActivities(activities) {
    console.log('Updating recent activities with:', activities);
    
    // Find the activity list element
    const activityList = document.querySelector('.activity-list');
    if (!activityList) {
        console.log('Activity list element not found');
        return;
    }
    
    // Clear existing activities
    activityList.innerHTML = '';
    
    // If no activities, show a message
    if (!activities || activities.length === 0) {
        activityList.innerHTML = `
            <li class="activity-item">
                <div class="activity-content">
                    <div class="activity-title">No recent activities</div>
                    <div class="activity-time">-</div>
                </div>
            </li>
        `;
        return;
    }
    
    // Add each activity to the list
    activities.forEach(activity => {
        const activityItem = document.createElement('li');
        activityItem.className = 'activity-item';
        
        // Determine icon based on activity type
        let iconClass = 'fas fa-info-circle';
        switch (activity.type) {
            case 'contribution':
                iconClass = 'fas fa-wallet';
                break;
            case 'loan':
                iconClass = 'fas fa-file-invoice-dollar';
                break;
            case 'fine':
                iconClass = 'fas fa-exclamation-triangle';
                break;
            case 'project':
                iconClass = 'fas fa-project-diagram';
                break;
        }
        
        // Format timestamp
        let timeAgo = 'Just now';
        if (activity.timestamp) {
            const activityTime = new Date(activity.timestamp);
            const now = new Date();
            const diffMs = now - activityTime;
            const diffMins = Math.floor(diffMs / 60000);
            const diffHours = Math.floor(diffMs / 3600000);
            const diffDays = Math.floor(diffMs / 86400000);
            
            if (diffMins < 1) {
                timeAgo = 'Just now';
            } else if (diffMins < 60) {
                timeAgo = `${diffMins} minute${diffMins > 1 ? 's' : ''} ago`;
            } else if (diffHours < 24) {
                timeAgo = `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
            } else {
                timeAgo = `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;
            }
        }
        
        activityItem.innerHTML = `
            <div class="activity-icon">
                <i class="${iconClass}"></i>
            </div>
            <div class="activity-content">
                <div class="activity-title">${activity.description || 'Activity'}</div>
                <div class="activity-time">${timeAgo}</div>
            </div>
        `;
        
        activityList.appendChild(activityItem);
    });
}

// Helper function to format project dates
function formatProjectDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const month = date.toLocaleDateString('en-US', { month: 'short' });
    const year = date.getFullYear();
    return `${month} ${year}`;
}