// MFT GROUP Contributions Management Real-Time Updates
// This script handles real-time updates for the contributions management dashboard

// Store all members for search functionality
let allMembers = [];

// Wait for the DOM to be fully loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize real-time client
    mftRealTime.connect('admin');
    
    // Register update callback
    mftRealTime.onUpdate(function(data, pageType) {
        if (pageType === 'admin') {
            updateContributionsManagementRealTime(data);
        }
    });
    
    // Load initial data
    loadContributionsData();
    
    // Set up event listeners for filter buttons with a delay to ensure elements are available
    setTimeout(setupFilterEventListeners, 500);
    
    // Add event listeners for tab changes to re-attach filter event listeners
    setupTabChangeListeners();
    
    // Set up member search functionality
    setupMemberSearch();
    
    console.log('Real-time updates started for contributions management dashboard');
});

// Function to set up event listeners for filter buttons
function setupFilterEventListeners() {
    // Add event listeners for filter buttons
    const applyFilterBtn = document.getElementById('applyFilter');
    const clearFilterBtn = document.getElementById('clearFilter');
    
    if (applyFilterBtn) {
        applyFilterBtn.addEventListener('click', applyFilters);
        console.log('Apply filter button event listener attached');
    } else {
        console.log('Apply filter button not found');
    }
    
    if (clearFilterBtn) {
        clearFilterBtn.addEventListener('click', clearFilters);
        console.log('Clear filter button event listener attached');
    } else {
        console.log('Clear filter button not found');
    }
}

// Function to set up tab change listeners
function setupTabChangeListeners() {
    // Re-attach filter event listeners when tabs are changed
    const tabs = document.querySelectorAll('.nav-tab');
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            // When a tab is clicked, re-attach filter event listeners after a short delay
            setTimeout(setupFilterEventListeners, 100);
        });
    });
}

// Function to set up member search functionality
function setupMemberSearch() {
    // Load all members for search
    loadAllMembers();
    
    // Set up search event listener
    const memberSearch = document.getElementById('memberSearch');
    if (memberSearch) {
        memberSearch.addEventListener('input', debounce(handleMemberSearch, 300));
    }
    
    // Also set up real-time filtering for other search fields
    const filterInputs = document.querySelectorAll('#contributions-tab input, #contributions-tab select');
    filterInputs.forEach(input => {
        if (input.id !== 'memberSearch') { // Skip the member search as it's handled separately
            input.addEventListener('input', debounce(applyFilters, 300));
        }
    });
    
    // For select elements, we'll use the 'change' event
    const filterSelects = document.querySelectorAll('#contributions-tab select');
    filterSelects.forEach(select => {
        if (select.id !== 'memberFilter') { // Skip the member filter as it's handled separately
            select.addEventListener('change', applyFilters);
        }
    });
}

// Debounce function to limit the rate of function calls
function debounce(func, delay) {
    let timeoutId;
    return function (...args) {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func.apply(this, args), delay);
    };
}

// Function to load all members
async function loadAllMembers() {
    try {
        const response = await fetch(getApiUrl('members.php'));
        const data = await response.json();
        
        if (data.members) {
            allMembers = data.members;
        }
    } catch (error) {
        console.error('Error loading members:', error);
    }
}

// Function to handle member search
function handleMemberSearch() {
    const searchTerm = this.value;
    const filteredMembers = filterMembers(allMembers, searchTerm);
    updateMemberDropdown('contributionsMemberSelect', filteredMembers);
}

// Function to filter members based on search term
function filterMembers(members, searchTerm) {
    if (!searchTerm) return members;
    
    const term = searchTerm.toLowerCase();
    return members.filter(member => 
        (member.first_name && member.first_name.toLowerCase().includes(term)) ||
        (member.last_name && member.last_name.toLowerCase().includes(term)) ||
        (member.member_id && member.member_id.toLowerCase().includes(term)) ||
        (member.name && member.name.toLowerCase().includes(term)) // For different data structures
    );
}

// Function to update a member dropdown with filtered members
function updateMemberDropdown(selectId, members) {
    const selectElement = document.getElementById(selectId);
    if (!selectElement) return;
    
    // Store the currently selected value
    const selectedValue = selectElement.value;
    
    // Clear existing options except the first one
    selectElement.innerHTML = '<option value="">Select Member</option>';
    
    // Add filtered members
    members.forEach(member => {
        const option = document.createElement('option');
        option.value = member.id || member.member_id;
        
        // Handle different data structures
        if (member.first_name && member.last_name) {
            option.textContent = `${member.first_name} ${member.last_name}`;
        } else if (member.name) {
            option.textContent = member.name;
        } else {
            option.textContent = member.member_id || member.id;
        }
        
        // Add member ID if available
        if (member.member_id) {
            option.textContent += ` (${member.member_id})`;
        }
        
        // Restore selection if it matches
        if ((member.id === selectedValue) || (member.member_id === selectedValue)) {
            option.selected = true;
        }
        
        selectElement.appendChild(option);
    });
}

// Function to load initial contributions data
async function loadContributionsData() {
    try {
        // Show loading states
        showLoadingStates();
        
        // Fetch data from API
        const overviewData = await fetchOverviewData();
        const recentContributions = await fetchRecentContributions();
        const allContributions = await fetchAllContributions();
        const members = await fetchMembers();
        
        // Update the dashboard with fetched data
        if (overviewData) {
            updateOverviewStats(overviewData);
        }
        
        if (recentContributions) {
            updateRecentContributions(recentContributions);
        }
        
        if (allContributions) {
            updateAllContributions(allContributions);
        }
        
        if (members) {
            updateMemberSelectOptions(members);
        }
    } catch (error) {
        console.error('Error loading contributions data:', error);
        showErrorStates();
    }
}

// Function to show loading states
function showLoadingStates() {
    // Update stats cards
    const statValues = document.querySelectorAll('.stat-value');
    statValues.forEach(stat => {
        stat.textContent = 'Loading...';
    });
    
    // Update recent contributions table
    const recentTableBody = document.querySelector('#overview-tab .mft-table tbody');
    if (recentTableBody) {
        recentTableBody.innerHTML = `
            <tr>
                <td colspan="7" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading recent contributions...</div>
                </td>
            </tr>`;
    }
    
    // Update all contributions table
    const allTableBody = document.querySelector('#contributions-tab .mft-table tbody');
    if (allTableBody) {
        allTableBody.innerHTML = `
            <tr>
                <td colspan="9" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading all contributions...</div>
                </td>
            </tr>`;
    }
    
    // Update pending contributions table
    const pendingTableBody = document.querySelector('#reminders-tab .mft-table tbody');
    if (pendingTableBody) {
        pendingTableBody.innerHTML = `
            <tr>
                <td colspan="7" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Loading pending contributions...</div>
                </td>
            </tr>`;
    }
}

// Function to show error states
function showErrorStates() {
    // Update stats cards
    const statValues = document.querySelectorAll('.stat-value');
    statValues.forEach(stat => {
        stat.textContent = 'Error';
    });
    
    // Update recent contributions table
    const recentTableBody = document.querySelector('#overview-tab .contribution-table tbody');
    if (recentTableBody) {
        recentTableBody.innerHTML = '<tr><td colspan="7" style="text-align: center; color: #F44336;">Error loading recent contributions</td></tr>';
    }
    
    // Update all contributions table
    const allTableBody = document.querySelector('#contributions-tab .contribution-table tbody');
    if (allTableBody) {
        allTableBody.innerHTML = '<tr><td colspan="9" style="text-align: center; color: #F44336;">Error loading all contributions</td></tr>';
    }
    
    // Update pending contributions table
    const pendingTableBody = document.querySelector('#reminders-tab .contribution-table tbody');
    if (pendingTableBody) {
        pendingTableBody.innerHTML = '<tr><td colspan="7" style="text-align: center; color: #F44336;">Error loading pending contributions</td></tr>';
    }
}

// Function to fetch overview data
async function fetchOverviewData() {
    try {
        // Fetch real data from the API
        const response = await fetch(getApiUrl('dashboard.php?action=contributions_summary'));
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error('Error fetching overview data:', error);
        // Try alternative endpoint if the first one fails
        try {
            console.log('Trying alternative endpoint...');
            const response = await fetch(getApiUrl('dashboard.php?type=contributions'));
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return await response.json();
        } catch (altError) {
            console.error('Error fetching overview data with alternative endpoint:', altError);
            return null;
        }
    }
}

// Function to fetch recent contributions
async function fetchRecentContributions() {
    try {
        // Fetch real data from the API
        const response = await fetch(getApiUrl('contributions.php?limit=5'));
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        return data.contributions || [];
    } catch (error) {
        console.error('Error fetching recent contributions:', error);
        return null;
    }
}

// Function to fetch all contributions
async function fetchAllContributions() {
    try {
        // Fetch real data from the API
        const response = await fetch(getApiUrl('contributions.php'));
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        return data.contributions || [];
    } catch (error) {
        console.error('Error fetching all contributions:', error);
        return null;
    }
}

// Function to fetch members
async function fetchMembers() {
    try {
        // Fetch real data from the API
        const response = await fetch(getApiUrl('members.php'));
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        return data.members || [];
    } catch (error) {
        console.error('Error fetching members:', error);
        return null;
    }
}

// Function to update the contributions management page with real-time data
function updateContributionsManagementRealTime(data) {
    console.log('Updating contributions management with real-time data:', data);
    
    // Debug: Log the structure of recentContributions data if it exists
    if (data.recentContributions) {
        console.log('Recent contributions data structure:', JSON.stringify(data.recentContributions, null, 2));
    }
    
    // Update different sections based on the data received
    if (data.overview) {
        updateOverviewStats(data.overview);
    }
    
    // Handle both possible data structures for recent contributions
    if (data.recentContributions) {
        updateRecentContributions(data.recentContributions);
    } else if (data.recent_contributions) {
        updateRecentContributions(data.recent_contributions);
    }
    
    if (data.allContributions) {
        updateAllContributions(data.allContributions);
    }
    
    if (data.pendingContributions) {
        updatePendingContributions(data.pendingContributions);
    }
    
    if (data.members) {
        updateMemberSelectOptions(data.members);
    }
}

// Function to update overview stats
function updateOverviewStats(data) {
    console.log('updateOverviewStats called with:', data);
    
    // Update total contributions
    const totalContributionsElement = document.querySelector('.stat-card:nth-child(1) .stat-value');
    if (totalContributionsElement) {
        // Handle both possible data structures
        const total = data.total_contributions || data.totalContributions || 0;
        totalContributionsElement.textContent = `KSh ${parseFloat(total).toLocaleString()}`;
    }
    
    // Update active members
    const activeMembersElement = document.querySelector('.stat-card:nth-child(2) .stat-value');
    if (activeMembersElement) {
        // Handle both possible data structures
        const active = data.active_members || data.activeMembers || 0;
        activeMembersElement.textContent = active;
    }
    
    // Update collection rate
    const collectionRateElement = document.querySelector('.stat-card:nth-child(3) .stat-value');
    if (collectionRateElement) {
        // Handle both possible data structures
        const rate = data.collection_rate || data.collectionRate || 0;
        collectionRateElement.textContent = `${rate}%`;
    }
    
    // Update pending payments
    const pendingPaymentsElement = document.querySelector('.stat-card:nth-child(4) .stat-value');
    if (pendingPaymentsElement) {
        // Handle both possible data structures
        const pending = data.pending_amount || data.pendingPayments || 0;
        pendingPaymentsElement.textContent = `KSh ${parseFloat(pending).toLocaleString()}`;
    }
    
    // Update member details
    const activeMembersLabel = document.querySelector('.stat-card:nth-child(2) .stat-label:last-child');
    if (activeMembersLabel) {
        // Handle both possible data structures
        const total = data.total_members || data.totalMembers || 0;
        activeMembersLabel.textContent = `Out of ${total}`;
    }
    
    const pendingPaymentsLabel = document.querySelector('.stat-card:nth-child(4) .stat-label:last-child');
    if (pendingPaymentsLabel) {
        // Handle both possible data structures
        const pending = data.pending_amount || data.pendingPayments || 0;
        pendingPaymentsLabel.textContent = `From ${Math.floor(parseFloat(pending) / 1000)} Members`;
    }
}

// Function to update recent contributions
function updateRecentContributions(contributions) {
    console.log('updateRecentContributions called with:', contributions);
    
    const tableBody = document.querySelector('#overview-tab .mft-table tbody');
    if (!tableBody) return;
    
    if (!contributions || contributions.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="7" class="empty-state">
                    <i class="fas fa-file-invoice-dollar"></i>
                    <h3>No Recent Contributions</h3>
                    <p>There are no recent contributions to display.</p>
                </td>
            </tr>`;
        return;
    }
    
    // Clear existing rows
    tableBody.innerHTML = '';
    
    // Add new rows
    contributions.forEach(contribution => {
        console.log('Processing contribution:', contribution);
        
        // Map API response to expected data structure
        const mappedContribution = {
            id: contribution.id,
            date: contribution.contribution_date || contribution.date,
            member: contribution.first_name && contribution.last_name ? 
                `${contribution.first_name} ${contribution.last_name}` : 
                (contribution.first_name || contribution.last_name || contribution.member_id || contribution.member || 'Unknown'),
            amount: parseFloat(contribution.amount),
            paymentMethod: contribution.payment_method || contribution.paymentMethod || 'N/A',
            category: contribution.category || 'Regular',
            status: contribution.status || 'Pending'
        };
        
        console.log('Mapped contribution:', mappedContribution);
        
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${formatDate(mappedContribution.date)}</td>
            <td>${mappedContribution.member}</td>
            <td>KSh ${mappedContribution.amount.toLocaleString()}</td>
            <td>${mappedContribution.paymentMethod}</td>
            <td>${mappedContribution.category}</td>
            <td><span class="status-badge status-${mappedContribution.status.toLowerCase()}">${mappedContribution.status}</span></td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline view-btn" title="View Details" data-id="${mappedContribution.id}">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-sm btn-primary edit-btn" title="Edit" data-id="${mappedContribution.id}">
                    <i class="fas fa-edit"></i>
                </button>
            </td>
        `;
        tableBody.appendChild(row);
    });
    
    // Add event listeners to action buttons
    attachActionListeners();
    
    // Add sorting functionality
    addSortingToTable('recentContributionsTableBody');
}

// Function to update all contributions
function updateAllContributions(contributions) {
    console.log('updateAllContributions called with:', contributions);
    
    const tableBody = document.querySelector('#contributions-tab .mft-table tbody');
    if (!tableBody) return;
    
    if (!contributions || contributions.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="9" class="empty-state">
                    <i class="fas fa-file-invoice-dollar"></i>
                    <h3>No Contributions Found</h3>
                    <p>There are no contributions in the system yet.</p>
                </td>
            </tr>`;
        return;
    }
    
    // Clear existing rows
    tableBody.innerHTML = '';
    
    // Add new rows
    contributions.forEach(contribution => {
        console.log('Processing contribution:', contribution);
        
        // Map API response to expected data structure
        const mappedContribution = {
            id: contribution.id,
            date: contribution.contribution_date,
            memberId: contribution.member_id || contribution.memberId || 'N/A',
            memberName: contribution.first_name && contribution.last_name ? 
                `${contribution.first_name} ${contribution.last_name}` : 
                (contribution.first_name || contribution.last_name || contribution.member_id || 'Unknown'),
            amount: parseFloat(contribution.amount),
            paymentMethod: contribution.payment_method || 'N/A',
            category: contribution.category || 'Regular',
            reference: contribution.payment_reference || contribution.reference || 'N/A',
            status: contribution.status || 'Pending'
        };
        
        console.log('Mapped contribution:', mappedContribution);
        
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${formatDate(mappedContribution.date)}</td>
            <td>${mappedContribution.memberId}</td>
            <td>${mappedContribution.memberName}</td>
            <td>KSh ${mappedContribution.amount.toLocaleString()}</td>
            <td>${mappedContribution.paymentMethod}</td>
            <td>${mappedContribution.category}</td>
            <td>${mappedContribution.reference}</td>
            <td><span class="status-badge status-${mappedContribution.status.toLowerCase()}">${mappedContribution.status}</span></td>
            <td class="action-cell">
                <button class="btn btn-sm btn-outline view-btn" title="View Details" data-id="${mappedContribution.id}">
                    <i class="fas fa-eye"></i>
                </button>
                <button class="btn btn-sm btn-primary edit-btn" title="Edit" data-id="${mappedContribution.id}">
                    <i class="fas fa-edit"></i>
                </button>
                ${mappedContribution.status === 'Paid' ? 
                    `<button class="btn btn-sm btn-danger delete-btn" title="Delete" data-id="${mappedContribution.id}"><i class="fas fa-trash"></i></button>` : 
                    `<button class="btn btn-sm btn-success mark-paid-btn" title="Mark as Paid" data-id="${mappedContribution.id}"><i class="fas fa-check"></i></button>`}
            </td>
        `;
        tableBody.appendChild(row);
    });
    
    // Add event listeners to action buttons
    attachActionListeners();
    
    // Add sorting functionality
    addSortingToTable('contributionsTableBody');
}

// Function to update pending contributions
function updatePendingContributions(contributions) {
    const tableBody = document.querySelector('#reminders-tab .mft-table tbody');
    if (!tableBody) return;
    
    if (!contributions || contributions.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="7" class="empty-state">
                    <i class="fas fa-exclamation-circle"></i>
                    <h3>No Pending Contributions</h3>
                    <p>There are no pending contributions at this time.</p>
                </td>
            </tr>`;
        return;
    }
    
    // Clear existing rows
    tableBody.innerHTML = '';
    
    // Add new rows
    contributions.forEach(contribution => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${contribution.memberName}</td>
            <td>KSh ${contribution.amount.toLocaleString()}</td>
            <td>${formatDate(contribution.dueDate)}</td>
            <td>${contribution.daysOverdue}</td>
            <td>${contribution.category}</td>
            <td><span class="status-badge status-${contribution.status.toLowerCase()}">${contribution.status}</span></td>
            <td class="action-cell">
                <button class="btn btn-sm ${contribution.status === 'Overdue' ? 'btn-danger' : 'btn-primary'} send-reminder-btn" title="Send Reminder" data-id="${contribution.id}">
                    <i class="fas fa-bell"></i>
                </button>
                <button class="btn btn-sm btn-outline view-btn" title="View Details" data-id="${contribution.id}">
                    <i class="fas fa-eye"></i>
                </button>
            </td>
        `;
        tableBody.appendChild(row);
    });
    
    // Add event listeners to action buttons
    attachActionListeners();
    
    // Add sorting functionality
    addSortingToTable('pendingContributionsTableBody');
}

// Function to add sorting to table
function addSortingToTable(tableBodyId) {
    // Fix: Get the table body element first, then find its closest .mft-table parent
    const tableBody = document.getElementById(tableBodyId);
    if (!tableBody) return;
    
    const table = tableBody.closest('.mft-table');
    if (!table) return;
    
    const tableHeaders = table.querySelectorAll('th.sortable-header');
    
    tableHeaders.forEach(header => {
        header.addEventListener('click', function() {
            const sortKey = this.getAttribute('data-sort');
            const isAscending = !this.classList.contains('asc');
            
            // Update header classes
            tableHeaders.forEach(h => h.classList.remove('asc', 'desc'));
            this.classList.toggle(isAscending ? 'asc' : 'desc', true);
            
            // Get table body and rows
            const tableBody = document.getElementById(tableBodyId);
            const rows = Array.from(tableBody.querySelectorAll('tr:not(.empty-state):not(.loading-state)'));
            
            // Sort rows
            rows.sort((a, b) => {
                const aText = a.cells[Array.from(header.parentNode.children).indexOf(header)].textContent.trim();
                const bText = b.cells[Array.from(header.parentNode.children).indexOf(header)].textContent.trim();
                
                // Handle numeric values
                const aNum = parseFloat(aText.replace(/[^0-9.-]+/g, ''));
                const bNum = parseFloat(bText.replace(/[^0-9.-]+/g, ''));
                
                if (!isNaN(aNum) && !isNaN(bNum)) {
                    return isAscending ? aNum - bNum : bNum - aNum;
                }
                
                // Handle date values
                const aDate = new Date(aText);
                const bDate = new Date(bText);
                
                if (aDate !== 'Invalid Date' && bDate !== 'Invalid Date') {
                    return isAscending ? aDate - bDate : bDate - aDate;
                }
                
                // Handle text values
                return isAscending ? 
                    aText.localeCompare(bText, undefined, {numeric: true}) : 
                    bText.localeCompare(aText, undefined, {numeric: true});
            });
            
            // Re-append sorted rows
            rows.forEach(row => tableBody.appendChild(row));
        });
    });
}

// Function to attach event listeners to action buttons
function attachActionListeners() {
    // Remove any existing event listeners to prevent duplicates
    document.querySelectorAll('.view-btn').forEach(button => {
        button.removeEventListener('click', handleViewClick);
        button.addEventListener('click', handleViewClick);
    });
    
    document.querySelectorAll('.edit-btn').forEach(button => {
        button.removeEventListener('click', handleEditClick);
        button.addEventListener('click', handleEditClick);
    });
    
    document.querySelectorAll('.delete-btn').forEach(button => {
        button.removeEventListener('click', handleDeleteClick);
        button.addEventListener('click', handleDeleteClick);
    });
    
    document.querySelectorAll('.mark-paid-btn').forEach(button => {
        button.removeEventListener('click', handleMarkPaidClick);
        button.addEventListener('click', handleMarkPaidClick);
    });
    
    document.querySelectorAll('.send-reminder-btn').forEach(button => {
        button.removeEventListener('click', handleSendReminderClick);
        button.addEventListener('click', handleSendReminderClick);
    });
}

// Handler functions for action buttons
function handleViewClick(event) {
    const contributionId = this.getAttribute('data-id');
    viewContribution(contributionId);
}

function handleEditClick(event) {
    const contributionId = this.getAttribute('data-id');
    editContribution(contributionId);
}

function handleDeleteClick(event) {
    const contributionId = this.getAttribute('data-id');
    deleteContribution(contributionId);
}

function handleMarkPaidClick(event) {
    const contributionId = this.getAttribute('data-id');
    markContributionAsPaid(contributionId);
}

function handleSendReminderClick(event) {
    const contributionId = this.getAttribute('data-id');
    sendContributionReminder(contributionId);
}

// Function to update member select options
function updateMemberSelectOptions(members) {
    const memberSelect = document.getElementById('contributionsMemberSelect');
    if (!memberSelect) return;
    
    if (!members || members.length === 0) {
        return;
    }
    
    // Clear existing options except the first one
    while (memberSelect.options.length > 1) {
        memberSelect.remove(1);
    }
    
    // Add new options
    members.forEach(member => {
        const option = document.createElement('option');
        // Use the correct property names from the API response
        option.value = member.member_id || member.id;
        // Create a proper display name
        const displayName = member.first_name && member.last_name ? 
            `${member.first_name} ${member.last_name}` : 
            (member.first_name || member.last_name || member.name || member.member_id || member.id);
        option.textContent = displayName;
        memberSelect.appendChild(option);
    });
}

// Helper function to format dates
function formatDate(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
        month: 'short', 
        day: 'numeric', 
        year: 'numeric' 
    });
}

// Function to apply filters
function applyFilters() {
    const monthFilter = document.getElementById('filterMonth').value;
    const yearFilter = document.getElementById('filterYear').value;
    const memberFilter = document.getElementById('memberFilter').value;
    const statusFilter = document.getElementById('statusFilter').value;
    
    // Show notification that filters are being applied
    showNotification('Applying filters...');
    
    // Show loading state
    const tableBody = document.querySelector('#contributions-tab .mft-table tbody');
    if (tableBody) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="9" class="loading-state">
                    <i class="fas fa-spinner"></i>
                    <div>Applying filters...</div>
                </td>
            </tr>`;
    }
    
    // Simulate API call with filtered data
    setTimeout(() => {
        // In a real implementation, you would make an API call with the filter parameters
        fetchAllContributions().then(data => {
            if (data) {
                // Apply filters locally for demonstration
                let filteredData = data;
                
                if (monthFilter) {
                    filteredData = filteredData.filter(contribution => {
                        const contributionDate = new Date(contribution.date);
                        return contributionDate.getMonth() + 1 == monthFilter;
                    });
                }
                
                if (yearFilter) {
                    filteredData = filteredData.filter(contribution => {
                        const contributionDate = new Date(contribution.date);
                        return contributionDate.getFullYear() == yearFilter;
                    });
                }
                
                if (memberFilter && memberFilter !== 'all') {
                    filteredData = filteredData.filter(contribution => contribution.memberId === memberFilter);
                }
                
                if (statusFilter && statusFilter !== 'all') {
                    filteredData = filteredData.filter(contribution => contribution.status === statusFilter);
                }
                
                updateAllContributions(filteredData);
                
                // Add a notification that filters are applied
                if (monthFilter || yearFilter || memberFilter !== 'all' || statusFilter !== 'all') {
                    showNotification(`Filters applied successfully! Showing ${filteredData.length} of ${data.length} contributions.`);
                } else {
                    showNotification(`Showing all ${data.length} contributions`);
                }
            }
        });
    }, 500);
}

// Function to clear filters
function clearFilters() {
    document.getElementById('filterMonth').value = '';
    document.getElementById('filterYear').value = '';
    document.getElementById('memberFilter').value = 'all';
    document.getElementById('statusFilter').value = 'all';
    
    // Show notification
    showNotification('Filters cleared!');
    
    // Reload data without filters
    loadContributionsData();
    
    console.log('Filters cleared');
}

// Function to handle contribution payment that fills previous missed months
function handleContributionPayment(memberId, amount, paymentDate) {
    // Get the payment date
    const paymentDateObj = new Date(paymentDate);
    const paymentMonth = paymentDateObj.getMonth();
    const paymentYear = paymentDateObj.getFullYearFullYear();
    
    // Calculate missed months (assuming monthly contributions are due on the 1st)
    const currentDate = new Date();
    const currentMonth = currentDate.getMonth();
    const currentYear = currentDate.getFullYear();
    
    // Calculate the number of months between current date and payment date
    let monthsDiff = (currentYear - paymentYear) * 12 + (currentMonth - paymentMonth);
    
    // If the payment is for the current month or future, no missed months
    if (monthsDiff <= 0) {
        monthsDiff = 0;
    }
    
    // Log the information
    console.log(`Payment of KSh ${amount} for member ${memberId} on ${paymentDate}`);
    console.log(`This payment covers ${monthsDiff + 1} months (including current month)`);
    
    // In a real implementation, this would:
    // 1. Create contribution records for each missed month
    // 2. Distribute the payment amount across the months
    // 3. Update the member's contribution history
    
    return monthsDiff + 1; // Return number of months covered
}

// Function to manually refresh the contributions management dashboard
function refreshContributionsManagement() {
    loadContributionsData();
}

// Function to show notification
function showNotification(message) {
    // Remove existing notification if any
    const existingNotification = document.querySelector('.filter-notification');
    if (existingNotification) {
        existingNotification.remove();
    }
    
    // Create new notification
    const notification = document.createElement('div');
    notification.className = 'filter-notification';
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed; 
        top: 20px; 
        right: 20px; 
        background: #4CAF50; 
        color: white; 
        padding: 15px; 
        border-radius: 5px; 
        z-index: 1000; 
        box-shadow: 0 2px 10px rgba(0,0,0,0.2);
        font-family: 'Inter', sans-serif;
    `;
    
    document.body.appendChild(notification);
    
    // Remove notification after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 3000);
}

// Function to view contribution details
function viewContribution(contributionId) {
    console.log('Viewing contribution:', contributionId);
    
    // In a full implementation, this would fetch the contribution details from the API
    // For now, we'll show a modal with placeholder data
    showContributionModal(contributionId);
}

// Function to show contribution details in a modal
function showContributionModal(contributionId) {
    // Create modal HTML
    const modalHtml = `
        <div id="contributionModal" class="modal">
            <div class="modal-content">
                <div class="modal-header">
                    <h2>Contribution Details</h2>
                    <span class="close" id="closeModal">&times;</span>
                </div>
                <div class="modal-body">
                    <div class="contribution-details">
                        <div class="detail-row">
                            <span class="label">Contribution ID:</span>
                            <span class="value">${contributionId}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Member:</span>
                            <span class="value">John Doe (MFT001)</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Amount:</span>
                            <span class="value">KSh 5,000</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Date:</span>
                            <span class="value">Oct 15, 2023</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Payment Method:</span>
                            <span class="value">M-PESA</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Reference:</span>
                            <span class="value">MP234567890</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Category:</span>
                            <span class="value">Monthly Contribution</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Status:</span>
                            <span class="value"><span class="status-badge status-paid">Paid</span></span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Notes:</span>
                            <span class="value">Regular monthly contribution for October 2023</span>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-outline" id="closeModalBtn">Close</button>
                    <button class="btn btn-primary edit-modal-btn" data-id="${contributionId}">Edit</button>
                </div>
            </div>
        </div>
    `;
    
    // Add modal to document
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Add styles for modal
    const modalStyles = `
        <style>
            .modal {
                display: block;
                position: fixed;
                z-index: 10000;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0,0,0,0.7);
            }
            
            .modal-content {
                background-color: var(--dark);
                margin: 5% auto;
                padding: 0;
                border: 1px solid var(--primary);
                border-radius: 10px;
                width: 80%;
                max-width: 600px;
                box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
            }
            
            .modal-header {
                padding: 20px;
                border-bottom: 1px solid var(--gray-dark);
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            
            .modal-header h2 {
                margin: 0;
                color: var(--primary);
            }
            
            .close {
                color: var(--gray);
                font-size: 28px;
                font-weight: bold;
                cursor: pointer;
            }
            
            .close:hover {
                color: var(--light);
            }
            
            .modal-body {
                padding: 20px;
            }
            
            .contribution-details {
                display: grid;
                gap: 15px;
            }
            
            .detail-row {
                display: flex;
                justify-content: space-between;
                padding: 10px 0;
                border-bottom: 1px solid var(--gray-dark);
            }
            
            .detail-row:last-child {
                border-bottom: none;
            }
            
            .label {
                font-weight: 600;
                color: var(--gray);
            }
            
            .value {
                text-align: right;
            }
            
            .modal-footer {
                padding: 20px;
                border-top: 1px solid var(--gray-dark);
                display: flex;
                justify-content: flex-end;
                gap: 10px;
            }
        </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', modalStyles);
    
    // Add event listeners for closing the modal
    document.getElementById('closeModal').addEventListener('click', closeContributionModal);
    document.getElementById('closeModalBtn').addEventListener('click', closeContributionModal);
    
    // Add event listener for edit button
    document.querySelector('.edit-modal-btn').addEventListener('click', function() {
        const id = this.getAttribute('data-id');
        closeContributionModal();
        editContribution(id);
    });
    
    // Close modal when clicking outside of it
    window.addEventListener('click', function(event) {
        const modal = document.getElementById('contributionModal');
        if (event.target === modal) {
            closeContributionModal();
        }
    });
}

// Function to close the contribution modal
function closeContributionModal() {
    const modal = document.getElementById('contributionModal');
    if (modal) {
        modal.remove();
    }
}

// Function to edit contribution details
function editContribution(contributionId) {
    console.log('Editing contribution:', contributionId);
    
    // In a full implementation, this would fetch the contribution details from the API
    // For now, we'll show an edit modal with placeholder data
    showEditContributionModal(contributionId);
}

// Function to show edit contribution modal
function showEditContributionModal(contributionId) {
    // Create modal HTML
    const modalHtml = `
        <div id="editContributionModal" class="modal">
            <div class="modal-content">
                <div class="modal-header">
                    <h2>Edit Contribution</h2>
                    <span class="close" id="closeEditModal">&times;</span>
                </div>
                <div class="modal-body">
                    <form id="editContributionForm">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="editMember">Member</label>
                                <input type="text" id="editMember" value="John Doe (MFT001)" disabled>
                            </div>
                            
                            <div class="form-group">
                                <label for="editAmount">Amount (KSh) *</label>
                                <input type="number" id="editAmount" min="1" value="5000" required>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="editDate">Date *</label>
                                <input type="date" id="editDate" value="2023-10-15" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="editPaymentMethod">Payment Method *</label>
                                <select id="editPaymentMethod" required>
                                    <option value="M-PESA" selected>M-PESA</option>
                                    <option value="Bank Transfer">Bank Transfer</option>
                                    <option value="Cash">Cash</option>
                                    <option value="Cheque">Cheque</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="editReference">Reference Number</label>
                                <input type="text" id="editReference" value="MP234567890">
                            </div>
                            
                            <div class="form-group">
                                <label for="editCategory">Category *</label>
                                <select id="editCategory" required>
                                    <option value="Monthly Contribution" selected>Monthly Contribution</option>
                                    <option value="Welfare Contribution">Welfare Contribution</option>
                                    <option value="Project Contribution">Project Contribution</option>
                                    <option value="Loan Repayment">Loan Repayment</option>
                                    <option value="Fine Payment">Fine Payment</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="editStatus">Status *</label>
                                <select id="editStatus" required>
                                    <option value="Paid" selected>Paid</option>
                                    <option value="Pending">Pending</option>
                                    <option value="Rejected">Rejected</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="editNotes">Notes</label>
                                <textarea id="editNotes">Regular monthly contribution for October 2023</textarea>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-outline" id="closeEditModalBtn">Cancel</button>
                    <button class="btn btn-primary" id="saveContributionBtn">Save Changes</button>
                </div>
            </div>
        </div>
    `;
    
    // Add modal to document
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Add event listeners for closing the modal
    document.getElementById('closeEditModal').addEventListener('click', closeEditContributionModal);
    document.getElementById('closeEditModalBtn').addEventListener('click', closeEditContributionModal);
    
    // Add event listener for save button
    document.getElementById('saveContributionBtn').addEventListener('click', function() {
        saveContributionChanges(contributionId);
    });
    
    // Close modal when clicking outside of it
    window.addEventListener('click', function(event) {
        const modal = document.getElementById('editContributionModal');
        if (event.target === modal) {
            closeEditContributionModal();
        }
    });
}

// Function to save contribution changes
function saveContributionChanges(contributionId) {
    // Get form values
    const amount = document.getElementById('editAmount').value;
    const date = document.getElementById('editDate').value;
    const paymentMethod = document.getElementById('editPaymentMethod').value;
    const reference = document.getElementById('editReference').value;
    const category = document.getElementById('editCategory').value;
    const status = document.getElementById('editStatus').value;
    const notes = document.getElementById('editNotes').value;
    
    // Validate required fields
    if (!amount || !date || !paymentMethod || !category || !status) {
        alert('Please fill in all required fields');
        return;
    }
    
    // In a full implementation, this would make an API call to update the contribution
    console.log('Saving contribution changes:', {
        contributionId,
        amount,
        date,
        paymentMethod,
        reference,
        category,
        status,
        notes
    });
    
    showNotification('Contribution updated successfully!');
    closeEditContributionModal();
    
    // Refresh the contributions data after editing
    loadContributionsData();
}

// Function to close the edit contribution modal
function closeEditContributionModal() {
    const modal = document.getElementById('editContributionModal');
    if (modal) {
        modal.remove();
    }
}

// Function to delete a contribution
function deleteContribution(contributionId) {
    // Show confirmation dialog
    const confirmation = confirm(`Are you sure you want to delete contribution ID: ${contributionId}? This action cannot be undone.`);
    
    if (confirmation) {
        // In a full implementation, this would make an API call to delete the contribution
        console.log('Deleting contribution:', contributionId);
        showNotification(`Deleting contribution ID: ${contributionId}`);
        
        // Simulate API call
        setTimeout(function() {
            alert(`Contribution ID: ${contributionId} has been deleted.\n\nIn a full implementation, this would make an API call to delete the contribution from the database.`);
            showNotification('Contribution deleted successfully!');
            
            // Refresh the contributions data after deletion
            loadContributionsData();
        }, 500);
    }
}

// Function to mark a contribution as paid
function markContributionAsPaid(contributionId) {
    if (confirm(`Mark contribution ID: ${contributionId} as Paid?`)) {
        console.log('Marking contribution as paid:', contributionId);
        showNotification(`Marking contribution ID: ${contributionId} as Paid`);
        // In a full implementation, this would make an API call to update the contribution status
        alert(`Contribution ID: ${contributionId} has been marked as Paid.\n\nIn a full implementation, this would make an API call to update the contribution status.`);
        
        // Refresh the contributions data after marking as paid
        loadContributionsData();
    }
}

// Function to send a reminder for a contribution
function sendContributionReminder(contributionId) {
    console.log('Sending reminder for contribution:', contributionId);
    showNotification(`Sending reminder for contribution ID: ${contributionId}`);
    // In a full implementation, this would make an API call to send a reminder
    alert(`Reminder sent for contribution ID: ${contributionId}.\n\nIn a full implementation, this would send an SMS/email reminder to the member.`);
}

// Add event listener for manual refresh (if needed)
document.addEventListener('keydown', function(event) {
    // Press F5 to manually refresh
    if (event.key === 'F5') {
        event.preventDefault();
        refreshContributionsManagement();
    }
});